"""
Author: Niklas Blomberg, EMBL 1998

some small changes by Jens Linge, 25-6-2001 (csi error ranges can be specified now)

This module contains some functions for reading CSI files and writing ARIA/CNS
formatted phi/psi- and hbond-restraints.

When invoked from the command line it prints to specified filenames in
the current directory otherwise it needs filenames/parameters as argument.

This module contains a function to generate secondary structure restraints
from a list [(1,H/B),...]

parseCSI('csifile') returns a list from the consensus column in the CSI output
makeRestraints(csilist,hbondfile,psifile,phifile)
"""

__author__   = "$Author: anonymous $"
__revision__ = "$Revision: 2.4 $"
__date__     = "$Date: 2001/07/23 08:48:27 $"

import string

def parseCSI(csifile):
    """
    parseCSI(csifile) returns a list for makeRestraints
    from the consensus column in the CSI output
    """
    sslist=[]
    file=open(csifile,'r')
    for line in file.readlines():
        linelist=string.split(line)
#        print linelist
        if len(line) < 2 or len(linelist) < 2: continue
        if linelist[0][0] != '#' :
            if linelist[-1]!='C':
                sslist.append((string.atoi(linelist[0]),linelist[-1]))
    file.close()
#    print sslist #test
    return sslist


## in aria1.0, it was:
## def makeRestraints(csilist, alphaphi=[-65,20],alphapsi=[-35,20],betaphi=[-135,20],betapsi=[135,20],bonds=[[[4,1,1],'N','O',3.2,1.0,0.0],[[4,1,1],'HN','O',2.2,1.0,0.0]]):


def makeRestraints(csilist, csierror="30.0", alphaphi=-65,alphapsi=-35,betaphi=-135,betapsi=135,bonds=[[[4,1,1],'N','O',3.2,1.0,0.0],[[4,1,1],'HN','O',2.2,1.0,0.0]]):
    """
    makeRestraints(csilist) returns a tuple of three lists (hbond, phi,psi) 
    that can be furter processed or written to disk
    The angles can be specified as arguments (angle,error) if used for something
    else (alphaxxx -> if H ; betaxxx if B)
    bonds is trickier - makes bonds between specified atoms i,i-4 (+1,-1)
    (ambiguous) - this can be supplied as argument for atoms marked 'H' to
    make whatever restraints you want
    Format: (i-what,-res,+res), atom1, atom2 (xplor syntax), d,d-,d+
    
    """
    #######################################################################
    # Some helper functions
    #######################################################################

    def makePhiRestraint(aa,angle,error):
        return '! CSI derived phi restraint:\nassign (resid %3i and name C)\n       (resid %3i and name N)\n       (resid %3i and name CA)\n       (resid %3i and name C)\n       1.0 %3i %3i 2\n\n' % (aa-1,aa,aa,aa,angle,error)
    def makePsiRestraint(aa,angle,error):
        return '! CSI derived psi restraint:\nassign (resid %3i and name N)\n       (resid %3i and name CA)\n       (resid %3i and name C)\n       (resid %3i and name N)\n       1.0 %3i %3i 2\n\n' %  (aa,aa,aa,aa+1,angle,error)
    def makeBondRestraint(atom,parameters):
        return '! CSI derived hbond restraint:assign (resid %3i and name %s) (resid %3i:%3i and name %s) %3.1f %3.1f %3.1f \n\n' % (atom,parameters[1],atom-parameters[0][0]-parameters[0][1],atom-parameters[0][0]+parameters[0][2],parameters[2],parameters[3],parameters[4],parameters[5])
    
    #######################################################################
    # Code:
    #######################################################################
    philist=[]
    psilist=[]
    hbondlist=[]
    bondlist=[]
    in_row=[]
    for i in bonds:
        bondlist.append([])
        in_row.append(i[0][0])
    for aa in csilist:
        #First deal with helices
        if aa[1]=='H':
            #Make phi/psi restraint
            philist.append(makePhiRestraint(aa[0],alphaphi,float(csierror)))
            psilist.append(makePsiRestraint(aa[0],alphapsi,float(csierror)))

            # OLD CODE see two lines below 25-6-2001 Jens
            ## philist.append(makePhiRestraint(aa[0],alphaphi[0],alphaphi[1]))
            ## psilist.append(makePsiRestraint(aa[0],alphapsi[0],alphapsi[1]))


            #Now check if its at least bonds[i][0] in a row,
            # if so make a H-bond restraint
            for i in range(len(bondlist)):
                if len(bondlist[i])<in_row[i]:
                    if not bondlist[i]:
                        bondlist[i].append(aa[0])
                    elif bondlist[i][-1]==aa[0]-1:
                        bondlist[i].append(aa[0]) 
                    else:
                        bondlist[i]=[] #Flush since we jumped in sequence
                elif len(bondlist[i])>=in_row[i]-1 and  bondlist[i][-1]==aa[0]-1:
                    hbondlist.append(makeBondRestraint(aa[0],bonds[i]))
                    bondlist[i].append(aa[0])
                else:
                    bondlist[i]=[] #Flush since we jumped in sequence
        elif aa[1]=='B':
            philist.append(makePhiRestraint(aa[0],betaphi,float(csierror)))
            psilist.append(makePsiRestraint(aa[0],betapsi,float(csierror)))

            # OLD CODE see two lines below (Jens, 25-6-2001)
##             philist.append(makePhiRestraint(aa[0],betaphi[0],betaphi[1]))
##             psilist.append(makePsiRestraint(aa[0],betapsi[0],betapsi[1]))

    return (hbondlist,philist,psilist)


def writeRestraint(restraint,filename,mode='w'):
    """
    writeRestraint(restraint,file)
    writes restraint (as preformatted list) to specified file
    """
    file=open(filename,mode)
    headerString = """! dihedral angle file generated by Csi2Aria.py

"""
    file.write(headerString)
    for i in restraint:
        file.write(i)
    file.close()
    return



###############################################################################
if __name__ == "__main__":
    import sys
    try:
        csifile=sys.argv[1]
    except IndexError:
        print "Usage: python Csi2Aria.py csioutput"
        sys.exit()
    list=parseCSI(csifile)
    hbond,phi,psi=makeRestraints(list)
    writeRestraint(hbond,'./hbond.tbl')
    print 'Wrote hbond.tbl'
    writeRestraint(phi,'./phi.tbl')
    print 'Wrote phi.tbl'
    writeRestraint(psi,'./psi.tbl')
    print 'Wrote psi.tbl'

        
