"""
This file is the main for calling methods to convert data to BMRB
STAR format

author: Jurgen Doreleijers, BMRB

15-06-2001 some small changes by Jens Linge for the ARIA1.1 plugin
"""
__author__    = "$Author: anonymous $"
___revision__ = "$Revision: 2.7 $"
___date__     = "$Date: 2001/09/08 19:12:04 $"

from Aria.DataIO        import PpmList,NoeList
from Aria.ThirdParty    import TextFile
from Aria.Nomenclature  import Nomenclature
from types import *
import string, sys, pickle, cmd, os


## No changes required after this line
################################################################################

class Lister:
    """Example from 'Learning Python from O'Reilly publisher'"""
    def __repr__(self):
        return ("<Instance of %s, address %s:\n%s>" %
           (self.__class__.__name__, id(self), self.attrnames()))

    def attrnames(self):
        result=''
        keys = self.__dict__.keys()
        keys.sort()
        for attr in keys:
            if attr[:2] == "__":
                result = result + "\tname %s=<built-in>\n" % attr
            else:
                result = result + "\tname %s=%s\n" % (attr, self.__dict__[attr])
        return result


class StereoGroup( Lister ):
    """
    Represents one (ambiguously) (un-)assigned stereogroup
    which is one row in the STAR file
    """
    def __init__(self):
            self.atomMemberID       = '.'
            self.residueSeqCode     = '.'
            self.residueLabel       = '.'
            self.atomPseudoName     = '.'
            self.stereoGroupType    = '.'
            self.assignmentCode     = '.'
            self.atomNames          = []
        
    def AbsorbGroup(self, \
            stereoGroupDict, residueSeqCode, residueLabel, atomPseudoName):

        self.residueSeqCode     = residueSeqCode
        self.residueLabel       = residueLabel
        self.atomPseudoName     = atomPseudoName
        self.stereoGroupType    = stereoGroupDict[0]
        self.assignmentCode     = "?"

        for EACHAtom in stereoGroupDict[1]:
            self.atomNames.append( EACHAtom )



class AssignedNOEPeak_NMRSTARRow( Lister ):
    """
    Represents one (ambiguously) assigned NOE peak row in the STAR file
    """
    def __init__(self):
        self.spectralPeakID                     = "."
        self.sourceExperimentCode               = "."
        self.peakVolumeValue                    = "."
        self.spectralDimensionID                = "."
        self.peakChemicalShiftValue             = "."
        self.contributionID                     = "."
        self.contributionChemicalShiftValue     = "."
        self.contributionResidueSequenceCode    = "."
        self.contributionResidueLabel           = "."
        self.contributionAtomName               = "."

        
class AssignedNOEPeak( Lister ):
    """
    Represents one (ambiguously) assigned NOE peak
    """
    def __init__(self):
        self.rows = []

    def AbsorbPeak(self, peak):

        number_contributions= 0
        
        for EACHC in peak.contributions:
            ## There is no quick way to determine the dimensionality of a spectrum from
            ## a peak is there, Jens?
            ## Neither for what is the order in which they are coded in the pulse scheme
            ## or at least which dimension corresponds to acquisition?
            ## Add peak info only once for first set of rows.
            
            number_dimensions = 0
            number_contributions = number_contributions + 1
            for EACHPossibleShift in ( EACHC.h1ppm, EACHC.p1ppm, EACHC.h2ppm, EACHC.p2ppm ):
                
                if ( EACHPossibleShift != None ):
                    number_dimensions = number_dimensions + 1
                    row = AssignedNOEPeak_NMRSTARRow()
                    
                    if ( ( number_dimensions == 1 ) and ( number_contributions == 1 ) ):
                        row.spectralPeakID              = EACHC.peakNumber
                        row.sourceExperimentCode        = EACHC.spectrumName[:]
                        row.peakVolumeValue             = EACHC.volume
                        
                    row.spectralDimensionID             = number_dimensions
                    row.peakChemicalShiftValue          = EACHPossibleShift
                    row.contributionID                  = number_contributions
                    
                    if ( EACHPossibleShift is EACHC.h1ppm ):
                        row.contributionChemicalShiftValue  = EACHC.assih1ppm
                        row.contributionResidueSequenceCode = EACHC.residue1
                        row.contributionResidueLabel        = EACHC.aa1[:]
                        atomn                               = EACHC.atomnameh1[0][:]
                    elif( EACHPossibleShift is EACHC.p1ppm ):
                        row.contributionChemicalShiftValue  = EACHC.assip1ppm
                        row.contributionResidueSequenceCode = EACHC.residue1
                        row.contributionResidueLabel        = EACHC.aa1[:]
                        atomn                               = EACHC.atomnamep1[0][:]
                    elif( EACHPossibleShift is EACHC.h2ppm ):
                        row.contributionChemicalShiftValue  = EACHC.assih2ppm
                        row.contributionResidueSequenceCode = EACHC.residue2
                        row.contributionResidueLabel        = EACHC.aa2[:]
                        atomn                               = EACHC.atomnameh2[0][:]
                    elif( EACHPossibleShift is EACHC.p2ppm ):
                        row.contributionChemicalShiftValue  = EACHC.assip2ppm
                        row.contributionResidueSequenceCode = EACHC.residue2
                        row.contributionResidueLabel        = EACHC.aa2[:]
                        atomn                               = EACHC.atomnamep2[0][:]
                    else:
                        tekst = "Code error"
                        _printWarn( __name__, tekst )

                    ## Adjust bogus values for cs and residue number.
                    if ( row.contributionChemicalShiftValue == "-999.000" ):
                        row.contributionChemicalShiftValue = "."
                    if ( row.contributionResidueSequenceCode == 999 ):
                        row.contributionResidueSequenceCode = "."
                        
                    atomn = Nomenclature.Convert_PseudoAtomName_CNS_2_IUPAC(
                             row.contributionResidueLabel, atomn)
                    atomn = Nomenclature.Convert_AtomName_IUPAC_2_BMRB(atomn)
                    row.contributionAtomName = atomn
                    
                    self.rows.append( row )
                    
            if ( number_dimensions < 2 ):
                text  = "Peak appeared to have %s dimension(s) in stead of\n" % number_dimensions
                text = text + "the expected minimum of 2 dimensions."
                _printWarn(__name__, text)



class DistanceRestraintList:

# overloaded
    def __init__(self, name = '', comment = 'BMRB general distance restraint list'):
        self.name               = name
        self.fileNames          = []
        self.dlist              = []
        self.stereogroups       = []

    def __repr__(self): # string representation
         return self.name

    def append(self, object):
         self.dlist.append(object)

# public methods

    def WriteStar(self, fileName):
        "Writes a BMRB general distance constraint list in STAR format"
         
        if len(self.dlist) == 0:
             print 'List empty. WriteStar method aborted.'
             return

        print 'Writing the data in STAR format to', fileName
        file = TextFile.TextFile(fileName, 'w')
        file.write( self._createStarHeader() )
        file.write( self._createStarEntryInformationStub() )
        file.write( self._createStarSoftwareDescription() )
        file.write( self._createStarDistanceConstraints() )
##        file.write( self._createStarAssignedNOEPeaks() )
        file.write( self._createStarStereoAssignments() )
        file.write( self._createStarReferences() )
        file.close()

    def StereoGroupListCreate( self, sequencelist):
        i = 0
        for EACHResidue in sequencelist:
            i = i + 1
            if ( Nomenclature.stereo_atom_lib.stereo_atom.has_key( EACHResidue ) ):
                stereoGroupDict = Nomenclature.stereo_atom_lib.stereo_atom[ 
                                    EACHResidue ]
            else:
                stereoGroupDict = {}
                
            for EACHStereoGroupDictKey in stereoGroupDict.keys():
                stereoGroup = StereoGroup()
                
                residueSeqCode  = i
                residueLabel    = EACHResidue
                atomPseudoName  = EACHStereoGroupDictKey
                
                stereoGroup.AbsorbGroup( stereoGroupDict[EACHStereoGroupDictKey],\
                    residueSeqCode, residueLabel, atomPseudoName )
                self.stereogroups.append( stereoGroup )

##    def StereoGroupListSetAssignmentCodesFromDRl( self ):
##        for EACHStereoGroup in 
    
    def AbsorbNoeList( self, noelist): 
        # Absorb filenames by making a literal copy
        self.fileNames = noelist.fileNames[:]
    
        for EACHP in noelist.peakslist:
            self.AbsorbNoe(EACHP)
    
    def AbsorbNoe(self, peak):

        ## Create a NMR-STAR distance restraint
        dR = DistanceRestraint()

        ## A part of this restraint type is a peak object
        dR.peak.AbsorbPeak( peak )

        ## Create a single row within a NMR-STAR distance restraint with "." initialized
        row = DistanceRestraint_NMRSTARRow()

        # Creating a local variable here
        nodeLogicId = 1
        
        FIRSTC = peak.contributions[ 0 ]
        row.nodeLogicId             = nodeLogicId
##        row.nodeRightId             = "."
        
        # Ambiguous
        if ( len(peak.contributions) > 1 ):
            row.nodeDownId                      = 2
            row.nodeLogicOperation              = 'OR'
            
            ## Append the row
            dR.rows.append( row )
        # Actually not ambiguous, create only 1 restraint node
        else:            
##            row.nodeDownId                      = "."
            pass
            
        for EACHC in peak.contributions:
            
            nodeLogicId = nodeLogicId + 1
            
##            if _emptyObject( EACHC.segid1 ):
##                EACHC.segid1 = "."
##            if _emptyObject( EACHC.segid2 ):
##                EACHC.segid2 = "."
            if _emptyObject( EACHC.residue1 ):
                EACHC.residue1 = "."
            if _emptyObject( EACHC.residue2 ):
                EACHC.residue2 = "."
            if _emptyObject( EACHC.aa1 ):
                EACHC.aa1 = "."
            if _emptyObject( EACHC.aa2 ):
                EACHC.aa2 = "."
                    
            ## Set atom identifiers
            for EACHAtom in ( 1, 2 ):
                if not ( ( len(peak.contributions) == 1 ) and ( EACHAtom == 1)):
                    row = DistanceRestraint_NMRSTARRow()
                
                row.nodeClassId = EACHAtom
                
                if ( len(peak.contributions) > 1 ):
                    row.nodeLogicId = nodeLogicId
                    if ( EACHAtom == 1 ):
##                        row.nodeDownId      = '.'
                        pass
                        if ( nodeLogicId == len(peak.contributions) + 1 ):
##                            row.nodeRightId = '.'
                            pass
                        else:
                            row.nodeRightId = nodeLogicId + 1
                else:
                    row.nodeLogicId = nodeLogicId - 1

                if ( ( EACHAtom == 1) and ( nodeLogicId != 1 ) ):
                    ## Add the distance and peak info only to the
                    ## first atom, first restraint node row
                    ## FIRSTC is declared above
##                    row.distanceValue           = EACHC.distanceAve
                    row.distanceLowerBoundValue = EACHC.lowerBound
                    row.distanceUpperBoundValue = EACHC.upperBound
                
                    row.sourceExperimentCode    = EACHC.spectrumName
                    row.spectralPeakID          = EACHC.peakNumber
                    row.peakVolumeValue         = EACHC.volume

                if EACHAtom == 1:
                    if len(peak.contributions) > 1:
                        row.contributionValue       = EACHC.contribution
                    else:
                        row.contributionValue       = "1.00"

                if ( EACHAtom == 1 ):
##                    chain = EACHC.segid1[:]
                    ppm   = EACHC.p1ppm
                    resid = EACHC.residue1[:]
                    resn  = EACHC.aa1[:]
                    atomn = EACHC.atomnamep1[0][:]
                else:
##                    chain = EACHC.segid2[:]
                    ppm   = EACHC.p2ppm
                    resid = EACHC.residue2[:]
                    resn  = EACHC.aa2[:]
                    atomn = EACHC.atomnamep2[0][:]
                    
                if _emptyObject( atomn ):
                    tekst = "atom name was not defined for atom number %s" % EACHAtom
                    _print_warn(__name__, tekst)
                    atomn = '.'
                    
                atomn = Nomenclature.Convert_PseudoAtomName_CNS_2_IUPAC(
                            resn, atomn)
                atomn = Nomenclature.Convert_AtomName_IUPAC_2_BMRB(atomn)
                    
##                row.subsystemAuthorCode     = chain
                row.residueAuthorSeqCode    = resid
                row.residueAuthorLabel      = resn
                row.atomAuthorName          = atomn
                if ( nodeLogicId == 2 ):
                    row.peakChemicalShiftValue  = ppm

                ## Append the row
                dR.rows.append( row )

        self.append( dR )        

        
# private methods
    def _cleanUp(self):
        #delete all existing distance restraints:
        self.name = ''
        self.fileNames = []
        self.dlist = []


    def _createStarDistanceConstraints(self):
        "Distance constraints of the STAR file"
        
        header_s = \
"""

##########################
#  Distance Constraints  #
##########################
save_distance_constraints
   _Saveframe_category           distance_constraints

   _Details
;
Derived from the file(s):
"""
        header_s = header_s + self.fileNames[0]
        header_s = header_s +\
"""
May contain ambiguous and non-ambiguous distance constraints
;
   _Constraint_type              NOE
   _Software_label              $Aria
   _Original_atom_nomenclature   Aria
   _Averaging_method             SUM
   _Floating_chirality           ?
        # Determines how violations are calculated
        # Enumeration:
        # all           All stereo groups were left floating
        # none          All stereo groups were stereospecifically assigned
        # aromatics     Only PHE and TYR ring atoms were left floating  
        # partial       Some stereo groups were floating
        #               An assignment save_frame (presented below)
        #               should be filled out to identify those floats
   _Potential_function           ?
        # Please provide a value like:
        # biharmonic, square, "soft square"
   _Function_detail
;
Described in:

-1-  M. Nilges and S.I. O'Donoghue. Ambiguous NOEs and automated NOE 
assignment. Prog.NMR spectr. 32:107-139, 1998.

-2-  Linge,J.P., O'Donoghue,S.I. and Nilges,M. in Methods Enzymol., Nuclear
Magnetic Resonance of Biological Macromolecules Part B.:71-90, Academic
Press, 2001.
;
"""

##        Creates the logical / restraint nodes table

        tables_s  = self._createStarDistanceConstraintsLogic()
        tables_s = tables_s + self._createStarDistanceConstraintsAtom()
        tables_s = tables_s + self._createStarDistanceConstraintsDistance()

        return( header_s + tables_s + '\nsave_\n')


    def _createStarDistanceConstraintsDistance(self):
        "Distance constraint distances"

        table = """
   loop_
      _Distance_constraint_ID
      _Distance_constraint_tree_node_ID
      _Source_experiment_code
      _Spectral_peak_ID
      _Contribution_fractional_value
      _Distance_value
      _Distance_lower_bound_value
      _Distance_upper_bound_value


"""
        constraint_ID = 1
        for EACHdR in self.dlist:
            for EACHRow in EACHdR.rows:
                ## Write only first row of each constraint node
                if ( EACHRow.nodeLogicOperation == "." and
                     EACHRow.nodeClassId == 1 ):
                    table = table + '      %4s %4s %5s %5s %6s %6s %6s %6s\n'\
                        % ( constraint_ID,\
                             EACHRow.nodeLogicId,\
                             EACHRow.sourceExperimentCode,\
                             EACHRow.spectralPeakID,\
                             EACHRow.contributionValue,\
                             EACHRow.distanceValue,\
                             EACHRow.distanceLowerBoundValue,\
                             EACHRow.distanceUpperBoundValue )
            constraint_ID = constraint_ID + 1

        table = table + '\n   stop_\n'

        return( table )


    def _createStarDistanceConstraintsAtom(self):
        "Distance constraint atoms"

        table = """
   loop_
      _Constraint_tree_node_member_constraint_ID
      _Constraint_tree_node_member_node_ID
      _Constraint_tree_node_member_ID
      _Mol_system_component_ID
      _Residue_seq_code
      _Residue_label
      _Atom_name


"""
        constraint_ID = 1
        for EACHdR in self.dlist:
            for EACHRow in EACHdR.rows:
                ## Write all rows except logic node row
                if EACHRow.nodeLogicOperation == ".":
                    table = table + '      %4s %4s %4s %2s %4s %4s %-4s\n'\
                        % ( constraint_ID,\
                             EACHRow.nodeLogicId,\
                             EACHRow.nodeClassId,\
                             '.',\
                             EACHRow.residueAuthorSeqCode,\
                             EACHRow.residueAuthorLabel,\
                             EACHRow.atomAuthorName)
            constraint_ID = constraint_ID + 1

        table = table + '\n   stop_\n'

        return( table )


    def _createStarDistanceConstraintsLogic(self):
        "Distance constraint logic"

        table = """
   loop_
      _Constraint_ID
      _Constraint_tree_node_ID
      _Constraint_tree_down_node_ID
      _Constraint_tree_right_node_ID
      _Constraint_tree_logic_operation

"""
        constraint_ID = 1
        for EACHdR in self.dlist:
            for EACHRow in EACHdR.rows:
                ## Write every node only once
                if (EACHRow.nodeClassId in [ 1, '.']):
                    table = table + '      %4s %4s %4s %4s %4s\n'\
                            % ( constraint_ID,\
                                 EACHRow.nodeLogicId,\
                                 EACHRow.nodeDownId,\
                                 EACHRow.nodeRightId,\
                                 EACHRow.nodeLogicOperation )                                             
            constraint_ID = constraint_ID + 1

        table = table + '\n   stop_\n'

        return( table )


    def _createStarSoftwareDescription(self):
        "Standard software for Aria"
        
        s = \
"""

############################
#  Computer software used  #
############################
save_Aria
   _Saveframe_category   software

   _Name                 Aria
   _Version              1.1
   _Details             "ARIA 1.1 is using CNS 1.1"

   loop_
      _Vendor
      _Address
      _Electronic_address

      "Dr. Michael Nilges, Institut Pasteur" 
      
;
Unite de Bioinformatique Structurale, 
Institut Pasteur, 25-28 rue du Dr Roux,
75015 Paris, France
; 
       nilges@pasteur.fr 
      

   stop_

   loop_
      _Task

      "Automated NOE assignment"  
      "NMR structure calculation" 

   stop_

   loop_
      _Citation_label

      $reference_1 
      $reference_2 

   stop_

save_
"""
        return s


    def _createStarEntryInformationStub(self):
        "Standard entry information stub for Aria, for versioning"
        
        s = \
"""
#######################
#  Entry information  #
#######################
save_entry_information
   _Saveframe_category   entry_information

   _NMR_STAR_version    "Aria 1.1 development"

save_
"""
        return s


    def _createStarReferences(self):
        "Cited references"
        
        s = \
"""

#######################################
#  Cited references within the entry  #
#######################################
save_reference_1
   _Saveframe_category   citation

   _Citation_class       citation
   _Citation_full       
;
M. Nilges and S.I. O'Donoghue. Ambiguous NOEs and automated NOE 
assignment. Prog.NMR spectr. 32:107-139, 1998.
;

save_


save_reference_2
   _Saveframe_category   citation

   _Citation_class       citation
   _Citation_full       
;
Linge,J.P., O'Donoghue,S.I. and Nilges,M. in Methods Enzymol., Nuclear Magnetic
Resonance of Biological Macromolecules Part B.:71-90, Academic Press, 2001.
;

save_
"""
        return s


    def _createStarAssignedNOEPeaks(self):
        "Assigned NOE peak list"
        
        s = \
"""


####################################
# Assigned NOE spectral peak list  #
####################################
save_assigned_NOE_spectral_peak_list
  _Saveframe_category       assigned_NOE_spectral_peak_list
  
  _Details                 "Derived from an Aria .list file"

   loop_
      _Software_label

       $Aria

   stop_

   loop_
      _Assigned_NOE_chem_shift_peak_ID
      _Spectral_peak_ID
      _Source_experiment_code
      _Peak_volume
      _Spectral_dimension_ID
      _Chem_shift_value
      _Contribution_ID
      _Contribution_chem_shift_value
      _Contribution_Residue_seq_code
      _Contribution_Residue_label
      _Contribution_Atom_name
      
"""
        peak_ID = 0
        for EACHdR in self.dlist:
            peak_ID = peak_ID + 1
            for EACHRow in EACHdR.peak.rows:
                s = s + '      %6s %6s %4s %10s %2s %8s'\
                        % ( peak_ID,\
                             EACHRow.spectralPeakID,\
                             EACHRow.sourceExperimentCode,\
                             EACHRow.peakVolumeValue,\
                             EACHRow.spectralDimensionID,\
                             EACHRow.peakChemicalShiftValue )                                             
                s = s + ' %3s %8s %4s %4s %-5s\n'\
                        % (  EACHRow.contributionID,\
                             EACHRow.contributionChemicalShiftValue,\
                             EACHRow.contributionResidueSequenceCode,\
                             EACHRow.contributionResidueLabel,\
                             EACHRow.contributionAtomName )

        if ( peak_ID == 0 ):
            text = "No peaks found"
            _printWarn(__name__, text)

        s = s + '\n   stop_\n\nsave_\n'
        return s


    def _createStarStereoAssignments(self):
        "Stereo assignment for NOE list"

        max_number_atoms = 0
        for EACHstereoGroup in self.stereogroups:
            if len( EACHstereoGroup.atomNames ) > max_number_atoms:
                max_number_atoms = len( EACHstereoGroup.atomNames )

        english = English()

        s = \
"""


#####################################
# Stereo assignments                #
#####################################
save_stereoassignments
  _Saveframe_category   stereoassignments

# This saveframe is MANDATORY when one or more save_frames have a value for
# _Floating_chirality of partial (aromatics etc. do not need to be identified).
# Notes:
# * Distance constraints with a pseudoatom can exist though the group is 
# defined to be assigned here.
# * The closely related atom nomenclature issues are dealt with in the 
# saveframe category conformer_atom_nom_map. Here the stereoassignments of the
# distance constraints with respect to the IUPAC nomenclature is described.
# A stereogroup is considered floating if the assignements differs over the 
# different models in the ensemble.
# Only fields marked with a question mark need to be filled in.

   _Details
;
Derived from an Aria sequence file
;
    
   loop_
      _Stereo_assignment_ID       # Runs from 1 to the number of groups defined
                                  # here.
      _Mol_system_component_ID
      _Residue_seq_code
      _Residue_label
      _Atom_pseudo_name           # use IUPAC pseudo atom names if available
      _Atom_pseudo_type           # Based on AQUA definitions
      _Assignment_code
"""
        for EACHAtom in range(1,max_number_atoms+1):
            name = "_Atom_" + english.numbers[ EACHAtom ] + "_atom_name"
            s = s + "      " + name + ' ' * ( 28 - len ( name ) ) + '\n'
##            if EACHAtom > 2:
##                s = s + "# COND-MANDATORY\n"
##            else:
##                s = s + "# MANDATORY\n"

        s = s + \
"""

# example:
##          1   30   VAL   QG   4  .  swap   MG1   MG2   .
##          2   43   LYS   QB   1  .  asis   HB2   HB3   .
##          3   50   PHE   QR   7  .  float  QD    QE    HZ
##          4    1  DADE  "Q2'" 1  .  asis  "H2'" "H2''" .

      # Enumeration for _Assignment_code:
      # asis      Stereo group is assigned as is
      # swap      Stereo group is assigned but needs to be swapped
      #           (This can be done at the BMRB)
      # float     Stereo group is left to float
    
"""

        assignment_ID = 0
##        modelNumber = '.' ## Not used right now.
        for EACHstereoGroup in self.stereogroups:
            assignment_ID = assignment_ID +  1
            s = s + '      %6s  .  %4s %4s  %-4s %2s %-5s'\
                    % ( assignment_ID,\
                         EACHstereoGroup.residueSeqCode,\
                         EACHstereoGroup.residueLabel,\
                         EACHstereoGroup.atomPseudoName,\
                         EACHstereoGroup.stereoGroupType,\
##                         modelNumber,\
                         EACHstereoGroup.assignmentCode )
##                         EACHstereoGroup.atomMemberID )

            for EACHAtom in range(1,max_number_atoms+1):
                if EACHAtom <= len( EACHstereoGroup.atomNames ):
                    atomname = EACHstereoGroup.atomNames[ EACHAtom - 1]
                else:
                    atomname = "."
                s = s + ' %-4s' % atomname
            s = s + '\n'

        if ( assignment_ID == 0 ):
            text = "No stereogroups found"
            _printWarn(__name__, text)


        s = s + '\n   stop_\n\nsave_\n'
        return s


    def _createStarHeader(self):
        "Standard header for distance constraint header"

        return "data_distance_constraints\n"


class DistanceRestraint(Lister):
     """
    Represents one general distance restraint that might be ambiguous
    This object just contains all the data as attributes, there are no
    methods available.
    For now I implemented the ambiguity of ARIA only. I might have to
    get more general later on meaning to make actual object out of nodes.
     """
     def __init__(self):
        self.rows = []
        self.peak = AssignedNOEPeak()

        
class DistanceRestraint_NMRSTARRow( Lister ):
    """
    Represents just one row of many possible rows representing a (ambiguous) restraint
    """
    # Note that constraintId and constraintMemberId are not stored because they just
    # represent the sequential number of the constraint and that can be retrieved
    # from the intrinsic properties of the list.
    def __init__(self):
        self.nodeLogicId                = "."
        
        self.nodeDownId                 = "."
        self.nodeRightId                = "."
        self.nodeClassId                = "."
        self.nodeLogicOperation         = "."

        self.distanceValue              = "."
        self.distanceLowerBoundValue    = "."
        self.distanceUpperBoundValue    = "."

        self.sourceExperimentCode       = "."
        self.spectralPeakID             = "."
        self.peakVolumeValue            = "."
##        self.peakIntensityValue         = "."
##        self.peakSpectralDimensionID    = "."
        self.peakChemicalShiftValue     = "."
        self.contributionValue          = "."

##        self.subsystemAuthorCode        = "."
        self.residueAuthorSeqCode       = "."
        self.residueAuthorLabel         = "."
        self.atomAuthorName             = "."


def _printWarn( place, text ):
    print "Warning from place: %s" % place
    print text


def _formatSTAR( filename ):
    "Running Steve Madings (BMRB) formatNMRSTAR program if available"
    ## Assume for now that this the program is not available
    print "Skipped pretty printing the STAR file"
    return 0

    print "Attempting to reformat STAR file using external program if available"
    
    if os.name != 'posix':
        tmpStr = "Non-Unix doesn't have formatNMRSTAR anyway"
##        raise OSError,tmpStr
        return 0

##  Try command and check for non-zero exit status
##  Note that these commands are only valid on Unix
##  A better way for the temporary file names/locations should be used.
    cmd = "%s < %s" % (FormatNMRSTAR_executable, filename)
    pipe = os.popen( cmd )
    output = pipe.read()
    
##  The program exit status is available by the following construct
##  The status will be the exit number unless the program executed
##  succesfully in which case it will be None.
    status = pipe.close()

    ## Success    
    if ( status == None ):
        try:
            outhandle = TextFile.TextFile(filename, 'w')
        except IOError:
            print 'could not open the file for writing', fileName
            return
        outhandle.write(output)
        outhandle.close()
        print "Reformatted STAR file"
    else:
        print "WARNING: Failed reformatting STAR file"


def _emptyObject( object ):
    if object == None:
        return 1            # Object contains special None value
                            # Object with type None always has the value None and will
                            # therefore return on this statement too
    if ( (type(object) is not StringType) and\
         (type(object) is not IntType) ):
        text = "Bug: didn't code for object type: %s\nWith value: %s" % \
               (type(object), `object`)
        raise TypeError, text
    
    if type(object) is StringType:
        objectCopy = string.strip( object )
        if ( objectCopy == "" ):
            #text = "Object (%s) is white space only" % object
            #_printWarn(__name__, text)
            return 1            # Object contains string with 
                                # whte space only (\t\n\r\v)
                                # from beginning to end
        else:
            return 0        # String object is not empty
    else:
        return 0            # Object is not empty


class BmrbCmdParser(cmd.Cmd):
    "Taken from example of the rolodex O'Reilly's 'Learning Python', p271"
    def __init__(self):
        cmd.Cmd.__init__(self)  
        self.prompt = "BMRB command parser: "
        self.print_topics

        if os.name == 'posix':
            self.root = '/'       ## Unix
        elif os.name == 'nt':
            ## Windows disk to use instead of the root path, e.g. 'h:\\'
##            self.root = Windows_disk_name
            self.root = 'h:\\'
        else:
            tmpStr = \
                "This program was designed for operating " + \
                "systems unix or windows, sorry"
            raise OSError,tmpStr

    def help_convert_cs(self):
        print "Converts chemical shifts to STAR format"
    def do_convert_cs(self, name):
        self.inputSeqFileName    = raw_input("Enter Input  Sequence File Name: ")
        self.inputShiftFileName  = raw_input("Enter Input  Shift    File Name: ")
        self.outputStarFileName  = raw_input("Enter Output Star     File Name: ")

        PPM = PpmList.PpmList(self.inputFileName, 3 )
        # dimension given here (3) is not important for conversion right?

        ## A new way programmed by Jens
        PPM.ReadXeasyProt(self.inputSeqFileName)
        PPM.Stdout()
        
        PPM.WriteBioMagResBank(self.outputStarFileName)

        ## Reformat the STAR file if program is available/executable
        _formatSTAR( self.outputStarFileName )

    def help_convert_tbl(self):
        print "Converts ARIA distance constraints to STAR format"
    def do_convert_tbl(self, name):
        self.inputFileName = raw_input("Enter Input File Name: ")
        self.outputFileName = raw_input("Enter Output File Name: ")

        #testDir                = os.path.join( self.root, 'usr', 'home', 'jurgen', 'aria', 'aria1.0', 'test' )        
        #self.inputFileName     = os.path.join( testDir, 'test.tbl' )
        #self.outputFileName    = os.path.join( testDir, 'test.str' )

                
        NL = NoeList.NoeList()
        NL.ReadTbl( self.inputFileName )        
        
        DRL = DistanceRestraintList()
        DRL.AbsorbNoeList( NL )
        DRL.WriteStar( self.outputFileName )

        # Reformat the STAR file if program is available/executable
        _formatSTAR( self.outputFileName )

    def cmdexit(self):
        print "Bye now\n"
        sys.exit(0)
        
    def emptyline(self):
        self.cmdexit()
        
    def help_EOF(self):
        print "Quits the program"
    def do_EOF(self, line):
        self.cmdexit()
        

class English:
    def __init__(self):
        self.numbers = ( "zero",
            "one", "two", "three", "four", "five",\
            "six", "seven", "eight", "nine", "ten" )

if __name__ == '__main__':   
    parser = BmrbCmdParser() 
    parser.cmdloop()
