"""
This is the main PYTHON script for running ARIA from a UNIX command line.
All important information is parsed out of the run.cns or new.html of your 
current run of your project.

For a comprehensive description, read the manual html pages.

Usage for setting up a new project:
1. Use the ARIA HTML page for setting up of a new project, save new.html
   in your ARIA program directory
2. Go in the directory where new.html was saved
3. call the executable /aria/Aria/RunAria.py

Usage for running ARIA: 
1.  Go in that directory where your current run.cns file was saved
2.  call the executable /aria/Aria/RunAria.py

Copyright: Jens Linge and Michael Nilges (EMBL Heidelberg)
"""
__ariaVersion__ = "1.0"
__author__   = "$Author: linge $"
__revision__ = "$Revision: 1.10 $"
__date__     = "$Date: 2000/04/06 11:54:16 $"


###############################################################################
import copy, glob, os, re, string, shutil, sys, time

#print sys.path
try:
    from Aria.Analysis import CnsAnalysis, Contacts, ProjectStatus
    from Aria.CNS import CallCns
    from Aria.DataIO import ConvertData, InputFileParser, NoeList, PpmList
    from Aria.Main import MAria, Messages, ParsePath, Setup
    from Aria.ThirdParty import DictWithDefault, TextFile
    from Aria.Water import WaterRefinement
except ImportError:
    print """
There was an ImportError with importing the ARIA Python modules.
Most probably, you haven't set the UNIX system variable $PYTHONPATH properly.

Try:
    echo $PYTHONPATH
    
It should contain something like /home/linge/aria1.0

Use:
    setenv PYTHONPATH /home/linge/aria1.0
in csh or tcsh.

Or use:
    export PYTHONPATH='/home/linge/aria1.0'
in zsh or bash.

Read the installation notes again if you should have more problems with
importing the ARIA Python modules.
"""
    sys.exit()
    
#write the first message to STDOUT:
Messages.StartAria(__ariaVersion__)

###############################################################################
#looking for new.html (for setup) or run.cns in current directory:
if (os.path.exists('new.html') == 0) and (os.path.exists('run.cns') == 0):
    print 'there is no run.cns OR new.html in your current directory.'
    print '=> ARIA stopped'
    Messages.StopAria()
elif (os.path.exists('new.html') == 1) and (os.path.exists('run.cns') == 1):
    print 'there is a new.html AND a run.cns in your current directory.'
    print 'remove one of them.'
    print '=> ARIA stopped'
    Messages.StopAria()
    
###############################################################################
#1. if new.html exists: parse new.html and set up a new project or a new run
if os.path.exists('new.html') == 1:
    new = DictWithDefault.DictWithDefault(None)
    #newparsed is a simple dictionary:
    newparsed = {}
    #get the dictionary for the variables:
    newparsed = InputFileParser.ParseNewHtml()
    #copy all the parameters to 'new':
    for kkparsed in newparsed.keys():
	new[kkparsed] = newparsed[kkparsed]

    #for a new run:
    if new['WHICH_SETUP'] == 'NEWRUN':
        Setup.SetupNewRun(new['ARIA_DIR'],\
                          new['PROJECT_DIR'], new['OLDRUN_NUMBER'],\
                          new['OLDIT_NUMBER'], new['RUN_NUMBER'],\
                          new['PROTOCOLS_COPY'], new['TOPPAR_COPY'])
        print 'run' + new['RUN_NUMBER'], 'has been set up.'
        print 'Now you have to edit run.cns in', new['PROJECT_DIR'] +\
              '/run' + new['RUN_NUMBER']
        Messages.StopAria()

    #for a new project:
    new['RUN_DIR'] = new['PROJECT_DIR'] + '/run' + \
                     new['RUN_NUMBER']  #just add this for the runs

    #set up the project:
    Setup.SetupNewProject(new['WHICH_SETUP'], new['ARIA_DIR'], \
                          new['PROJECT_DIR'], \
                          new['RUN_NUMBER'], new['ASPECTRUM_1'], \
                          new['ASPECTRUM_2'], new['ASPECTRUM_3'],\
                          new['ASPECTRUM_4'], new['ASPECTRUM_5'],\
                          new['SEQ_PDB_FILE'], new['DIHED_FILE'],\
                          new['C1_FILE'], new['C2_FILE'],\
                          new['C3_FILE'], new['C4_FILE'],\
                          new['C5_FILE'],\
                          new['RDC1_FILE'], new['RDC2_FILE'],\
                          new['RDC3_FILE'], new['RDC4_FILE'],\
                          new['RDC5_FILE'],\
                          new['HBOND_FILE'], new['CSI_FILE'],\
                          new['PROT_SEGID_1'])

    #for already calibrated data in the ARIA .tbl format:
    if new['WHICH_SETUP'] == 'ARIA_CAL_TBL':
	print 'copying ARIA format data, data already calibrated'
        os.mkdir(new['RUN_DIR'] + '/data/allSpectra')
        if new['UNAMBIG_TBL']:
            if os.path.exists(new['UNAMBIG_TBL']) == 1:
                copyunambigtbl2data = new['RUN_DIR'] + '/data/allSpectra/unambig.tbl'
                copyunambigtbl2it0 = new['RUN_DIR']+'/structures/it0/unambig.tbl'
                shutil.copyfile(new['UNAMBIG_TBL'], copyunambigtbl2data)
                print '  copying unambiguous data to', copyunambigtbl2data 
                shutil.copyfile(new['UNAMBIG_TBL'], copyunambigtbl2it0)
                print '  copying unambiguous data to',  copyunambigtbl2it0
            else:
                print '  could not find', new['UNAMBIG_TBL']
                Messages.StopAria()
        else: #touch to get an empty file
            touchFile = open(new['RUN_DIR'] + '/data/allSpectra/unambig.tbl', 'w')
            touchFile.close()
            
	if new['AMBIG_TBL']:
	    if os.path.exists(new['AMBIG_TBL']) == 1:
		copyambigtbl2data = new['RUN_DIR'] + '/data/allSpectra/ambig.tbl'
		copyambigtbl2it0 = new['RUN_DIR']+'/structures/it0/ambig.tbl'
		shutil.copyfile(new['AMBIG_TBL'], copyambigtbl2data)
		print '  copying ambiguous data to', copyambigtbl2data
		shutil.copyfile(new['AMBIG_TBL'], copyambigtbl2it0)
		print '  copying ambiguous data to', copyambigtbl2it0
	    else:
		print 'could not find', new['AMBIG_TBL']
                Messages.StopAria()
        else: #touch to get an empty file
            touchFile = open(new['RUN_DIR'] + '/data/allSpectra/ambig.tbl', 'w')
            touchFile.close()
        #create one big file with all the data:
        unambigHandle = open(new['RUN_DIR'] + '/data/allSpectra/unambig.tbl')
        ambigHandle = open(new['RUN_DIR'] + '/data/allSpectra/ambig.tbl')
        oneBigFile = open(new['RUN_DIR'] + '/data/allSpectra/allSpectra.tbl', 'w')
        oneBigFile.write('!unambiguous data:\n')
        oneBigFile.write(unambigHandle.read())
        oneBigFile.write('\n\n!ambiguous data:\n')
        oneBigFile.write(ambigHandle.read())
        oneBigFile.close()

    #for uncalibrated data in the ARIA .tbl format:
    elif new['WHICH_SETUP'] == 'ARIA_UNCAL_TBL':
        print 'use uncalibrated data in ARIA .tbl format'
        if new['SPECTRUM1_TBLWITHC'] and new['SPECTRUM1_PPMWITHC']:
            ConvertData.AriaTblWithC(new['SPECTRUM1_TBLWITHC'], new['SPECTRUM1_PPMWITHC'],\
                                     new['ASPECTRUM_1'], new['RUN_DIR'])
        if new['SPECTRUM2_TBLWITHC'] and new['SPECTRUM2_PPMWITHC']:
            ConvertData.AriaTblWithC(new['SPECTRUM2_TBLWITHC'], new['SPECTRUM2_PPMWITHC'],\
                                     new['ASPECTRUM_2'], new['RUN_DIR'])
        if new['SPECTRUM3_TBLWITHC'] and new['SPECTRUM3_PPMWITHC']:
            ConvertData.AriaTblWithC(new['SPECTRUM3_TBLWITHC'], new['SPECTRUM3_PPMWITHC'],\
                                     new['ASPECTRUM_3'], new['RUN_DIR'])
        if new['SPECTRUM4_TBLWITHC'] and new['SPECTRUM4_PPMWITHC']:
            ConvertData.AriaTblWithC(new['SPECTRUM4_TBLWITHC'], new['SPECTRUM4_PPMWITHC'],\
                                     new['ASPECTRUM_4'], new['RUN_DIR'])
        if new['SPECTRUM5_TBLWITHC'] and new['SPECTRUM5_PPMWITHC']:
            ConvertData.AriaTblWithC(new['SPECTRUM5_TBLWITHC'], new['SPECTRUM5_PPMWITHC'],\
                                     new['ASPECTRUM_5'], new['RUN_DIR'])

    #for uncalibrated ARIA data in the .list format:
    elif new['WHICH_SETUP'] == 'ARIA_UNCAL_LIST':
        print 'convert uncalibrated data in the ARIA .list format'
        if new['SPECTRUM1_UNCAL_LIST']:
            ConvertData.AriaUncalList(new['SPECTRUM1_UNCAL_LIST'], new['SPECTRUM1_PPM'],\
                                      new['ASPECTRUM_1'], new['RUN_DIR'])
        if new['SPECTRUM2_UNCAL_LIST']:
            ConvertData.AriaUncalList(new['SPECTRUM2_UNCAL_LIST'], new['SPECTRUM2_PPM'],\
                                      new['ASPECTRUM_2'], new['RUN_DIR'])
        if new['SPECTRUM3_UNCAL_LIST']:
            ConvertData.AriaUncalList(new['SPECTRUM3_UNCAL_LIST'], new['SPECTRUM3_PPM'],\
                                      new['ASPECTRUM_3'], new['RUN_DIR'])
        if new['SPECTRUM4_UNCAL_LIST']:
            ConvertData.AriaUncalList(new['SPECTRUM4_UNCAL_LIST'], new['SPECTRUM4_PPM'],\
                                      new['ASPECTRUM_4'], new['RUN_DIR'])
        if new['SPECTRUM5_UNCAL_LIST']:
            ConvertData.AriaUncalList(new['SPECTRUM5_UNCAL_LIST'], new['SPECTRUM5_PPM'],\
                                      new['ASPECTRUM_5'], new['RUN_DIR'])
                   
    #for calibrated ARIA data in the .list format:
    elif new['WHICH_SETUP'] == 'ARIA_CAL_LIST':
        print 'convert calibrated data in the ARIA .list format'
        if new['SPECTRUM1_CAL_LIST']:
            ConvertData.AriaCalList(new['SPECTRUM1_CAL_LIST'], new['SPECTRUM1_PPM'],\
                                    new['ASPECTRUM_1'], new['RUN_DIR'])
        if new['SPECTRUM2_CAL_LIST']:
            ConvertData.AriaCalList(new['SPECTRUM2_CAL_LIST'], new['SPECTRUM2_PPM'],\
                                    new['ASPECTRUM_2'], new['RUN_DIR'])
        if new['SPECTRUM3_CAL_LIST']:
            ConvertData.AriaCalList(new['SPECTRUM3_CAL_LIST'], new['SPECTRUM3_PPM'],\
                                    new['ASPECTRUM_3'], new['RUN_DIR'])
        if new['SPECTRUM4_CAL_LIST']:
            ConvertData.AriaCalList(new['SPECTRUM4_CAL_LIST'], new['SPECTRUM4_PPM'],\
                                    new['ASPECTRUM_4'], new['RUN_DIR'])
        if new['SPECTRUM5_CAL_LIST']:
            ConvertData.AriaCalList(new['SPECTRUM5_CAL_LIST'], new['SPECTRUM5_PPM'],\
                                    new['ASPECTRUM_5'], new['RUN_DIR'])
                  
    #for uncalibrated data in the ARIA .tbl format:
    elif new['WHICH_SETUP'] == 'REGINE':
        print 'use uncalibrated data in Regine format'
        if new['SPECTRUM1_PEAKS'] and new['SPECTRUM1_SHIFTS']:
            ConvertData.Regine2Aria(new['SPECTRUM1_PEAKS'], new['SPECTRUM1_SHIFTS'],\
                                    new['ASPECTRUM_1'], new['RUN_DIR'],\
                                    new['PPMDHET1_1'], new['PPMDPRO1_1'],\
                                    new['PPMDHET2_1'], new['PPMDPRO2_1'])
        if new['SPECTRUM2_PEAKS'] and new['SPECTRUM2_SHIFTS']:
            ConvertData.Regine2Aria(new['SPECTRUM2_PEAKS'], new['SPECTRUM2_SHIFTS'],\
                                    new['ASPECTRUM_2'], new['RUN_DIR'],\
                                    new['PPMDHET1_2'], new['PPMDPRO1_2'],\
                                    new['PPMDHET2_2'], new['PPMDPRO2_2'])
        if new['SPECTRUM3_PEAKS'] and new['SPECTRUM3_SHIFTS']:
            ConvertData.Regine2Aria(new['SPECTRUM3_PEAKS'], new['SPECTRUM3_SHIFTS'],\
                                    new['ASPECTRUM_3'], new['RUN_DIR'],\
                                    new['PPMDHET1_3'], new['PPMDPRO1_3'],\
                                    new['PPMDHET2_3'], new['PPMDPRO2_3'])
        if new['SPECTRUM4_PEAKS'] and new['SPECTRUM4_SHIFTS']:
            ConvertData.Regine2Aria(new['SPECTRUM4_PEAKS'], new['SPECTRUM4_SHIFTS'],\
                                    new['ASPECTRUM_4'], new['RUN_DIR'],\
                                    new['PPMDHET1_4'], new['PPMDPRO1_4'],\
                                    new['PPMDHET2_4'], new['PPMDPRO2_4'])
        if new['SPECTRUM5_PEAKS'] and new['SPECTRUM5_SHIFTS']:
            ConvertData.Regine2Aria(new['SPECTRUM5_PEAKS'], new['SPECTRUM5_SHIFTS'],\
                                    new['ASPECTRUM_5'], new['RUN_DIR'],\
                                    new['PPMDHET1_5'], new['PPMDPRO1_5'],\
                                    new['PPMDHET2_5'], new['PPMDPRO2_5'])
 
    #for XEASY data format:
    elif new['WHICH_SETUP'] == 'XEASY':
	if new['XPROT_1']:
            #build the filename for the output files:
            copypeaks = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_1'] + '/' + new['ASPECTRUM_1'] + '.peaks'
            copyprot = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_1'] + '/' + new['ASPECTRUM_1'] + '.prot'
            idfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_1'] + '/' + new['ASPECTRUM_1'] + '.id'
            chemfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_1'] + '/' + new['ASPECTRUM_1'] + '.ppm'
            tblfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_1'] + '/' + new['ASPECTRUM_1'] + '.tbl'
            listfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_1'] + '/' + new['ASPECTRUM_1'] + '.list'
            #copy the XEASY file to /data:
            print 'copying', new['XPEAKS_1'], 'to',  copypeaks
            print 'copying', new['XPROT_1'], 'to',  copyprot
            shutil.copyfile(new['XPEAKS_1'], copypeaks)
            shutil.copyfile(new['XPROT_1'], copyprot)
            #convert the data in .tbl format:
            NOE= NoeList.NoeList('spectrum1')
            if new['XASSIGN_1']:
                copyassign = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_1'] + '/' + new['ASPECTRUM_1'] + '.assign'
                print 'copying', new['XASSIGN_1'], 'to',  copyassign
                shutil.copyfile(new['XASSIGN_1'], copyassign)
                NOE.ReadAssignPeaksProt(new['XASSIGN_1'], new['XPEAKS_1'], new['XPROT_1'],\
                                        new['HET1_1'], new['PRO1_1'],\
                                        new['HET2_1'], new['PRO2_1'])
            else:
                NOE.ReadPeaksProt(new['XPEAKS_1'], new['XPROT_1'],\
                                  new['HET1_1'], new['PRO1_1'],\
                                  new['HET2_1'], new['PRO2_1'])
            NOE.WriteTbl(tblfile, new['PPMDHET1_1'], new['PPMDPRO1_1'],\
                         new['PPMDHET2_1'], new['PPMDPRO2_1'],\
                         new['WHICH_PEAKS_1'])
            NOE.WriteList(listfile)
            #read .prot file and write .ppm file:
            PPM = PpmList.PpmList()
            PPM.ReadXeasyProt(new['XPROT_1'])
            PPM.WriteChem(chemfile)
            PPM.WriteId(idfile)
            del(PPM)
            del(NOE)
	if new['XPROT_2']:
            #build the filename for the output files:
            copypeaks = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_2'] + '/' + new['ASPECTRUM_2'] + '.peaks'
            copyprot = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_2'] + '/' + new['ASPECTRUM_2'] + '.prot'
            idfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_2'] + '/' + new['ASPECTRUM_2'] + '.id'
            chemfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_2'] + '/' + new['ASPECTRUM_2'] + '.ppm'
            tblfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_2'] + '/' + new['ASPECTRUM_2'] + '.tbl'
            listfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_2'] + '/' + new['ASPECTRUM_2'] + '.list'
            #copy the XEASY file to /data:
            print 'copying', new['XPEAKS_2'], 'to',  copypeaks
            print 'copying', new['XPROT_2'], 'to',  copyprot
            shutil.copyfile(new['XPEAKS_2'], copypeaks)
            shutil.copyfile(new['XPROT_2'], copyprot)
            #convert the data in .tbl format:
            NOE= NoeList.NoeList('spectrum2')
            if new['XASSIGN_2']:
                copyassign = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_2'] + '/' + new['ASPECTRUM_2'] + '.assign'
                print 'copying', new['XASSIGN_2'], 'to',  copyassign
                shutil.copyfile(new['XASSIGN_2'], copyassign)
                NOE.ReadAssignPeaksProt(new['XASSIGN_2'], new['XPEAKS_2'], new['XPROT_2'],\
                                        new['HET1_2'], new['PRO1_2'],\
                                        new['HET2_2'], new['PRO2_2'])
            else:
                NOE.ReadPeaksProt(new['XPEAKS_2'], new['XPROT_2'],\
                                  new['HET1_2'], new['PRO1_2'],\
                                  new['HET2_2'], new['PRO2_2'])
            NOE.WriteTbl(tblfile, new['PPMDHET1_2'], new['PPMDPRO1_2'],\
                         new['PPMDHET2_2'], new['PPMDPRO2_2'],\
                         new['WHICH_PEAKS_2'])
            NOE.WriteList(listfile)
            #read .prot file and write .ppm file:
            PPM = PpmList.PpmList()
            PPM.ReadXeasyProt(new['XPROT_2'])
            PPM.WriteChem(chemfile)
            PPM.WriteId(idfile)
            del(PPM)
            del(NOE)
	if new['XPROT_3']:
            #build the filename for the output files:
            copypeaks = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_3'] + '/' + new['ASPECTRUM_3'] + '.peaks'
            copyprot = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_3'] + '/' + new['ASPECTRUM_3'] + '.prot'
            idfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_3'] + '/' + new['ASPECTRUM_3'] + '.id'
            chemfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_3'] + '/' + new['ASPECTRUM_3'] + '.ppm'
            tblfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_3'] + '/' + new['ASPECTRUM_3'] + '.tbl'
            listfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_3'] + '/' + new['ASPECTRUM_3'] + '.list'
            #copy the XEASY file to /data:
            print 'copying', new['XPEAKS_3'], 'to',  copypeaks
            print 'copying', new['XPROT_3'], 'to',  copyprot
            shutil.copyfile(new['XPEAKS_3'], copypeaks)
            shutil.copyfile(new['XPROT_3'], copyprot)
            #convert the data in .tbl format:
            NOE= NoeList.NoeList('spectrum3')
            if new['XASSIGN_3']:
                copyassign = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_3'] + '/' + new['ASPECTRUM_3'] + '.assign'
                print 'copying', new['XASSIGN_3'], 'to',  copyassign
                shutil.copyfile(new['XASSIGN_3'], copyassign)
                NOE.ReadAssignPeaksProt(new['XASSIGN_3'], new['XPEAKS_3'], new['XPROT_3'],\
                                        new['HET1_3'], new['PRO1_3'],\
                                        new['HET2_3'], new['PRO2_3'])
            else:
                NOE.ReadPeaksProt(new['XPEAKS_3'], new['XPROT_3'],\
                                  new['HET1_3'], new['PRO1_3'],\
                                  new['HET2_3'], new['PRO2_3'])
            NOE.WriteTbl(tblfile, new['PPMDHET1_3'], new['PPMDPRO1_3'],\
                         new['PPMDHET2_3'], new['PPMDPRO2_3'],\
                         new['WHICH_PEAKS_3'])
            NOE.WriteList(listfile)
            #read .prot file and write .ppm file:
            PPM = PpmList.PpmList()
            PPM.ReadXeasyProt(new['XPROT_3'])
            PPM.WriteChem(chemfile)
            PPM.WriteId(idfile)
            del(PPM)
            del(NOE)
	if new['XPROT_4']:
            #build the filename for the output files:
            copypeaks = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_4'] + '/' + new['ASPECTRUM_4'] + '.peaks'
            copyprot = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_4'] + '/' + new['ASPECTRUM_4'] + '.prot'
            idfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_4'] + '/' + new['ASPECTRUM_4'] + '.id'
            chemfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_4'] + '/' + new['ASPECTRUM_4'] + '.ppm'
            tblfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_4'] + '/' + new['ASPECTRUM_4'] + '.tbl'
            listfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_4'] + '/' + new['ASPECTRUM_4'] + '.list'
            #copy the XEASY file to /data:
            print 'copying', new['XPEAKS_4'], 'to',  copypeaks
            print 'copying', new['XPROT_4'], 'to',  copyprot
            shutil.copyfile(new['XPEAKS_4'], copypeaks)
            shutil.copyfile(new['XPROT_4'], copyprot)
            #convert the data in .tbl format:
            NOE= NoeList.NoeList('spectrum4')
            if new['XASSIGN_4']:
                copyassign = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_4'] + '/' + new['ASPECTRUM_4'] + '.assign'
                print 'copying', new['XASSIGN_4'], 'to',  copyassign
                shutil.copyfile(new['XASSIGN_4'], copyassign)
                NOE.ReadAssignPeaksProt(new['XASSIGN_4'], new['XPEAKS_4'], new['XPROT_4'],\
                                        new['HET1_4'], new['PRO1_4'],\
                                        new['HET2_4'], new['PRO2_4'])
            else:
                NOE.ReadPeaksProt(new['XPEAKS_4'], new['XPROT_4'],\
                                  new['HET1_4'], new['PRO1_4'],\
                                  new['HET2_4'], new['PRO2_4'])
            NOE.WriteTbl(tblfile, new['PPMDHET1_4'], new['PPMDPRO1_4'],\
                         new['PPMDHET2_4'], new['PPMDPRO2_4'],\
                         new['WHICH_PEAKS_4'])
            NOE.WriteList(listfile)
            #read .prot file and write .ppm file:
            PPM = PpmList.PpmList()
            PPM.ReadXeasyProt(new['XPROT_4'])
            PPM.WriteChem(chemfile)
            PPM.WriteId(idfile)
            del(PPM)
            del(NOE)
	if new['XPROT_5']:
            #build the filename for the output files:
            copypeaks = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_5'] + '/' + new['ASPECTRUM_5'] + '.peaks'
            copyprot = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_5'] + '/' + new['ASPECTRUM_5'] + '.prot'
            idfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_5'] + '/' + new['ASPECTRUM_5'] + '.id'
            chemfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_5'] + '/' + new['ASPECTRUM_5'] + '.ppm'
            tblfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_5'] + '/' + new['ASPECTRUM_5'] + '.tbl'
            listfile = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_5'] + '/' + new['ASPECTRUM_5'] + '.list'
            #copy the XEASY file to /data:
            print 'copying', new['XPEAKS_5'], 'to',  copypeaks
            print 'copying', new['XPROT_5'], 'to',  copyprot
            shutil.copyfile(new['XPEAKS_5'], copypeaks)
            shutil.copyfile(new['XPROT_5'], copyprot)
            #convert the data in .tbl format:
            NOE= NoeList.NoeList('spectrum5')
            if new['XASSIGN_5']:
                copyassign = new['RUN_DIR'] + '/data/' + new['ASPECTRUM_5'] + '/' + new['ASPECTRUM_5'] + '.assign'
                print 'copying', new['XASSIGN_5'], 'to',  copyassign
                shutil.copyfile(new['XASSIGN_5'], copyassign)
                NOE.ReadAssignPeaksProt(new['XASSIGN_5'], new['XPEAKS_5'], new['XPROT_5'],\
                                        new['HET1_5'], new['PRO1_5'],\
                                        new['HET2_5'], new['PRO2_5'])
            else:
                NOE.ReadPeaksProt(new['XPEAKS_5'], new['XPROT_5'],\
                                  new['HET1_5'], new['PRO1_5'],\
                                  new['HET2_5'], new['PRO2_5'])
            NOE.WriteTbl(tblfile, new['PPMDHET1_5'], new['PPMDPRO1_5'],\
                         new['PPMDHET2_5'], new['PPMDPRO2_5'],\
                         new['WHICH_PEAKS_5'])
            NOE.WriteList(listfile)
            #read .prot file and write .ppm file:
            PPM = PpmList.PpmList()
            PPM.ReadXeasyProt(new['XPROT_5'])
            PPM.WriteChem(chemfile)
            PPM.WriteId(idfile)
            del(PPM)
            del(NOE)


##     #for AURELIA data format:
##     elif new['WHICH_SETUP'] == 'AURELIA':
## 	if new['SPECTRUM1_DAT']:
## 	    ConvertData.Aurelia2aria(new['SPECTRUM1_DAT'], new['SPECTRUM1_PPM'],\
##                                new['PPMD1_1'], new['PPMD2_1'],\
##                                new['PPMDHET_1'],\
##                                new['RUN_DIR'], new['ASPECTRUM_1'])
## 	if new['SPECTRUM2_DAT']:
## 	    ConvertData.Aurelia2aria(new['SPECTRUM2_DAT'], new['SPECTRUM2_PPM'],\
##                                new['PPMD1_2'], new['PPMD2_2'],\
##                                new['PPMDHET_2'],\
##                                new['RUN_DIR'], new['ASPECTRUM_2'])
## 	if new['SPECTRUM3_DAT']):
## 	    ConvertData.Aurelia2aria(new['SPECTRUM3_DAT'], new['SPECTRUM3_PPM'],\
##                                new['PPMD1_3'], new['PPMD2_3'],\
##                                new['PPMDHET_3'],\
##                                new['RUN_DIR'], new['ASPECTRUM_3'])
## 	if new['SPECTRUM4_DAT']:
## 	    ConvertData.Aurelia2aria(new['SPECTRUM4_DAT'], new['SPECTRUM4_PPM'],\
##                                new['PPMD1_4'], new['PPMD2_4'],\
##                                new['PPMDHET_4'],\
##                                new['RUN_DIR'], new['ASPECTRUM_4'])
## 	if new['SPECTRUM5_DAT']):
## 	    ConvertData.Aurelia2aria(new['SPECTRUM5_DAT'], new['SPECTRUM5_PPM'],\
##                                new['PPMD1_5'], new['PPMD2_5'],\
##                                new['PPMDHET_5'],\
##                                new['RUN_DIR'], new['ASPECTRUM_5'])

    #for ANSIG data:
    elif new['WHICH_SETUP'] == 'ANSIG':
        if new['ASPECTRUM_1']:
            ConvertData.Ansig2Aria(new['RUN_DIR'], new['ASPECTRUM_1'],\
                                   new['ANSIGNOE_1'], new['ANSIGPPM_1'],\
                                   new['HET1_1'], new['PRO1_1'],\
                                   new['HET2_1'], new['PRO2_1'],\
                                   new['PPMDHET1_1'], new['PPMDPRO1_1'],\
                                   new['PPMDHET2_1'], new['PPMDPRO2_1'],\
                                   new['WHICH_PEAKS_1'])
        if new['ASPECTRUM_2']:
            ConvertData.Ansig2Aria(new['RUN_DIR'], new['ASPECTRUM_2'],\
                                   new['ANSIGNOE_2'], new['ANSIGPPM_2'],\
                                   new['HET1_2'], new['PRO1_2'],\
                                   new['HET2_2'], new['PRO2_2'],\
                                   new['PPMDHET1_2'], new['PPMDPRO1_2'],\
                                   new['PPMDHET2_2'], new['PPMDPRO2_2'],\
                                   new['WHICH_PEAKS_2'])    
        if new['ASPECTRUM_3']:
            ConvertData.Ansig2Aria(new['RUN_DIR'], new['ASPECTRUM_3'],\
                                   new['ANSIGNOE_3'], new['ANSIGPPM_3'],\
                                   new['HET1_3'], new['PRO1_3'],\
                                   new['HET2_3'], new['PRO2_3'],\
                                   new['PPMDHET1_3'], new['PPMDPRO1_3'],\
                                   new['PPMDHET2_3'], new['PPMDPRO2_3'],\
                                   new['WHICH_PEAKS_3'])
        if new['ASPECTRUM_4']:
            ConvertData.Ansig2Aria(new['RUN_DIR'], new['ASPECTRUM_4'],\
                                   new['ANSIGNOE_4'], new['ANSIGPPM_4'],\
                                   new['HET1_4'], new['PRO1_4'],\
                                   new['HET2_4'], new['PRO2_4'],\
                                   new['PPMDHET1_4'], new['PPMDPRO1_4'],\
                                   new['PPMDHET2_4'], new['PPMDPRO2_4'],\
                                   new['WHICH_PEAKS_4'])
        if new['ASPECTRUM_5']:
            ConvertData.Ansig2Aria(new['RUN_DIR'], new['ASPECTRUM_5'],\
                                   new['ANSIGNOE_5'], new['ANSIGPPM_5'],\
                                   new['HET1_5'], new['PRO1_5'],\
                                   new['HET2_5'], new['PRO2_5'],\
                                   new['PPMDHET1_5'], new['PPMDPRO1_5'],\
                                   new['PPMDHET2_5'], new['PPMDPRO2_5'],\
                                   new['WHICH_PEAKS_5'])
            
    elif new['WHICH_SETUP'] == 'PIPPPCK':
        if new['ASPECTRUM_1']:
            ConvertData.Pipp2Aria(new['RUN_DIR'], new['ASPECTRUM_1'],\
                                  new['PIPPPCK_1'], new['PIPPSHIFTS_1'],\
                                  new['HET1_1'], new['PRO1_1'],\
                                  new['HET2_1'], new['PRO2_1'],\
                                  new['PPMDHET1_1'], new['PPMDPRO1_1'],\
                                  new['PPMDHET2_1'], new['PPMDPRO2_1'],\
                                  new['WHICH_PEAKS_1'], new['PIPPASS1_1'],\
                                  new['PIPPASS2_1'])
        if new['ASPECTRUM_2']:
            ConvertData.Pipp2Aria(new['RUN_DIR'], new['ASPECTRUM_2'],\
                                  new['PIPPPCK_2'], new['PIPPSHIFTS_2'],\
                                  new['HET1_2'], new['PRO1_2'],\
                                  new['HET2_2'], new['PRO2_2'],\
                                  new['PPMDHET1_2'], new['PPMDPRO1_2'],\
                                  new['PPMDHET2_2'], new['PPMDPRO2_2'],\
                                  new['WHICH_PEAKS_2'], new['PIPPASS1_2'],\
                                  new['PIPPASS2_2'])  
        if new['ASPECTRUM_3']:
            ConvertData.Pipp2Aria(new['RUN_DIR'], new['ASPECTRUM_3'],\
                                  new['PIPPPCK_3'], new['PIPPSHIFTS_3'],\
                                  new['HET1_3'], new['PRO1_3'],\
                                  new['HET2_3'], new['PRO2_3'],\
                                  new['PPMDHET1_3'], new['PPMDPRO1_3'],\
                                  new['PPMDHET2_3'], new['PPMDPRO2_3'],\
                                  new['WHICH_PEAKS_3'], new['PIPPASS1_3'],\
                                  new['PIPPASS2_3'])  
        if new['ASPECTRUM_4']:
            ConvertData.Pipp2Aria(new['RUN_DIR'], new['ASPECTRUM_4'],\
                                  new['PIPPPCK_4'], new['PIPPSHIFTS_4'],\
                                  new['HET1_4'], new['PRO1_4'],\
                                  new['HET2_4'], new['PRO2_4'],\
                                  new['PPMDHET1_4'], new['PPMDPRO1_4'],\
                                  new['PPMDHET2_4'], new['PPMDPRO2_4'],\
                                  new['WHICH_PEAKS_4'], new['PIPPASS1_4'],\
                                  new['PIPPASS2_4'])
        if new['ASPECTRUM_5']:
            ConvertData.Pipp2Aria(new['RUN_DIR'], new['ASPECTRUM_5'],\
                                  new['PIPPPCK_5'], new['PIPPSHIFTS_5'],\
                                  new['HET1_5'], new['PRO1_5'],\
                                  new['HET2_5'], new['PRO2_5'],\
                                  new['PPMDHET1_5'], new['PPMDPRO1_5'],\
                                  new['PPMDHET2_5'], new['PPMDPRO2_5'],\
                                  new['WHICH_PEAKS_5'], new['PIPPASS1_5'],\
                                  new['PIPPASS2_5'])  

    #NMRView setup with Kristy's perl scripts:
    elif new['WHICH_SETUP'] == 'NMRVIEW':
        if new['ASPECTRUM_1']:
            ConvertData.Nmrview2Aria(new['RUN_DIR'], new['ARIA_DIR'],\
                                     new['ASPECTRUM_1'], new['DIMENSION_1'],\
                                     new['NMRVIEWXPK_1'], new['NMRVIEWPPM_1'])
        if new['ASPECTRUM_2']:
            ConvertData.Nmrview2Aria(new['RUN_DIR'], new['ARIA_DIR'],\
                                     new['ASPECTRUM_2'], new['DIMENSION_2'],\
                                     new['NMRVIEWXPK_2'], new['NMRVIEWPPM_2'])
        if new['ASPECTRUM_3']:
            ConvertData.Nmrview2Aria(new['RUN_DIR'], new['ARIA_DIR'],\
                                     new['ASPECTRUM_3'], new['DIMENSION_3'],\
                                     new['NMRVIEWXPK_3'], new['NMRVIEWPPM_3'])
        if new['ASPECTRUM_4']:
            ConvertData.Nmrview2Aria(new['RUN_DIR'], new['ARIA_DIR'],\
                                     new['ASPECTRUM_4'], new['DIMENSION_4'],\
                                     new['NMRVIEWXPK_4'], new['NMRVIEWPPM_4'])
        if new['ASPECTRUM_5']:
            ConvertData.Nmrview2Aria(new['RUN_DIR'], new['ARIA_DIR'],\
                                     new['ASPECTRUM_5'], new['DIMENSION_5'],\
                                     new['NMRVIEWXPK_5'], new['NMRVIEWPPM_5'])
            
            
    #waterrefinement:
    elif new['WHICH_SETUP'] == 'WATERREFINEMENT':
        pass
    
    #generation of psf and template files:
    elif new['WHICH_SETUP'] == 'GENERATION':
        pass
    
    #end of the setup:
    print 'new project',  new['PROJECT_DIR'], 'has been set up.'
    print 'Now you have to edit run.cns in', new['RUN_DIR']
    Messages.StopAria()

	
###############################################################################
#2. if run.cns exists:
elif os.path.exists('run.cns') == 1:
    #parse variables out of run.cns in current directory
    print 'parsing run.cns file'
    run = {}
    run = InputFileParser.ParseRunCns()    #define a dictionary run (variablename:value)

    #parse new.html in the data directory:
    print 'parsing new.html in', run['datadir']
    new = DictWithDefault.DictWithDefault(None)
    #newparsed is a simple dictionary:
    newparsed = {}
    #get the variables from new.html:
    newparsed = InputFileParser.ParseNewHtml(run['datadir'])
    #copy from newparsed to new:
    for kkparsed in newparsed.keys():
	new[kkparsed] = newparsed[kkparsed]
        
    #set segid to four spaces if it is not used:
    if (string.strip(run['prot_segid_1']) == ''):
        run['prot_segid_i'] = '    '
    
    print 'looking for existing files'
    status = {}
    status = ProjectStatus.GetStatus(run)
    
    #get the integer values of the cpunumber, set 0 if '' or None:
    if run['cpunumber_1']:
        cpuNumber1 = string.atoi(run['cpunumber_1'])
    else:
        cpuNumber1 = 0
    if run['cpunumber_2']:
        cpuNumber2 = string.atoi(run['cpunumber_2'])
    else:
        cpuNumber2 = 0
    if run['cpunumber_3']:
        cpuNumber3 = string.atoi(run['cpunumber_3'])
    else:
        cpuNumber3 = 0
    if run['cpunumber_4']:
        cpuNumber4 = string.atoi(run['cpunumber_4'])
    else:
        cpuNumber4 = 0
    if run['cpunumber_5']:
        cpuNumber5 = string.atoi(run['cpunumber_5'])
    else:
        cpuNumber5 = 0
    if run['cpunumber_6']:
        cpuNumber6 = string.atoi(run['cpunumber_6'])
    else:
        cpuNumber6 = 0
    if run['cpunumber_7']:
        cpuNumber7 = string.atoi(run['cpunumber_7'])
    else:
        cpuNumber7 = 0
    if run['cpunumber_8']:
        cpuNumber8 = string.atoi(run['cpunumber_8'])
    else:
        cpuNumber8 = 0
    if run['cpunumber_9']:
        cpuNumber9 = string.atoi(run['cpunumber_9'])
    else:
        cpuNumber9 = 0
    if run['cpunumber_10']:
        cpuNumber10 = string.atoi(run['cpunumber_10'])
    else:
        cpuNumber10 = 0

    #create the queueDic:
    queueDic = {}
    if run['queue_1']:
        queueDic[run['queue_1']] = [run['cns_exe_1'], cpuNumber1, []]
    if run['queue_2']:
        queueDic[run['queue_2']] = [run['cns_exe_2'], cpuNumber2, []]
    if run['queue_3']:
        queueDic[run['queue_3']] = [run['cns_exe_3'], cpuNumber3, []]
    if run['queue_4']:
        queueDic[run['queue_4']] = [run['cns_exe_4'], cpuNumber4, []]
    if run['queue_5']:
        queueDic[run['queue_5']] = [run['cns_exe_5'], cpuNumber5, []]
    if run['queue_6']:
        queueDic[run['queue_6']] = [run['cns_exe_6'], cpuNumber6, []]
    if run['queue_7']:
        queueDic[run['queue_7']] = [run['cns_exe_7'], cpuNumber7, []]
    if run['queue_8']:
        queueDic[run['queue_8']] = [run['cns_exe_8'], cpuNumber8, []]
    if run['queue_9']:
        queueDic[run['queue_9']] = [run['cns_exe_9'], cpuNumber9, []]
    if run['queue_10']:
        queueDic[run['queue_10']] = [run['cns_exe_10'], cpuNumber10, []]

    
    #get the runPlusNumber as string (e.g. 'run12'):
    runPlusNumber = ParsePath.GetTail(run['run_dir'])

    if status['psffile'] == 0:
        print 'generating PSF file with the command:'
        #write the temporary job files in the tempTrashDir:
        jobString = """cd %s
setenv ARIA %s
setenv CURRIT %s
setenv NEWIT %s
setenv PREVIT %s
setenv RUN %s
setenv SPECTRUM %s
setenv TEMPTRASH %s
%s < %s > %s
        """ % (run['temptrash_dir'],\
               run['aria_dir'],
               '0',\
               '',\
               '',\
               run['run_dir'],\
               '',\
               run['temptrash_dir'],\
               run['cns_exe_1'],\
               run['protocolsdir'] + '/generate.inp',\
               run['begindir'] + '/generate.out')
        jobFN = os.path.join(run['temptrash_dir'], run['fileroot'] + '_' +\
                             runPlusNumber +\
                             '_generate.job')
        jobStream = open(jobFN, 'w')
        jobStream.write(jobString)
        jobStream.close()
        
        #start the job with the first queue:
        queueList = queueDic.keys()
        whatToDo = queueList[0] + ' ' + jobFN + ' &'
        print ' ', whatToDo
        os.system(whatToDo)
        
        #wait for the .psf file:
        print '  waiting for the psf file...'
        while 1:
            if status['psffile']: break
            status = ProjectStatus.GetStatus(run)
            time.sleep(1)
        
    #generating template file:
    if status['templatefile'] == 0:
        print 'generating template file with the command:'
        time.sleep(2) #just to make sure that .psf file is written out...
        #write the temporary job files in the tempTrashDir:
        jobString = """cd %s
setenv ARIA %s
setenv CURRIT %s
setenv NEWIT %s
setenv PREVIT %s
setenv RUN %s
setenv SPECTRUM %s
setenv TEMPTRASH %s
%s < %s > %s
        """ % (run['temptrash_dir'],\
               run['aria_dir'],
               '0',\
               '',\
               '',\
               run['run_dir'],\
               '',\
               run['temptrash_dir'],\
               run['cns_exe_1'],\
               run['protocolsdir'] + '/generate_template.inp',\
               run['begindir'] + '/generate_template.out')
        jobFN = os.path.join(run['temptrash_dir'], run['fileroot'] + '_' +\
                             runPlusNumber +\
                             '_generate_template.job')
        jobStream = open(jobFN, 'w')
        jobStream.write(jobString)
        jobStream.close()
        
        #start the job with the first queue:
        queueList = queueDic.keys()
        whatToDo = queueList[0] + ' ' + jobFN + ' &'
        print ' ', whatToDo
        os.system(whatToDo)
        
        #wait for the template file:
        print '  waiting for the template file...'
        while 1:
            if status['templatefile']: break
            status = ProjectStatus.GetStatus(run)
            time.sleep(1)

    for currit in range(0, 9):
        #for already assigned data in .tbl format, stop after it0:
        if currit > 0 and new['WHICH_SETUP'] == 'ARIA_ASSIGN':
            print 'If you start with already calibrated data, only iteration 0 will be calculated.'
            print '=> ARIA stopped.'
#            Messages.StopAria()
        
        run['newit'] = os.path.join(run['run_dir'], 'structures/it'+str(currit))
	status = ProjectStatus.GetStatus(run)
	print 'looking for iteration', currit
	fileliststatus = 'filelist' + str(currit)
	if status[fileliststatus] == 0:
	    print 'working on iteration ' + str(currit)
	    if currit == 0:
		run['previt'] = os.path.join(run['run_dir'], 'begin')
	    else:
		lastit = currit - 1
		run['previt'] = os.path.join(run['run_dir'], 'structures/it'+str(lastit))

            #set some values for the free rfactor testing: #test
            if not run.has_key('rfree_on'): #test
                run['rfree_on'] = 'false' #test
                run['rfree_oneset'] = 'false' #test
                run['rfree_percent'] = '5' #test
                run['rfree_method'] =  'jackknife' #test
                
	    MAria.ForAllIterations(run['run_dir'], run['protocolsdir'],\
                                   run['newit'], run['temptrash_dir'],\
                                   currit, run['nspectra'],\
                                   run['aspectrum_1'],
                                   run['aspectrum_2'], run['aspectrum_3'],\
                                   run['aspectrum_4'], run['aspectrum_5'],\
                                   run['md_type'],\
                                   run['fileroot'], run['previt'],\
                                   run['templatefile'],\
                                   run['rfree_on'], run['rfree_oneset'],\
                                   run['rfree_percent'], run['rfree_method'],\
                                   run['ini_count_' + str(currit)],\
                                   run['structures_' + str(currit)],\
                                   run['keepstruct_' + str(currit)],\
                                   run['aria_dir'], run)
	
            #data conversion:
            if new['WHICH_SETUP'] == 'XEASY':
                print 'starting data conversion to XEASY'
                for ppp in range(1, 6):
                    ppp = str(ppp)
                    outS = "spectrumx = 'aspectrum_" + ppp + "';"
                    outS = outS + "het1x = new['HET1_" + ppp + "'];"
                    outS = outS + "pro1x = new['PRO1_" + ppp + "'];"
                    outS = outS + "het2x = new['HET2_" + ppp + "'];"
                    outS = outS + "pro2x = new['PRO2_" + ppp + "'];"
                    exec outS
                    if string.strip(run[spectrumx]) != '':
                        print 'working on spectrum', run[spectrumx]
                        #input:
                        oldlistname = run['newit'] + '/' + run[spectrumx] + '.list'
                        datapeaksname = run['run_dir'] + '/data/' + run[spectrumx] + '/' +\
                                        run[spectrumx] + '.peaks'
                        dataprotname = run['run_dir'] + '/data/' + run[spectrumx] + '/' +\
                                        run[spectrumx] + '.prot'
                        #output:
                        newassignpeaksroot = run['newit'] + '/' + run[spectrumx]
                        molmolroot = run['newit']+ '/' + run[spectrumx]
                        #conversion:
                        NOE = NoeList.NoeList()
                        try:
                            NOE.ReadListPeaksProt(oldlistname,\
                                                  datapeaksname, dataprotname,\
                                                  het1x, pro1x, het2x, pro2x)
                        except:
                            print 'WARNING: An error occured with NOE.ReadListPeaksProt'
                        #write XEASY .assign and .peaks file:
                        try:
                            NOE.WriteXeasyAssignPeaks(newassignpeaksroot,\
                                                      het1x, pro1x, het2x, pro2x)
                        except:
                            print 'WARNING: An error occured with NOE.WriteXeasyAssignPeaks'
                        #write .upl and .lol files for molmol:
#                        NOE.WriteUplLol(molmolroot)
#                        del(NOE)
            else:
                for ppp in range(1, 6):
                    ppp = str(ppp)
                    outS = "spectrumx = 'aspectrum_" + ppp + "';"
                    outS = outS + "het1x = new['HET1_" + ppp + "'];"
                    outS = outS + "pro1x = new['PRO1_" + ppp + "'];"
                    outS = outS + "het2x = new['HET2_" + ppp + "'];"
                    outS = outS + "pro2x = new['PRO2_" + ppp + "'];"
                    exec outS
                    if string.strip(run[spectrumx]) != '':
                        #input:
                        oldlistname = run['newit'] + '/' + run[spectrumx] + '.list'
                        #output:
                        molmolroot = run['newit'] + '/' + run[spectrumx]
#                        #conversion:
#                        NOE = NoeList.NoeList()
#                        NOE._ReadOldList(oldlistname)
#                        #write .upl and .lol files for molmol:
#                        NOE.WriteUplLol(molmolroot)
#                        del(NOE)

        else:
	    print 'file.list exists => iteration', \
		  currit, 'finished.'

#create final .tbl and .list files after iteration 8:
    if currit == 8:
        expectedTblFiles = []
        if status['finaldata'] == 0:
            print 'last calibration and merging of the data after it8'
            print '  writing files to', run['run_dir'] + '/structures/it8/finaldata'
            spectrumtogo = [run['aspectrum_1'], run['aspectrum_2'], \
                            run['aspectrum_3'], run['aspectrum_4'], run['aspectrum_5']]
            for spectrumy in spectrumtogo:
                if string.strip(spectrumy):
                    print '    calibrating spectrum', spectrumy

                    #look which queue is free, take that:
                    queueIndex = 0
                    while 1:
                        queueList = queueDic.keys()
                        eachQueue = queueList[queueIndex]
                        #1. if empty, take it directly:
                        if queueDic[eachQueue][1] > len(queueDic[eachQueue][2]):
                            break
                        #2. if fileNames exist, check if the file already exists:
                        else:
                            breakOut = 0
                            for eachF in queueDic[eachQueue][2]:
                                if os.path.exists(eachF):
                                    queueDic[eachQueue][2].remove(eachF)
                                    breakOut = 1
                            if breakOut: break
                        if queueIndex == (len(queueList) - 1):
                            queueIndex = 0
                        else:
                            queueIndex = queueIndex + 1
                        time.sleep(2)

                    #write the temporary job files in the tempTrashDir:
                    jobString = """cd %s
    setenv ARIA %s
    setenv CURRIT %s
    setenv NEWIT %s
    setenv PREVIT %s
    setenv RUN %s
    setenv SPECTRUM %s
    setenv TEMPTRASH %s
    %s < %s > %s
            """ % (run['temptrash_dir'],\
                   run['aria_dir'],\
                   '8',\
                   run['run_dir'] + '/structures/it8/finaldata',\
                   run['run_dir'] + '/structures/it8',\
                   run['run_dir'],\
                   run['run_dir'] + '/data/' + spectrumy,\
                   run['temptrash_dir'],\
                   queueDic[eachQueue][0],\
                   run['run_dir'] + '/protocols/calib.inp',\
                   run['run_dir'] + '/structures/it8/finaldata/calib_' +\
                   spectrumy + '_it' + str(currit) + '.out')
                    jobFN = os.path.join(run['temptrash_dir'], run['fileroot'] + '_' +\
                                         runPlusNumber +\
                                         '_it' + str(currit) +\
                                         '_' + spectrumy + '.job')
                    jobStream = open(jobFN, 'w')
                    jobStream.write(jobString)
                    jobStream.close()

                    #add the expected output file to the list:
                    expectedTblFiles.append(run['run_dir'] + '/structures/it8/finaldata/' +\
                                            spectrumy + '.tbl')
                    queueDic[eachQueue][2].append(run['run_dir'] + '/structures/it8/finaldata/' +\
                                                  spectrumy + '.tbl')
                    
                    #start the queue with the proper command and job file:
                    whatToDo = eachQueue + ' ' + ParsePath.GetTail(jobFN) + ' &'
                    print '      queue command:\n     ', whatToDo
                    os.chdir(run['temptrash_dir'])
                    os.system(whatToDo)

                    #move to next queue (after the last one, start from 0 again):
                    if queueIndex == (len(queueList) - 1):
                        queueIndex = 0
                    else:
                        queueIndex = queueIndex + 1

            #wait until all the spectra are calibrated:
            print '    waiting for the calibrated spectra...'
            while 1:
                spectraTblFiles = glob.glob(run['run_dir'] + '/structures/it8/finaldata/*.tbl')
                for eachTbl in spectraTblFiles:
                    if eachTbl in expectedTblFiles:
                        expectedTblFiles.remove(eachTbl)
                if len(expectedTblFiles) == 0:
                    break
                time.sleep(1)
            
            time.sleep(3) #wait for slow networks

            if string.atoi(run['nspectra']) != 0:
                print '    merging of the data'
                queueList = queueDic.keys()
                jobString = """cd %s
    setenv ARIA %s
    setenv CURRIT %s
    setenv NEWIT %s
    setenv PREVIT %s
    setenv RUN %s
    setenv SPECTRUM %s
    setenv TEMPTRASH %s
    %s < %s > %s
            """ % (run['temptrash_dir'],\
                   run['aria_dir'],\
                   '8',\
                   run['run_dir'] + '/structures/it8/finaldata',\
                   run['run_dir'] + '/structures/it8',\
                   run['run_dir'],\
                   '',\
                   run['temptrash_dir'],\
                   queueDic[queueList[0]][0],\
                   run['run_dir'] + '/protocols/merge.inp',\
                   run['run_dir'] + '/structures/it8/finaldata/merge_it' + str(currit) + '.out')
                jobFN = os.path.join(run['temptrash_dir'], run['fileroot'] + '_' +\
                                     runPlusNumber + '_it_merge.job')
                jobStream = open(jobFN, 'w')
                jobStream.write(jobString)
                jobStream.close()
                #start the queue with the proper command and job file:
                whatToDo = queueList[0] + ' ' + ParsePath.GetTail(jobFN) + ' &'
                print '      queue command:\n     ', whatToDo
                os.chdir(run['temptrash_dir'])
                os.system(whatToDo)

                #wait until all the spectra are calibrated:
                print '    waiting for the merged NOE list...'
                while 1:
                    if os.path.exists(run['run_dir'] + '/structures/it8/finaldata/unambig.tbl') and\
                       os.path.exists(run['run_dir'] + '/structures/it8/finaldata/ambig.tbl'):
                        break
                    time.sleep(1)
        else:
            print 'the final data files already exist.'


        #print molmol files:
        print 'writing MOLMOL files to /structures/it8:'
        try:
            spectrumtogo = [run['aspectrum_1'], run['aspectrum_2'], \
                            run['aspectrum_3'], run['aspectrum_4'], run['aspectrum_5']]
            for spectrumy in spectrumtogo:
                if string.strip(spectrumy):
                    print '  reading spectrum', spectrumy
                    NL=NoeList.NoeList()
                    NL.ReadList(run['run_dir'] + '/structures/it8/' + spectrumy + '.list')
                    NL.WriteUplLol(run['run_dir'] + '/structures/it8/' + spectrumy)
        except:
            print 'could not write out MOLMOL files. An error occured. Skipping it.'

            
#analysis of iteration 8:
        if status['analysis'] == 0:
##            #1. ariaAnalysis module:
##            try:
##                ariaAnalysis.analyseIteration(ariaAnalysis.ariaIteration(run['run_dir'] +\
##                                                                         '/structures/it'+str(currit)))
##            except:
##                print 'WARNING: ariaAnalysis.py raised some errors!'
##                print '         check the sources if you want this kind of analysis.'
##            try:
##                ariaAnalysis.analyseAriaRun(['it0', 'it1', 'it2', 'it3', 'it4',\
##                                             'it5', 'it6', 'it7', 'it8'], \
##                                            run['run_dir'] + '/structures/')
##            except:
##                print 'WARNING: ariaAnalysis.py raised some errors!'
##                print '         check the source code if you want this kind of analysis.'
                  
            #2. CNS protocols:
            CnsAnalysis.Analysis(run['run_dir'], run['aria_dir'], run['protocolsdir'],\
                                 run['newit'], run['temptrash_dir'],\
                                 currit, queueDic, run['fileroot'])
        else:
            print '    analysis files already exist.'

#water refinement of iteration 8:
    if currit == 8 and run['firstwater'] == "yes":
        WaterRefinement.WaterRefine(run['run_dir'], run['protocolsdir'],\
                                    run['newit'], run['temptrash_dir'],\
                                    currit, run['fileroot'],\
                                    string.atoi(run['waterrefine']),\
                                    run['aria_dir'], run)

        #analysis of the waterrefined structures (after iteration 8):
        #1. copy unambig.tbl and ambig.tbl from it8 to it8/water:
        shutil.copyfile(run['newit'] + '/ambig.tbl', run['newit'] + '/water/ambig.tbl')
        shutil.copyfile(run['newit'] + '/unambig.tbl', run['newit'] + '/water/unambig.tbl')
        #2. start the cns protocols:
        CnsAnalysis.Analysis(run['run_dir'], run['aria_dir'], run['protocolsdir'],\
                             run['newit'] + '/water', run['temptrash_dir'],\
                             currit, queueDic, run['fileroot'])
        

###############################################################################
#regular end of ARIA:
Messages.StopAria()
