"""
This module provides methods for the conversion of cns atomnames in
pseudoatomes and vice versa.

The nomenclature of the pseudoatoms is in the same spirit as in:
Markley et al. JMB 280, 933-952 (1998)

There are 3 methods:
    Pseudo2Atom(atom)   e.g. 'QA' -> 'ha#'
                             'QR' -> 'hd# or name he# or name hz'
    Pseudo2Tuple(atom)  e.g. 'QA' -> ('ha#',)
                             'QR' -> ('hd#', 'he#', 'hz#')
    Atom2Pseudo(atomname, aminoacid)
                        e.g. 'HB1', 'TYR' -> ('QB', )
                             'HG11', 'VAL' ->  ('QG1', 'QQG')
                             
"""
__author__   = "$Author: linge $"
__revision__ = "$Revision: 1.1.1.1 $"
__date__     = "$Date: 1999/09/02 13:35:23 $"


import string


###############################################################################
def Pseudo2Atom(atom):
    """
    converts pseudoatoms into a CNS readable format
    input:   atom name (can be a pseudoatom)
    output:  CNS readable atom name
    """    
    replacedic = {'QA': 'ha#',
		  'QB': 'hb#',
		  'QG': 'hg#',
		  'QG1': 'hg1#',
		  'QG2': 'hg2#',
		  'QQG': 'hg#',
		  'QD': 'hd#',
		  'QD1': 'hd1#',
		  'QD2': 'hd2#',
		  'QQD': 'hd#',
		  'QE': 'he#',
		  'QE2': 'he2#',
		  'QR': 'hd# or name he# or name hz',
		  'QZ': 'hz#',
		  'QH1': 'hh1#',
		  'QH2': 'hh2#'}
    atom = string.upper(atom)
    atom = string.strip(atom)
    if replacedic.has_key(atom):
        atom = replacedic[atom]
    return atom


###############################################################################
def Pseudo2Tuple(atom):
    """
    converts pseudoatoms into a tuple of atomnames in a CNS readable format
    input:   atom name (can be a pseudoatom)
    output:  tuple of CNS readable atoms
    """    
    replacedic = {'QA': ('ha#', ),
		  'QB': ('hb#', ),
		  'QG': ('hg#', ),
		  'QG1': ('hg1#', ),
		  'QG2': ('hg2#', ),
		  'QQG': ('hg#', ),
		  'QD': ('hd#', ),
		  'QD1': ('hd1#', ),
		  'QD2': ('hd2#', ),
		  'QQD': ('hd#', ),
		  'QE': ('he#', ),
		  'QE2': ('he2#', ),
		  'QR': ('hd#', 'he#', 'hz'),
		  'QZ': ('hz#', ),
		  'QH1': ('hh1#', ),
		  'QH2': ('hh2#', )}
    atom = string.upper(atom)
    atom = string.strip(atom)
    if replacedic.has_key(atom):
	    atom = replacedic[atom]
    else:
        atom = (atom, )
    return atom


###############################################################################
def Atom2Pseudo(atomname, aminoacid):
    """
    IMPUT:   cns atomname and aminoacid type in 3-letter code
    OUTPUT:  returns a tuple of possible pseudoatoms, best hit comes first
             within the tuple
             if it is not possible to find a pseudoatom, an empty tuple
             will be returned

    The nomenclature of the pseudoatoms is in the same spirit as in:
    Markley et al. JMB 280, 933-952 (1998)
    easy:       ALA, ASP, ASN, CYS, GLN, GLU, GLY, HIS, LYS, MET,
                PRO, SER, THR, TRP
    ambiguous:  ARG: QH1, QH2 or QH
                ILE: QG1, QG2 or QQG
                LEU: QD1, QD2 or QQD
                PHE: QR
                TYR: QR
                VAL: QG1, QG2 or QQG
    """
    pseudoatom = ()
    aminoacid = string.upper(aminoacid)
    aminoacid = string.strip(aminoacid)
    atomname = string.upper(atomname)
    atomname = string.strip(atomname)
    if aminoacid == 'ALA':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
    elif aminoacid == 'ARG':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
        elif atomname in ['HG1', 'HG2', 'HG3', 'HG%', 'HG#']:
            pseudoatom = ('QG', )
        elif atomname in ['HD1', 'HD2', 'HD3', 'HD%', 'HD#']:
            pseudoatom = ('QD', )
        elif atomname in ['HH11', 'HH12', 'HH13', 'HH1%', 'HH1#']:
            pseudoatom = ('QH1', )
        elif atomname in ['HH21', 'HH22', 'HH23', 'HH2%', 'HH2#']:
            pseudoatom = ('QH2', )
    elif aminoacid == 'ASN':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
        elif atomname in ['HD1', 'HD2', 'HD3', 'HD%', 'HD#']:
            pseudoatom = ('QD2', )
    elif aminoacid == 'ASP':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
    elif aminoacid == 'CYS':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
    elif aminoacid == 'GLU':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
        elif atomname in ['HG1', 'HG2', 'HG3', 'HG%', 'HG#']:
            pseudoatom = ('QG', )   
    elif aminoacid == 'GLN':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
        elif atomname in ['HG1', 'HG2', 'HG3', 'HG%', 'HG#']:
            pseudoatom = ('QG', )
        elif atomname in ['HE21', 'HE22', 'HE2%', 'HE2#']:
            pseudoatom = ('QE2', )
    elif aminoacid == 'GLY':
        if atomname in ['HA1', 'HA2', 'HA3', 'HA%', 'HA#']:
            pseudoatom = ('QA', )
    elif aminoacid == 'HIS':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
    elif aminoacid == 'ILE':
        if atomname in ['HD11', 'HD12', 'HD13', 'HD1%', 'HD1#']:
            pseudoatom = ('QD1', )
        elif atomname in ['HG11', 'HG12', 'HG13', 'HG1%', 'HG1#']:
            pseudoatom = ('QG1', 'QQG')
        elif atomname in ['HG21', 'HG22', 'HG23', 'HG2%', 'HG2#']:
            pseudoatom = ('QG2', 'QQG')
        elif atomname[:2] == 'HG':
            pseudoatom == ('QQG', 'QG1', 'QG2')
    elif aminoacid == 'LEU':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
        elif atomname in ['HD11', 'HD12', 'HD13', 'HD1%', 'HD1#']:
            pseudoatom = ('QD1', 'QQD')
        elif atomname in ['HD21', 'HD22', 'HD23', 'HD2%', 'HD2#']:
            pseudoatom = ('QD2', 'QQD')
        elif atomname[:2] == 'HD':
            pseudoatom = ('QQD', 'QD1', 'QD2')
    elif aminoacid == 'LYS':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
        elif atomname in ['HG1', 'HG2', 'HG3', 'HG%', 'HG#']:
            pseudoatom = ('QG', )
        elif atomname in ['HD1', 'HD2', 'HD3', 'HD%', 'HD#']:
            pseudoatom = ('QD', )   
        elif atomname in ['HE1', 'HE2', 'HE3', 'HE%', 'HE#']:
            pseudoatom = ('QE', )
        elif atomname in ['HZ1', 'HZ2', 'HZ3', 'HZ%', 'HZ#']:
            pseudoatom = ('QZ', )
    elif aminoacid == 'MET':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
        elif atomname in ['HG1', 'HG2', 'HG3', 'HG%', 'HG#']:
            pseudoatom = ('QG', )
        elif atomname in ['HE1', 'HE2', 'HE3', 'HE%', 'HE#']:
            pseudoatom = ('QE', )    
    elif aminoacid == 'PHE':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
        elif atomname in ['HD1', 'HD2', 'HD3', 'HD%', 'HD#']:
            pseudoatom = ('QD', 'QR')   
        elif atomname in ['HE1', 'HE2', 'HE3', 'HE%', 'HE#']:
            pseudoatom = ('QE', 'QR')
    elif aminoacid == 'PRO':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
        elif atomname in ['HG1', 'HG2', 'HG3', 'HG%', 'HG#']:
            pseudoatom = ('QG', )   
        elif atomname in ['HD1', 'HD2', 'HD3', 'HD%', 'HD#']:
            pseudoatom = ('QD', )
    elif aminoacid == 'SER':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
    elif aminoacid == 'THR':
        if atomname in ['HG21', 'HG22', 'HG23', 'HG2%', 'HG2#']:
            pseudoatom = ('QG2', )
    elif aminoacid == 'TRP':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
    elif aminoacid == 'TYR':
        if atomname in ['HB1', 'HB2', 'HB3', 'HB%', 'HB#']:
            pseudoatom = ('QB', )
        elif atomname in ['HD1', 'HD2', 'HD3', 'HD%', 'HD#']:
            pseudoatom = ('QD', 'QR')   
        elif atomname in ['HE1', 'HE2', 'HE3', 'HE%', 'HE#']:
            pseudoatom = ('QE', 'QR')
    elif aminoacid == 'VAL':
        if atomname in ['HG11', 'HG12', 'HG13', 'HG1%', 'HG1#']:
            pseudoatom = ('QG1', 'QQG')
        elif atomname in ['HG21', 'HG22', 'HG23', 'HG2%', 'HG2#']:
            pseudoatom = ('QG2', 'QQG')
        elif pseudoatom[:2] == 'HG':
            pseudoatom = ('QQG', 'QG1', 'QG2')
    return pseudoatom

