"""
ModifyList.py
a class to read and write CNS modify.list files

CNS nomenclature:
    modify "ABCD" 1234 test=1 end

Jens Linge 3.8.99
"""
__author__   = "$Author: linge $"
__revision__ = "$Revision: 1.1.1.1 $"
__date__     = "$Date: 1999/09/02 13:35:19 $"

import re, string

class ModifyList:
    """
    contains a list of modifications

    attributes:
        modifications      a list of OneModification instances

    public methods:
        AddModification    adds a OneModification instance
        ReadList           reads a CNS modification list
        Sort               sorting of the self.modifications list
        Stdout             writes all the data to stdout
        WriteList          writes a CNS modification list
    """
    def __init__(self):
        self.modifications = []

    def AddModification(self, spectrumName, peakNumber, **pairs):
        """
        adds a modification
        """
        spectrumName = str(spectrumName)
        peakNumber = str(peakNumber)
        ONEMOD = OneModification(spectrumName, peakNumber, pairs)
        self.modifications.append(ONEMOD)
        
    def ReadList(self, fileName):
        """
        reads a list in CNS readable format
        """
        from Aria.Comments import DeleteCnsComments
        bigString = DeleteCnsComments.GetString(fileName)
        #replace 'test = 1' by 'test=1':
        equalSign = re.compile('\s+=\s+')
        bigString = equalSign.sub('=', bigString)
        #create a list with the 'end' as last entry:
        bigList = string.split(bigString, 'modify')[1:]
        for eachMod in bigList:
            modList = string.split(eachMod)
            spectrumName = modList[0]
            peakNumber = modList[1]
            listIndex = 2
            pairDic = {}
            while 1:
                if string.lower(modList[listIndex]) == 'end':
                    break
                variable, value = string.split(modList[listIndex], '=')
                pairDic[variable] = value

                listIndex = listIndex + 1
            ONEMOD=OneModification(spectrumName, peakNumber, pairDic)
            self.modifications.append(ONEMOD)

    def Sort(self):
        """
        sorts all the data:   1. after spectrumName
                              2. after peakNumber
        """
        from Aria.DataIO import Comparisons
        self.modifications.sort(Comparisons.CmpComposite(Comparisons.CmpAttr('spectrumName', 0),\
                                                         Comparisons.CmpAttr('peakNumber', 1)))

        
    def Stdout(self):
        """
        writes all the data to stdout
        """
        outList = self._OutPutString()
        for line in outList:
            print line
    
    def WriteList(self, fileName):
        """
        writes a list in CNS readable format
        """
        outList = self._OutPutString()
        try:  
            fileHandle = open(fileName, 'w')
        except:
            print 'WARNING: could not write to', fileName
            print 'writeList() aborted!'
            return
        for line in outList:
            fileHandle.write(line + '\n')
        fileHandle.close()

    def _OutPutString(self):
        """
        helper method which gives back a list containing all the lines
        with the data in the CNS modify format

        used for WriteList and Stdout methods
        """
        outList = []
        for EACH in self.modifications:
            attrList = dir(EACH)
            attrList.remove('spectrumName')
            attrList.remove('peakNumber')
            for element in attrList:
                outString = 'modify %s %s %s=%s end' % (EACH.spectrumName, EACH.peakNumber,\
                                                        element, getattr(EACH, element))
                outList.append(outString)
        return outList
                                                             

class OneModification:
    """
    serves as a data container for one modification statement
    
    attributes:
        spectrumName
        peakNumber
        modifications are given as dictionary

    access this class only through its constructor
    only use strings as input parameters
    
    e.g.
        OM=OneModification('ABCD', '1234', {'test':'1'})
        creates the following attributes:
            OM.spectrumName = 'ABCS'
            OM.peakNumber = '1234'
            OM.test = '1'

    Please note that all values are stored internally as strings!
    """
    def __init__(self, spectrumName, peakNumber, pairDictionary):
        self.spectrumName = str(spectrumName)
        self.peakNumber = str(peakNumber)
        for keyword in pairDictionary.keys():
            setattr(self, str(keyword), str(pairDictionary[keyword]))


#test code:
if __name__ == "__main__":
    print 'testing module ModifyList:'
    ML=ModifyList()
    ML.AddModification('ABCD', '1234', test=0)
    ML.AddModification('EFGH', '5678', test=1)
    ML.Stdout()
