########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.         #
#                                                                                      #
########################################################################################

import unittest2 as unittest
import os
import sys
import shutil
from lxml import etree
from time import sleep
from signal import SIGTERM

MOBYLEHOME = os.path.abspath( os.path.join( os.path.dirname( __file__ ) , "../../../" ) )
os.environ['MOBYLEHOME'] = MOBYLEHOME

if ( MOBYLEHOME ) not in sys.path:
    sys.path.append( MOBYLEHOME )
if ( os.path.join( MOBYLEHOME , 'Src' ) ) not in sys.path:    
    sys.path.append( os.path.join( MOBYLEHOME , 'Src' ) )

import Mobyle.Test.MobyleTest
from Mobyle.StatusManager import StatusManager
from Mobyle.Status import Status
from Mobyle.MobyleError import MobyleError

DATADIR = os.path.dirname( __file__ )

class StatusManagerTest(unittest.TestCase):
    """Tests the functionalities of Transaction"""
    
    def setUp(self):
        """ 
        setting up the configuration for the test, including:
        - test configuration that does not check dns or try to send confirmation emails
        - test session email
        """
        self.cfg = Mobyle.ConfigManager.Config()
        shutil.rmtree(self.cfg.test_dir, ignore_errors=True)
        os.makedirs(self.cfg.test_dir)
        self.jobKey = "Q12345678901234"
        self.jobDir = os.path.join( self.cfg.test_dir , self.jobKey )
        os.makedirs( self.jobDir )
    
    def tearDown(self):
        #shutil.rmtree( self.cfg.test_dir , ignore_errors= True )
        pass
    
    def testCreation(self):
        #create( filename , status )
        unknown = Status( code = -1 )
        StatusManager.create( self.jobDir , unknown )
        filename = os.path.normpath( os.path.join( self.jobDir , StatusManager.file_name ) )
        doc = etree.parse( filename  ) 
        root = doc.getroot()
        self.assertEqual( root.tag , 'status' )
        children = list(root)
        self.assertEqual( len( children ) , 2 )
        self.assertEqual( children[0].tag , 'value')
        self.assertEqual( children[0].text , 'unknown' )
        self.assertEqual( children[1].tag , 'message')
        self.assertEqual( children[1].text , None )
        os.unlink(filename )
        
        building = Status( code = 0 )
        StatusManager.create( self.jobDir, building )
        doc = etree.parse( filename  ) 
        root = doc.getroot()
        self.assertEqual( root.tag , 'status' )
        children = list(root)
        self.assertEqual( len( children ) , 2 )
        self.assertEqual( children[0].tag , 'value')
        self.assertEqual( children[0].text , 'building' )
        self.assertEqual( children[1].tag , 'message')
        self.assertEqual( children[1].text , None )
        os.unlink(filename )
        
        submitted = Status( code = 1, message= 'test message' )
        StatusManager.create( self.jobDir, submitted )
        doc = etree.parse( filename  ) 
        root = doc.getroot()
        self.assertEqual( root.tag , 'status' )
        children = list(root)
        self.assertEqual( len( children ) , 2 )
        self.assertEqual( children[0].tag , 'value')
        self.assertEqual( children[0].text , 'submitted' )
        self.assertEqual( children[1].tag , 'message')
        self.assertEqual( children[1].text , 'test message' )
        os.unlink(filename )
        
        running = Status( string='running' )
        StatusManager.create( self.jobDir, running ) 
        doc = etree.parse( filename  ) 
        root = doc.getroot()
        self.assertEqual( root.tag , 'status' )
        children = list(root)
        self.assertEqual( len( children ) , 2 )
        self.assertEqual( children[0].tag , 'value')
        self.assertEqual( children[0].text , 'running' )
        self.assertEqual( children[1].tag , 'message')
        self.assertEqual( children[1].text , None )
        os.unlink(filename )
        

    def testGetStatus( self ):
        running = Status( string='running' )
        filename = os.path.normpath( os.path.join( self.jobDir , StatusManager.file_name ) )
        StatusManager.create( self.jobDir, running )
        sm = StatusManager()
        recieved_status = sm.getStatus( self.jobDir )
        self.assertEqual( recieved_status , running )
        os.unlink(filename )

        killed = Status( string='killed' , message= "your job has been canceled" )
        StatusManager.create( self.jobDir, killed )
        sm = StatusManager()
        recieved_status = sm.getStatus( self.jobDir )
        self.assertEqual( recieved_status , killed )
        os.unlink(filename )
        

    def testSetstatus( self):
        filename = os.path.normpath( os.path.join( self.jobDir , StatusManager.file_name ) )
        StatusManager.create( self.jobDir, Status( string='submitted' ) )
        
        pending = Status( string= 'pending' )
        sm = StatusManager()
        sm.setStatus( self.jobDir , pending )
        recieved_status = sm.getStatus( self.jobDir )
        self.assertEqual( recieved_status , pending )
        
        finished = Status( string='finished' , message = 'your job finnished with an unusual status code, check youre results carefully') 
        sm.setStatus( self.jobDir , finished )
        recieved_status = sm.getStatus( self.jobDir )
        self.assertEqual( recieved_status , finished )
        
        #an ended status cannot be changed anymore 
        running= Status( string= 'running')
        sm.setStatus( self.jobDir , running )
        recieved_status = sm.getStatus( self.jobDir )
        self.assertNotEqual( recieved_status , running )
        self.assertEqual( recieved_status , finished )
        os.unlink(filename )
        
    def testConcurency(self):
        import fcntl 
        
        filename = os.path.normpath( os.path.join( self.jobDir , StatusManager.file_name ) )
        status = Status( string='submitted' )
        StatusManager.create( self.jobDir, status )
        
        ## sub-process start
        childPid = os.fork()
        if childPid: #father
            sleep(1)
            sm = StatusManager()
            self.assertEqual( status , sm.getStatus( self.jobDir ) )
            self.assertRaises( IOError , sm.setStatus , self.jobDir, status )
            
        else: #child
            File = open( filename , 'r' )
            fcntl.lockf( File , fcntl.LOCK_SH | fcntl.LOCK_NB )
            sleep( 3600 )
            fcntl.lockf( File , fcntl.LOCK_UN  )
            File.close()
        ## sub-process end
        os.kill( childPid , SIGTERM )
        
        childPid = os.fork()
        if childPid: #father
            sleep(1)
            sm = StatusManager()
            recieved_status = sm.getStatus( self.jobDir )
            self.assertEqual( recieved_status , Status( string= "unknown" )  )
            self.assertRaises( IOError , sm.setStatus , self.jobDir, status )
            
        else: #child
            File = open( filename , 'r+' )
            fcntl.lockf( File , fcntl.LOCK_EX | fcntl.LOCK_NB )
            sleep( 3600 )
            fcntl.lockf( File , fcntl.LOCK_UN  )
            File.close()
        ## sub-process end
        os.kill( childPid , SIGTERM )
        

if __name__ == '__main__':
    unittest.main()
