########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.         #
#                                                                                      #
########################################################################################


import os
from time import localtime, strftime , strptime

import logging
u_log = logging.getLogger( __name__ )

from Mobyle.MobyleError import MobyleError , UserValueError


class Admin:
    """
    manage the informations in the .admin file.
    be careful there is no management of concurrent access file
    thus if there is different instance of Admin with the same path
    it could be problem of data integrity
    """
    FIELDS = ( 'DATE'    ,
               'EMAIL'   ,
               'REMOTE'  ,
               'SESSION' ,
               'JOBNAME' ,
               'JOBID'   ,
               'MD5'     ,
               'BATCH'   ,
               'NUMBER'  ,
               'QUEUE'   ,
               'STATUS'  ,
               'MESSAGE'
               )

    
    def __init__( self, path ):
        self.me = {}
        path = os.path.abspath( path )
        
        if os.path.exists( path ):
            if os.path.isfile( path ):
                self.path = path
                self._parse()
            elif os.path.isdir( path ):
                self.path = os.path.join( path , ".admin" )
                if os.path.isfile( self.path ):
                    self._parse()
                else:
                    raise MobyleError , "invalid job path : " + self.path
        else:
            raise MobyleError , "invalid job path : " + path

    
    @staticmethod
    def create( remote , jobName , jobID  , userEmail =None , sessionID = None ):
        """create a minimal admin file"""
        if os.path.exists( ".admin" ):
            raise MobyleError , "an \"admin\" file already exist in %s . can't create a new one"%( os.getcwd() ) 
        else:
            adminFile = open( ".admin"  , "w" )
            
        args = {'DATE' : strftime( "%x %X" , localtime() ) ,
                'REMOTE' : remote ,
                'JOBNAME' : jobName ,
                'JOBID' : jobID ,
                'STATUS' : 'building' ,
                }
        if userEmail:
            args[ 'EMAIL'] = userEmail
        if sessionID:
            args[ 'SESSION' ] = sessionID
        for key in Admin.FIELDS:
            try:
                value =  args[ key ]   
                adminFile.write( "%s : %s\n" %( key , value ) )
            except KeyError :
                continue  
             
        adminFile.close() 
            
    def __str__( self ):
        res = ''
        for key in self.__class__.FIELDS:
            try:
                if key == 'DATE':
                    value = strftime( "%x %X" , self.me['DATE'] )
                elif key == 'STATUS':
                    value = str( self.me[ key ] )
                else:
                    value =  self.me[ key ]
            except KeyError :
                continue
            
            res = res + "%s : %s\n" %( key , value )

        return res

    def _parse ( self ):
        """
        parse the file .admin
        """
        try:
            fh = open( self.path , 'r' )

            for line in fh:
                datas = line[:-1].split( ' : ' )
                key = datas[0]
                value = ' : '.join( datas[1:] )
                if key == 'DATE':
                    value = strptime( value , "%x %X" )
                self.me[ key ] = value
        except Exception , err :
            import sys
            u_log.critical( "an error occured during %s/.admin parsing (call by: %s) : %s " %( self.path ,
                                                                                               os.path.basename( sys.argv[0] ) ,
                                                                                               err ) , 
                                                                                               exc_info = True  )
            raise MobyleError , err 
            
        finally:
            try:
                fh.close()
            except:
                pass    
            if not self.me:
                import sys
                msg = "admin %s object cannot be intanciated: is empty ( call by %s ) " %( self.path , os.path.basename( sys.argv[0] ) )
                u_log.critical( msg)
                #pas d'erreur de leve tant qu'on a pas identifier dans quel cas celle-ci se produit 
             
    def refresh( self ):
        self._parse()

    def commit( self ):
        """
        Write the string representation of this instance on the file .admin
        """
        if not self.me.values():
            import sys
            msg = "cannot commit admin file %s admin instance have no values (call by %s) "%( self.path ,  os.path.basename( sys.argv[0] ) )
            u_log.critical( msg )
            #pas d'erreur de leve tant qu'on a pas identifier dans quel cas celle-ci se produit 
            
        try:
            tmpFile = open( "%s.%d" %( self.path , os.getpid() )  , 'w' )
            tmpFile.write( self.__str__() )
            tmpFile.close()
            os.rename( tmpFile.name , self.path )
        except Exception , err :
            import sys
            u_log.critical( "an error occured during %s/.admin commit (call by: %s) : %s " %( self.path ,
                                                                                               os.path.basename( sys.argv[0] ) ,
                                                                                               err ) , 
                                                                                               exc_info = True  )
            raise MobyleError , err     
        finally:
            try:
                tmpFile.close()
            except:
                pass

    def getDate( self ):
        """
        @return: the date of the job submission
        @rtype: time.struct_time
        """
        try:
            return  self.me[ 'DATE' ] 
        except KeyError :
            return None
 

    def getEmail( self ):
        """
        @return: the email of the user who run the job 
        @rtype: string
        """
        try:
            return self.me[ 'EMAIL' ]
        except KeyError :
            return None


    def getRemote( self ):
        """
        @return: the remote of the job 
        @rtype: string
        """
        try:
            return self.me[ 'REMOTE' ]
        except KeyError :
            return None


    def getSession( self ):
        """
        @return: the Session path of the job 
        @rtype: string
        """
        try:
            return self.me[ 'SESSION' ]
        except KeyError :
            return None


    def getJobName( self ):
        """
        @return: the name of the job ( blast2 , toppred )
        @rtype: string
        """
        try:
            return self.me[ 'JOBNAME' ]
        except KeyError :
            return None


    def getJobID( self ):
        """
        @return: the job identifier.
        @rtype: string
        """
        try:
            return self.me[ 'JOBID' ]
        except KeyError :
            return None


    def getMd5( self ):
        """
        @return: the md5 of the job
        @rtype: string
        """
        try:
            return self.me[ 'MD5' ]
        except KeyError :
            return None


    def setMd5( self , md5 ):
        """
        set the md5 of the job
        @param : the identifier of the job
        @type : string
        """
        self.me[ 'MD5' ] = md5

    def getBatch( self ):
        """
        @return: the batch system name used to run the job  
        @rtype: string
        """
        try:
            return self.me[ 'BATCH' ]
        except KeyError :
            return None


    def setBatch( self , batch):
        """
        set the  batch system used to run of the job
        @param : the name of the batch system 
        @type : string
        """
        self.me[ 'BATCH' ] = batch


    def getNumber( self ):
        """
        @return: the number the job the meaning of this value depend of BATCH value.
         - BATCH = Sys number is the job pid
         - BATCH = SGE number is the result of qstat
        @rtype: string
        """
        try:
            return self.me[ 'NUMBER' ]
        except KeyError :
            return None


    def setNumber( self , number ):
        """
        set the number of the job this number depend of the batch value
        if BATCH = Sys number is the pid
        if BATCH = SGE number is the 
        @param : the number of the job
        @type : string
        """
        self.me[ 'NUMBER' ] = number

    def getQueue( self ):
        """
        @return: return the queue name of the job 
        @rtype: string
        """
        try:
            return self.me[ 'QUEUE' ]
        except KeyError :
            return None


    def setQueue( self, queue ):
        """
        set the queue name of the job
        @param : the queuename of the job
        @type : string
        """
        self.me[ 'QUEUE' ] = queue


    def getStatus( self ):
        """
        @return: the job status."
        @rtype: L{Status} instance
        """
        try:
            status = self.me[ 'STATUS' ] 
        except KeyError :
            return Status( code= -1 ) #unknown
        try:
            msg = self.me[ 'MESSAGE' ]
        except KeyError :
            msg = None
        return  Status( string= status, message =  msg )
        

    def setStatus( self , newStatus):
        """
        set the  status of the job
        @param newStatus: the status of the job.
        @type newStatus: L{Status} instance.
        """
        self._parse()
        oldStatus = self.getStatus()
        if oldStatus.isKnown() and oldStatus.isEnded():
            #a terminal status cannot be changed anymore
            import sys
            u_log.warning( "try to update the %s/%s job status from %s to %s ( call by %s )" %( self.getJobName(), 
                                                                                             self.getJobID() ,
                                                                                             oldStatus ,
                                                                                             newStatus ,
                                                                                             os.path.basename( sys.argv[0] ) 
                                                                                            ))
            return None
        self.me[ 'STATUS' ] = str( newStatus )
        if newStatus.message:
            self.me[ 'MESSAGE' ] = newStatus.message

        
    def getMessage( self ):
        """
        @return: the job status. the values could be \"summitted\" ,\"pending\",\"running\" ,\"finished\",\"error\"
        @rtype: string
        """
        try:
            return self.me[ 'MESSAGE' ]
        except KeyError :
            return None


    def setMessage( self , message ):
        """
        set a  message used when status == error
        @param : a error message
        @type : string
        """
        self.me[ 'MESSAGE' ] = message


class Status:

    _CODE_2_STATUS = { 
                     -1 : "uknown"    , #the status cannot be determined
                      0 : "building"  , # the job directory has been created
                      1 : "submitted" , # the job.run method has been called
                      2 : "pending"   , # the job has been submitted to the batch manager but wait for a slot 
                      3 : "running"   , # the job is running in the batch manager
                      4 : "finished"  , # the job is finished without error from Mobyle
                      5 : "error"     , # the job is encounter a MobyleError 
                      6 : "killed"    , # the job has been removed by the user, or killed by the admin
                      7 : "hold"      , # the job is hold by the batch manager
                      }
    
    def __init__(self , code = None , string = None , message= ''):
        """
        @param code: the code of the status 
        @type code: integer 
        @param string: the code of the status representing by a string
        @type string: string
        @param message: the message associated to the status
        @type message: string
        """
        if code is None and not string :
            raise MobyleError , "status code or string must be specify"
        elif code is not None and string :
            raise MobyleError, "status code or string must be specify, NOT both"
        elif code is not None :
            if code in self._CODE_2_STATUS.keys():
                self.code = code
            else:
                raise MobyleError , "invalid status code :" + str( code )
        elif string:
            if string in  self._CODE_2_STATUS.values():
                iterator = self._CODE_2_STATUS.iteritems()
                for code , status in iterator:
                    if status == string:
                        self.code = code
            else:
                raise MobyleError , "invalid status :" + str( string ) 
        
        if message:
            self.message = message
        else:
            self.message = ''
            
        
    def __eq__(self , other):
        return self.code == other.code and self.message == other.message
    
    def __ne__(self , other ):
        return self.code != other.code or self.message != other.message
    
    def __str__(self):
        return self._CODE_2_STATUS[ self.code ]
    
    def isEnded(self):
        """
         4 : "finished"  , # the job is finished without error from Mobyle
         5 : "error"     , # the job is encounter a MobyleError 
         6 : "killed"    , # the job has been removed by the user, or killed by the admin
         7 : "hold"      , # the job is hold by the batch manager
        """
        return self.code in ( 4 , 5 , 6 )
        # finishing (8) is not properly ended but is't too late to kill it
        
    def isQueryable(self):
        """
        1 : "submitted" , # the job.run method has been called
        2 : "pending"   , # the job has been submitted to the batch manager but wait for a slot 
        3 : "running"   , # the job is running in the batch manager
        7 : "hold"      , # the job is hold by the batch manager
        """
        return self.code in( 1 , 2 , 3 , 7 )
    
    def isKnown(self):
        return self.code != -1

def executionClassLoader( jobName = None , jobID = None ):
    if jobName and jobID :
        raise MobyleError, "jobName either jobID must be specified, not both"
    if not ( jobName or jobID ):
        raise MobyleError , "jobName either jobID must be specified"
    
    if jobName:
        from Mobyle.ConfigManager import Config
        cfg = Config()
        name = cfg.batch( jobName )
    else:
        from Mobyle.JobState import normUri
        from urlparse import urlparse
        
        path = normUri( jobID )
        protocol ,_, _, _,_,_ = urlparse( path )
        
        if protocol == "http":
            raise  NotImplementedError  , "trying to querying a distant server"
    
        if path[-9:] == "index.xml":
            path = path[:-10 ]
        
        adm = Admin( path )
        name = adm.getBatch()
    if not name:
        msg = "cant determine the Execution system for %s " %( ( jobName , jobID )[ bool(jobID) ] )
        u_log.error( msg )
        raise MobyleError, msg
     
    try:
        module = __import__( 'Mobyle.Execution.%s'%name )
    except ImportError , err:
        msg = "The Execution.%s module is missing" %name
        u_log.critical( msg )
        raise MobyleError, msg
    except Exception , err:
        msg = "an error occurred during the Execution.%s import: %s" %( name , err )
        u_log.critical( msg )
        raise MobyleError, msg
    try:
        klass = module.Execution.__dict__[ name ].__dict__[ name ]
        return klass
    except KeyError , err :
        msg = "The Execution class %s does not exist" %name
        u_log.critical( msg )
        raise MobyleError, msg
    except Exception , err:
        msg = "an error occurred during the class %s loading : %s" %( name, err )
        u_log.critical( msg )
        raise MobyleError, msg
    
    
def getStatus( jobID ):
    """
    @param jobID: the url of the job
    @type jobID: string
    @return: the current status of the job
    @rtype: string
    @raise MobyleError: if the job has no number or if the job doesn't exist anymore
    @raise OSError: if the user is not the owner of the process
    """
    import Mobyle.JobState
    import Mobyle.RunnerChild
    import urlparse
    
    path = Mobyle.JobState.normUri( jobID )
    protocol ,_, path, _,_,_ = urlparse.urlparse( path )
    if protocol == "http":
        raise  NotImplementedError  , "trying to querying a distant server"
    
    if path[-9:] == "index.xml":
        path = path[:-10 ]
    adm = Admin( path )
    oldStatus = adm.getStatus()
    #'killed' , 'finished' , 'error' the status cannot change anymore
    #'submitted', 'building' these jobs have not yet sge number

    #  ( 'finished' , 'error' , 'killed' , 'building' ):
    if not oldStatus.isQueryable():
        return oldStatus
    else:
        batch = adm.getBatch()
        jobNum = adm.getNumber()
        
        if batch is None :
            return oldStatus
        if jobNum is None:
            return oldStatus
        try:
            execKlass = executionClassLoader( jobID = jobID )
            newStatus = execKlass.getStatus( jobNum )
        except MobyleError ,err : 
            u_log.error( str( err ) , exc_info = True )
            raise err
        if not newStatus.isKnown():
                return oldStatus
        if newStatus != oldStatus :
            adm.setStatus( newStatus )
            adm.commit()
            jobState = Mobyle.JobState.JobState( jobID )
            jobState.setStatus( newStatus ) 
            jobState.commit()
            
        return newStatus 


def isExecuting( jobID ):
    """
    @param jobID: the url of the job
    @type jobID: string
    @return True if the job is currently executing ( submitted , running , pending , hold ).
    False otherwise ( building, finished , error , killed )
    @rtype: boolean
    @raise MobyleError: if the job has no number 
    @raise OSError: if the user is not the owner of the process
    """
    import Mobyle.JobState
    import Mobyle.RunnerChild
    import urlparse
    
    path = Mobyle.JobState.normUri( jobID )
    protocol ,host, path, a,b,c = urlparse.urlparse( path )
    if protocol == "http":
        raise  NotImplementedError  , "trying to querying a distant server"
    
    if path[-9:] == "index.xml":
        path = path[:-10 ]
    adm = Admin( path )
    batch = adm.getBatch()
    jobNum = adm.getNumber()
    
    if batch is None or jobNum is None:
        status = adm.getStatus()
        if not status.isQueryable():
            return False
        else:
            raise MobyleError( "inconsistency in .admin file %s" % path )
    try:
        execKlass = executionClassLoader( jobID = jobID )
        newStatus = execKlass.getStatus( jobNum )
    except MobyleError ,err : 
        u_log.error( str( err ) , exc_info = True )
        raise err
    return newStatus.isQueryable()



def killJob( jobID ):
    """
    @param jobID: the url of the job or a sequence of jobID
    @type jobID: string or sequence of jobID
    @return: 
    @rtype: 
    @raise MobyleError: if the job has no number or if the job doesn't exist anymore
    @todo: tester la partie sge
    """
    
    import Mobyle.RunnerChild
    import types
    from Mobyle.MobyleError import MobyleError
    from Mobyle.JobState import JobState
    
    jobIDType = type( jobID )
    
    if jobIDType == types.StringType :
        jobIDs = [ jobID ]
    elif jobIDType == types.TupleType or jobIDType == types.ListType :
        jobIDs = jobID
    else:
        raise MobyleError , "jobID must be a string ar a Sequence of strings"
    errors = []
    for jobID in jobIDs :
        try:
            path = Mobyle.JobState.normUri( jobID )
        except MobyleError , err :
            errors.append( ( jobID , str( err ) ) )
            continue
        if path[:4] == 'http' :
            #the jobID is not on this Mobyle server
            errors.append( ( jobID , "can't kill distant job" ) )
            continue
        try:
            adm = Admin( path )
            jobState = JobState( uri=jobID )
        except MobyleError , err :
            errors.append( ( jobID , "invalid jobID" ) )
            continue
        jobNum = adm.getNumber()
        
        if jobNum is None:
            # an error occured on this jobID but I continue to kill the other jobIDs
            errors.append( ( jobID , 'no jobNum for this job' ) )
            continue
        try:
            execKlass = executionClassLoader( jobID = jobID )
            execKlass.kill( jobNum )            
            killed = Status( code= 6 , message = "Your job has been cancelled" )
            adm.setStatus( killed ) 
            adm.commit()
            jobState.setStatus( killed )
            jobState.commit()
        except MobyleError ,err :
            errors.append( ( jobID , str( err ) ) )
    if errors:
        msg = ''
        for jobID , msgErr in errors :
            msg = "%s killJob( %s ) - %s\n" %( msg , jobID , msgErr )
            
        raise MobyleError , msg



def safeFileName( fileName ):
    import string , re
    
    if fileName in ( 'index.xml' , '.admin' , '.command' ,'.forChild.dump' ,'.session.xml'):
        raise UserValueError( msg = "value \"" + str( fileName ) + "\" is not allowed" )
    
    for car in fileName :
        if car not in string.printable : #we don't allow  non ascii char
            fileName = fileName.replace( car , '_')
    
    #SECURITY: substitution of shell special characters
    fileName = re.sub( "[ #\"\'<>&\*;$`\|()\[\]\{\}\?\s ]" , '_' , fileName )
                  
    #SECURITY: transform an absolute path in relative path
    fileName = re.sub( "^.*[\\\:/]", "" , fileName )
    
    return fileName



def makeService( serviceUrl ):
    import Mobyle.Parser
    
    try:
        parser = Mobyle.Parser.ServiceParser()
        service = parser.parse( serviceUrl )
        return service
    except IOError , err:
        raise MobyleError , str( err )

      
def sizeFormat(bytes, precision=2):
    """Returns a humanized string for a given amount of bytes"""
    import math
    bytes = int(bytes)
    if bytes is 0:
        return '0 bytes'
    log = math.floor( math.log( bytes , 1024 ) )
    return "%.*f%s" % ( precision , bytes / math.pow(1024, log), [ 'bytes', 'KiB', 'MiB' , 'GiB' ][ int(log) ] )
