
########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.         #
#                                                                                      #
########################################################################################


"""
Classes executing the command and managing the results  
"""
import os 
import logging
_log = logging.getLogger(__name__)

import Mobyle.JobState
import Mobyle.ConfigManager
import Mobyle.Utils
from Mobyle.Execution.Batch import _Batch
import drmaa

from Mobyle.MobyleError import *


_cfg = Mobyle.ConfigManager.Config()
__extra_epydoc_fields__ = [('call', 'Called by','Called by')]




class DRMAA(_Batch):
    """
    Run the commandline by a system call
    """
    @staticmethod
    def _drmaaStatus2mobyleStatus( drmaaStatus ):
        if drmaaStatus == 'running':
            return Mobyle.Utils.Status( 3 ) #running
        elif drmaaStatus == 'undetermined':
            return Mobyle.Utils.Status( -1 ) #unknown
        elif drmaaStatus == 'queued_active':
            return Mobyle.Utils.Status( 2 ) #pending
        elif drmaaStatus == 'done':
            return Mobyle.Utils.Status( 4 ) #finished
        elif drmaaStatus == 'failed':
            return Mobyle.Utils.Status( 5 ) # error
        elif drmaaStatus == 'system_on_hold':
            return Mobyle.Utils.Status( 7 ) # hold
        elif drmaaStatus == 'user_on_hold':
            return Mobyle.Utils.Status( 7 ) # hold
        elif drmaaStatus == 'user_system_on_hold':
            return Mobyle.Utils.Status( 7 ) # hold
        elif drmaaStatus == 'system_suspended':
            return Mobyle.Utils.Status( 7 ) # hold
        elif drmaaStatus == 'user_suspended':
            return Mobyle.Utils.Status( 7 ) # hold
        elif drmaaStatus == 'user_system_suspended':
            return Mobyle.Utils.Status( 7 ) # hold
        else:
            return Mobyle.Utils.Status( -1 )

    def run(self):
        """
        Run the commandLine 
        redirect the standard error and output on service.name.out and service.name.err, then restore the sys.stderr and sys.stdout
        @return: the L{Mobyle.Utils.Status} of this job and a message
        @rtype: Status
        @call: in _Batch.__init__
        """
        if (os.getcwd() != os.path.abspath(self.dirPath) ):
            msg = "the process is not in the working directory"

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg )

            raise MobyleError , msg

        else:
            jobKey = self.getKey()
            fout = open( self.serviceName + ".out" , 'w' )
            ferr = open( self.serviceName + ".err" , 'w' )
            try:
                contactString = drmaa.Session.contact
                if contactString.find( ',' ) != -1 :
                    _log.critical( "several drm are found on your system ( %s ), Mobyle do not know manage several drm" %contactString )
                    raise MobyleError  , "Internal Server Error"
                drmaaSession = drmaa.Session( contactString = contactString )
                jt = drmaaSession.createJobTemplate()
                jt.workingDirectory = self.dirPath
                jt.jobName = jobKey
                jt.outputPath = os.path.join( self.dirPath , fout.name )
                jt.errorPath = os.path.join( self.dirPath , ferr.name )
                jt.joinFiles = False
                jt.jobEnvironment = self.xmlEnv
                jt.remoteCommand = "sh"
                jt.args = [ os.path.join( self.dirPath , ".command" ) ]
                # workaround the bug in librmaa from opendsp
                #if not os.environ.has_key('HOME'):
                #    os.environ[ 'HOME' ] = '/dev/null' 
                jt.nativeSpecification = "-q %s" % self.queue
                drmJobid = drmaaSession.runJob( jt )
            except drmaa.errors , err :
                try:
                    drmaaSession.deleteJobTemplate( jt )
                    drmaaSession.exit()
                except Exception , err :
                    _log.error( "cannot exit from drmaa properly : " + str( err ) )
               
                msg= "System execution failed: " +str( err ) 
                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                _log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   jobKey ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg 
            except  :
                _log.debug( "pb nontrapee" )
                raise MobyleError , "pb nontrapee"
                    
            adm = Mobyle.Utils.Admin( self.dirPath )
            adm.setBatch( 'DRMAA' )  
            adm.setNumber( drmJobid ) 
            adm.commit()

            linkName = ( "%s/%s.%s" %( _cfg.admindir() ,
                                       self.serviceName ,
                                       jobKey
                                    )
                         )
            
            try:
                os.symlink(
                    os.path.join( self.dirPath , '.admin') ,
                    linkName
                    )
            except OSError , err:
                try:
                    drmaaSession.deleteJobTemplate( jt )
                    drmaaSession.exit()
                except Exception , err :
                    _log.error( "cannot exit from drmaa properly : " + str( err ) )
                    
                msg = "can't create symbolic link %s in ADMINDIR: %s" %( linkName , err )

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                _log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   self.getKey() ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg
            
            #JobInfo =( jobId , hasExited , hasSignal , terminatedSignal, hasCoreDump, wasAborted, exitStatus, resourceUsage)
            #            0          1          2              3               4            5           6           7
            jobInfos = drmaaSession.wait( drmJobid , drmaa.Session.TIMEOUT_WAIT_FOREVER )
            
            try:
                os.unlink( linkName )
            except OSError , err:
                try:
                    drmaaSession.deleteJobTemplate( jt )
                    drmaaSession.exit()
                except Exception , err :
                    _log.error( "cannot exit from drmaa properly : " + str( err ) )
                    
                msg = "cannot remove symbolic link %s in ADMINDIR: %s" %( linkName , err )

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                _log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   self.getKey() ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg
            fout.close()
            ferr.close()
            
            adm = Mobyle.Utils.Admin( self.dirPath )
            oldStatus = adm.getStatus()
            
            if oldStatus.isEnded():
                status= oldStatus
            else:
                if jobInfos[ 5 ] :#wasAborted
                    status = Mobyle.Utils.Status( code = 6 , message = "Your job has been cancelled" ) #killed
                else:
                    if jobInfos[ 1 ]:#hasExited
                        if jobInfos[ 6 ] == 0:#exitStatus
                            status = Mobyle.Utils.Status( code = 4 ) #finished
                        else:
                            status = Mobyle.Utils.Status( code = 4 , message = "Your job finished with an unusual status code ( %s ), check your results carefully." % jobInfos[6] )
                    else:
                        status = Mobyle.Utils.Status( code = 6 , message = "Your job execution failed ( %s )" %jobInfos[6] ) 

            try:
                drmaaSession.deleteJobTemplate( jt )
                drmaaSession.exit()
            except :
                _log.error( "cannot exit from drmaa properly" )
            
            return status
        

    @staticmethod
    def getStatus( key ):
        """
        @param key: the value associate to the key "NUMBER" in Admin object (and .admin file )
        @type key: string
        @return: the status of the job corresponding to the key 
        @rtype: Mobyle.Utils.Status instance
        @call: by L{Utils.getStatus}
        """
        contactString = drmaa.Session.contact
        if contactString.find( ',' ) != -1 :
            _log.critical( "several drm are found on your system ( %s ), Mobyle do not know manage several drm" %contactString )
            raise MobyleError , "Internal Server Error"
        try:
            s = drmaa.Session( contactString = contactString )
        except Exception , err:
            _log.error( "getStatus(%s) cannot open drmma session : %s " %( key , err ) )
            return Mobyle.Utils.Status( -1 ) #unknown 
        try:
            drmaaStatus = s.jobStatus( key )
        except :
            s.exit()
            return Mobyle.Utils.Status( -1 ) #unknown 
        s.exit()
        return DRMAA._drmaaStatus2mobyleStatus( drmaaStatus ) 


    @staticmethod
    def kill( key ):
        """
        kill a job
        @param key : the value associate to the key "NUMBER" in Admin object (and .admin file )
        @type key: string
        @raise MobyleError: if can't kill the job
        @call: by L{Utils.Mkill}
        """
        contactString = drmaa.Session.contact
        if contactString.find( ',' ) != -1 :
            _log.critical( "several drm are found on your system ( %s ), Mobyle do not know manage several drm" %contactString )
            raise MobyleError , "Internal Server Error"
        try:
            s = drmaa.Session( contactString = contactString )
        except Exception , err:
            _log.error( "kill( %s ) cannot open drmma session : %s " %( key , err ) )
            return
        try:
            s.control( key , drmaa.JobControlAction.TERMINATE )
        except :
            _log.error( "error when trying to kill job %s : %s" %( key , err ) )
        s.exit()
        return
        
