########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.        #
#                                                                                      #
########################################################################################

"""
Content the basics Mobyle Parameter types 
"""
 

import os 
import re
import types
import logging
import shutil

import Mobyle.Utils
from Mobyle.Classes.DataType import DataType

from Mobyle.MobyleError import MobyleError , UserValueError , UnDefAttrError

c_log = logging.getLogger(__name__)
b_log = logging.getLogger('Mobyle.builder')



def safeMask(  mask ):
    import string
    for car in mask :
        if car not in string.printable :
            mask = mask.replace( car , '_')
    #don't use re.UNICODE because safeFileName don't permit to use unicode char.
    #we must work whit the same char set    
    mask = re.sub( "(;|`|$\(\s)+" , '_' , mask )
    mask = re.sub( "^.*[\\\:/]", "" ,  mask  )
    return mask





class DataTypeTemplate( DataType ):

    @staticmethod
    def convert(value):
        """
        cast the value in rigth type
        @return: the casted value 
        @raise UseValueError: if the cast failed
        """
        return "DataTypeTemplate convert: " + str( value )
    
    @staticmethod    
    def validate( param ):
        """
        @return: True if the value is valid, False otherwise
        """
        return "DataTypeTemplate validate "
        
        
        
class BooleanDataType( DataType ):
    """
    """
    @staticmethod
    def convert( value , param ):
        """
        cast the value in a Boolean.
        The values: "off", "false" ,"0" or '' (case insensitive) are false,
        all others values are True.  
        @param value: the value provide by the User for this parameter
        @type value: String
        @return: the value converted in Boolean
        """
        #type controls
        if value is None:
            # in html form boolean appear as checkbox
            # if the chexkbox is not selected
            # the parameter is not send in the request
            return False
        if isinstance( value , types.BooleanType ):
            return value
        
        elif isinstance( value , types.StringTypes ):
            value = value.lower()
            if value == "off" or value == "false" or value == "0" or value == '' or value == "''" or value == '""':
                return False
            elif value == "on" or value == "true" or value == "1" :
                return True
        else:
            msg = "Invalid value: " + str( value ) + " is not a boolean."
            raise UserValueError( parameter = param , msg = msg)



    @staticmethod
    def validate( param ):
        """
        do type controls. if the value is True or False or None return True,
        otherwise return False.        
        """
        value = param.getValue()
        
        if value == True or value == False:
            return True
    
        else:
            return False


        
class IntegerDataType( DataType ):


    @staticmethod
    def convert( value , param ):
        """
        Try to cast the value in integer. the allowed values are digit
        and strings.
        
        @param value: the value provide by the User for this parameter
        @type value: 
        @return: the value converted in Integer.
        @rtype: int
        @raise UserValueError: if the cast fail a UserValueError is raised.
        Unlike python, this method convert "8.0" in 8 and
        raised a UserValueError if you try to convert 8.2 .
        """
        if value is None:
            return None
        
        #type controls
            # int("8.0") failed and a  ValueError is raised
            # int(8.1) return 8
        
        try:
            f= float( value )
            i = int( f  )
            if ( (f - i) == 0):
                return i 
            else:
                msg = "\"%s\" : this parameter must be an integer" %value
                raise  UserValueError( parameter = param , msg = msg)
        except OverflowError:
            raise UserValueError( parameter = param , msg = "this value is too big" )
        except ( ValueError , TypeError ):
            msg = "\"%s\" : this parameter must be an integer" %value
            raise UserValueError( parameter = param , msg = msg)


    @staticmethod
    def validate( param ):
        """
        @return True if the value is an integer, False othewise.
        """   
        value = param.getValue()

        if value is None:
            return True                       
        try:
            int( value )
        except ( TypeError , ValueError ):
            return False



class FloatDataType( DataType ):


    @staticmethod
    def convert( value , param ):
        """
        Try to cast the value in float.
        
        @param value: the value provide by the User for this parameter
        @type value: 
        @return: the value converted in Float
        @rtype: float
        """
        if value is None:
            return None
                
        try:
            return float( value )
        except ( ValueError, TypeError ):
            msg = str( value ) + " this parameter must be a Float"
            raise UserValueError( parameter = param , msg = msg )
        

    @staticmethod
    def validate( param ):
        value = param.getValue()
        
        if value is None:
            return True     
        try:
            float( value )
        except ( ValueError , TypeError ):
            return False




class StringDataType( DataType ):


    @staticmethod
    def convert( value , param ):
        """
        Try to cast the value in String. 
        
        @param value: the value provide by the User for this parameter
        @type value: 
        @return: the value converted in string.
        @rtype: String
        @raise UserValueError: if the cast fail a UserValueError is raised.
        """
        if value is None:
            return None
            #msg = str( value )+": the parameter \"%s\" should be a String" %( param.getName() )
            #raise UserValueError( parameter = param , msg = msg )

        #type controls
        try:
            value = str( value )
            
            #the string with space are alowed
            #but if the string will appear as shell instruction it must be quoted
            if value.find(' ') != -1 :
                if not param.hasParamfile():
                    value = "'%s'" %value
            
            return value
            
        except ValueError :
            msg = " the parameter \"%s\" should be a String" %( param.getName() )
            raise UserValueError( parameter = param , msg = msg )
        
        return value                        
        
        
    @staticmethod
    def validate( param  ):

        value = param.getValue()
        if value is None:
            return True
        #try:
        #    str( value )
        #except ( ValueError, TypeError ):
        #    return False
        
        #allowed characters:
        #the words , space, ' , - , + , and dot if is not followed by another dot 
        #and eventually surrounded by commas
        reg = "(\w|\ |-|\+|,|\.(?!\.))+"
        if re.search( "^(%s|'%s')$" % (reg, reg) ,  value ) :         
            return True
        else:
            msg = "this value : \"" + str( value ) + "\" , is not allowed"
            raise UserValueError( parameter = param , msg = msg )
        
       


class ChoiceDataType( StringDataType ):
    #the values of ChoiceDataType are literals thus they are String

    @staticmethod        
    def convert( value , param ):
        """
        The values of ChoiceDataType are literals thus this method try to cast
        the value in string.
                
        @param value: the value provide by the User for this parameter
        @type value: 
        @return: the value converted in String
        @rtype: string
        """
        if value is None:
            return None
                
        try:
            value  = str( value )
            if value == param.getListUndefValue():
                return None
            else:
                return value
        except ValueError:
            
            msg = str( value )+" this parameter is a Choice its value should be a String" 
            raise UserValueError( parameter = param , msg = msg )


    @staticmethod
    def validate( param ):
        """
        @return: True if the value is valid. that 's mean the value
        should be a string among the list defined in the xml.
        otherwise a MobyleError is raised.
        @rtype: boolean
        @param value: a value from a Choice parameter
        @type value: Choice value
        @raise UserValueError: if the value is not a string or is not among the
        list defined in the xml vlist.
        """
        value = param.getValue()
        if param.hasVlist() :
            authorizedValues = param.getVlistValues()
        elif param.hasFlist() :
            authorizedValues = param.getFlistValues()
        else:
            msg = "%s a choice must have a flist or vlist" %( param.getName() )
            c_log.error( msg )
            raise MobyleError , msg     
        
        if value is None or value in authorizedValues:
            return True
        else:
            paramName = param.getName()
            logMsg = "Unauthorized value for the parameter : %s : authorized values = %s : provided value = %s" %( paramName , 
                                                                                                                  authorizedValues ,
                                                                                                                  value 
                                                                                                                  )
            c_log.error( logMsg )
            
            msg = "Unauthorized value for the parameter : %s" %( param.getName() )
            raise UserValueError( parameter = param , msg = msg )
                
               
           




class MultipleChoiceDataType( StringDataType ):

        

    
    @staticmethod
    def convert( value , param ):
        """
        The MutipleChoiceDataType value are literals thus this method try to cast
        the value in a list of string.
                
        @param value: the values provide by the User for this parameter
        @type value: list
        @return: a string based on each selected value and join by the separator.
        @rtype: String .
        @raise UserValueError: if the value can't be convert in a string.
        """       
        if value is None:
            return None
        valueType = type( value )
       
        sep = param.getSeparator()
        if sep is None:
            sep = ''
       
        if isinstance( value , types.StringTypes ) :
            if value.find( sep ) == -1 :
                raise MobyleError , "MultipleChoiceDataType can't find separator in value"
            if sep == '':
                values = [ char for char in str( value ) ]
            else:
                values = str( value ).split( sep )[ 0 : -1 ]
                
        elif valueType == types.ListType or valueType == types.TupleType :
            try:
                values = [ str( elem ) for elem in value ]
            except ValueError:
                msg  = "this parameter is a MultipleChoice its all values must be Strings" %value 
                raise UserValueError( parameter = param , msg = msg )
        else:
            c_log.debug( "value=@%s@ sep=@%s@ value type=%s"%( value, sep , valueType ) )
            raise MobyleError , "MultipleChoiceDataType accept only strings or list of strings as value"           
        
        for oneValue in values:
                if sep == '' and len( oneValue ) > 1:
                    raise MobyleError , "in MultipleChoiceDataType if seperator is the empty string, each value length must be 1 (povided :%s)"%oneValue

        return sep.join( values )
    

    @staticmethod
    def validate( param ):
        
        userValues = param.getValue() #it's a string
        sep = param.getSeparator()
        if sep == '':
            userValues = [ i for i in userValues ]
        else:
            userValues = userValues.split( sep )

        authorizedValues =  param.getVlistValues()
        
        for value in userValues:
            if value not in authorizedValues :
                msg = "the value %s is not allowed (allowed values: %s " % (
                str( value ) ,
                str( param.getVlistValues() )
                )
                raise UserValueError( parameter = param , msg = msg )

        return True



    

class AbstractTextDataType( DataType ):

    
    @staticmethod
    def isFile( ):
        return True  

    
    @staticmethod    
    def head( data ):
        return data[ 0 : 50 ]
    
    @staticmethod
    def cleanData( data ):
        # trying to guess the encoding, before to convert the data to ascii
        try:
            # trying ascii
            data = unicode(data.decode('ascii','strict'))
        except:
            try:
                # utf8 codec with BOM support
                data = unicode(data,'utf_8_sig')
            except:
                try:
                    # utf16 (default Windows Unicode encoding)
                    data = unicode(data,'utf_16')
                except:
                    # latin1
                    data = unicode(data,'latin1')
                    # converting the unicode data to ascii
                    data = data.encode('ascii','replace')
      
        return  re.sub( "\r\n|\r|\n" , '\n' , data )


    @staticmethod    
    def _toFile(  param , data  , dest , destFileName , src , srcFileName ):
        """
        Write file (of user data) in dest directory .
        @param fileName:
        @type fileName: string
        @param data: the content of the file
        @type data: string
        @param dest:
        @type dest:
        @param src:
        @type src:
        @return: the name ( asbolute path )of the created file
        @rtype: string
        @raise: L{UserValueError} when filename is not allowed (for security reason)
        @raise: L{MobyleError} if an error occured during the file creation
        """
            
        try:
            destSafeFileName = Mobyle.Utils.safeFileName( destFileName )            
        except UserValueError, err:
            raise UserValueError( parameter = param , msg = "this value : %s is not allowed for a file name, please change it" % destFileName )

      
        abs_DestFileName= os.path.join( dest.getDir() , destSafeFileName )

        # if the user upload 2 files with the same basename Mobyle.Utils.safeFileName
        # return the same safeFileName
        # I add an extension to avoid _toFile to erase the existing file.

        ext = 1
        completeName = abs_DestFileName.split( '.' )
        base = completeName[0]
        suffixe = '.'.join( completeName[1:] )

        while os.path.exists( abs_DestFileName ):
            abs_DestFileName = base + '.' + str( ext ) + '.' + suffixe
            ext = ext + 1
        
        if src:
            
            if src.isLocal():

                try:
                    srcSafeFileName = Mobyle.Utils.safeFileName( srcFileName )
                except UserValueError, err:
                    raise UserValueError( parameter = param , msg = "this value : %s is not allowed for a file name, please change it" % srcFileName )
    
                #the realpath is because if the abs_SrcFileName is a soft link ( some results are ) the
                # hardlink point to softlink and it causse ane error : no such file 
                abs_SrcFileName = os.path.realpath( os.path.join( src.getDir() , srcSafeFileName ) )
                
                try:
                    os.link(  abs_SrcFileName , abs_DestFileName )
                except OSError :
                    #if the src and dest are not on the same device
                    #an OSError: [Errno 18] Invalid cross-device link , is raised
                    try:
                        shutil.copy( abs_SrcFileName , abs_DestFileName )
                    except IOError ,err:
                        # I don't know  - neither the service ( if it exists )
                        #               - nor the job or session ID 
                        # I keep the Job or the Session to log this error 
    
                        msg = "can't copy data from %s to %s : %s" %( abs_SrcFileName ,
                                                                      abs_DestFileName ,
                                                                      err )
                                                        
                        raise MobyleError , "can't copy data : "+ str(err)
                    
            else: #src is a job , jobState , MobyleJOb instance ( session is Local )
                
                data = src.getOutputFile( srcFileName )
                try:
                    f = open( abs_DestFileName , 'w' )
                    #faut il nettoyer la donnee ? normalement c'est deja fait , non ?
                    f.write( data )
                    f.close()
                except IOError ,err:
                    pass
                
        else:
        
            clean_content = AbstractTextDataType.cleanData( data )

            try:
                fh = open( abs_DestFileName , "w" )
                fh.write( clean_content )
                fh.close()
            except IOError , err:
                # I don't know  - neither the service ( if it exists )
                #               - nor the job or session ID 
                # I keep the Job or the Session to log this error 

                msg = "error occur when creating file : " + abs_DestFileName + str( err )
                raise MobyleError , msg


        size = os.path.getsize( abs_DestFileName )
        return os.path.basename( abs_DestFileName ) , size
    
          
    @staticmethod    
    def convert( value , param ):
        """
        @param value: is a tuple ( destFileName , data , dest , src , srcFileName) 
        @type value: tuple ( string filename , string content , L{Job} or L{Session} instnace )
          - filename is mandatory
          - data is a string reprensenting the data. it could be None if src is specify
          - dest is L{Job} or L{Session} instance is where the data will be store
          - src is L{Job} or L{Session} instance is where the data come from (it must be specify only if data is None).
        @type value: ( string filename, string data , L{Job} or L{Session} instance dest ,L{Job} or L{Session} instance, src)
        @return: the fileName ( basename ) of the text file
        @rtype: string
        """
        if param.isout():
            raise UserValueError( parameter = param , msg = "out parameter can't be modify by users" )
            #if value is None :
            #    raise UserValueError( parameter = param , msg = msg )
            #else:
            #    safeFileName = Mobyle.Utils.safeFileName( value )
            #    return safeFileName
        
        if value is None:
            return None        
        if len( value ) == 5 :
            data , dest , destFileName ,  src , srcFileName = value
        else:
            raise MobyleError ,"value must be a tuple of 5 elements: ( destFileName , data , Job/MobyleJob/JobState or Session instance , Job/MobyleJob/JobState or Session instance , srcFileName )"

        if destFileName is None:
            return None
        
        if dest is None:
            raise MobyleError, "the destination is mandatory"

        if  data and src :
            raise MobyleError, "you cannot specify data and src in the same times"

        if not data and ( not dest or not src ) :
            raise MobyleError , "if data is not specify, dest and src must be defined"

        if src and not srcFileName :
            raise MobyleError , "if src is specify , srcFileName must be also specify"
        
        if param.isInfile():
            #_toFile return a basename
            fileName  , size = AbstractTextDataType._toFile( param , data  , dest , destFileName , src , srcFileName )
        else:
            fileName = Mobyle.Utils.safeFileName( destFileName )

        try:
            jobstate = dest.jobState
        except AttributeError :
            jobstate = None
   
        if jobstate is not None:
            lang = param.cfg.lang()
            if param.promptHas_lang( lang ):
                prompt = param.getPrompt( lang = lang )
            else:
                prompt = None
                lang   = None
                
            jobstate.setInputDataFile( param.getName() ,
                                       ( prompt , lang ) ,
                                       param.getType() , # et pourquoi pas self
                                       ( fileName , size ,None )
                                       )
            jobstate.commit()
        return fileName


    @staticmethod
    def validate( param ):
        """
        @todo: faut t'il le passer dans une regexp? apriori Non il est deja passer par toFile et donc par safeFilename
        """
        value = param.getValue()
        if param.isout():
            if value is not None : #un parametre isout ne doit pas etre modifier par l'utilisateur 
                return False
            else:
                
                
                #####################################################
                #                                                   #
                #  check if the Parameter have a secure filenames   #
                #                                                   #
                #####################################################
                
                try:
                    debug = param.getDebug()
                    if debug > 1:
                        b_log.debug( "check if the Parameter have a secure filename" )
    
                    #getFilenames return list of strings representing a unix file mask which is the result of a code evaluation
                    #getFilenames return None if there is no mask for a parameter.
                    filenames = param.getFilenames( ) 
    
                    for filename in filenames :
                        if filename is None:
                            continue
                        mask = safeMask( filename )
    
                        if debug > 1:
                            b_log.debug( "filename= %s    safeMask = %s"%(filename, mask))
                        if  not mask or mask != filename :
                            # comment logger ce genre d'erreur 
                            # la faire remonter a MobyleJob validate ???
                            msg = "The Parameter:%s, have an unsecure filenames value: %s " %( param.getName() ,
                                                                                                filename )
                            c_log.error( admMsg = "MobyleJob._validateParameters : " + msg ,
                                            logMsg = None ,
                                            userMsg = "Mobyle Internal Server Error"
                                            )
                            if debug == 0:
                                c_log.critical( "%s : %s" %( param.getService().getName(), msg  ) )
                            raise MobyleError( "Mobyle Internal Server Error" ) 
                        else:
                            if debug > 1:
                                b_log.debug( "filename = %s ...........OK" % filename )
    
                                               
                except UnDefAttrError :
                    b_log.debug("no filenames")
                
        else:
            if value is None:
                return True #an infile Text ne peut pas avoir de vdef mais peut il etre a None" => oui s'il n'est pas obligatoire
            else:
                return os.path.exists( param.getValue() )
        


class TextDataType( AbstractTextDataType ):
    # this trick is to avoid that SequenceDataType is a subclass of TextDataType
    pass

class ReportDataType( AbstractTextDataType ):
    pass

class BinaryDataType( DataType ):

    @staticmethod   
    def isFile():
        return True
    
    @staticmethod
    def head( data ):
        return 'Binary data'
            
    @staticmethod          
    def cleanData( data ):
        """
        prepare data prior to write it on a disk
        @param data: 
        @type data:a buffer
        """
        return data


    @staticmethod   
    def _toFile( param , data , dest , destFileName , src , srcFileName ):
        """
        Write file (of user data) in the working directory .
        @param fileName:
        @type fileName: string
        @param content: the content of the file
        @type content: string
        @return: the name ( absolute path ) of the created file created ( could be different than the arg fileName )
        @rtype: string
        @call: L{MobyleJob._fillEvaluator}
        """
        

        try:
            destSafeFileName = Mobyle.Utils.safeFileName( destFileName )
        except UserValueError, err:
            raise UserValueError( parameter = param , msg = "this value : %s is not allowed for a file name, please change it" % destFileName )

        abs_DestFileName = os.path.join( dest.getDir() , destSafeFileName )

        # if the user upload 2 files with the same basename Mobyle.Utils.safeFileName
        # return the same safeFileName
        # I add an extension to avoid _toFile to erase the existing file.

        ext = 1
        completeName = abs_DestFileName.split( '.' )
        base = completeName[0]
        suffixe = '.'.join( completeName[1:] )

        while os.path.exists( abs_DestFileName ):
            abs_DestFileName = base + '.' + str( ext ) + '.' + suffixe
            ext = ext + 1

        if src:
            
            if src.isLocal():
                
                try:
                    srcSafeFileName = Mobyle.Utils.safeFileName( srcFileName )
                except UserValueError, err:
                    raise UserValueError( parameter = param , msg = "this value : %s is not allowed for a file name, please change it" % srcFileName  )
                
                #the realpath is because if the abs_SrcFileName is a soft link ( some results are ) the
                # hardlink point to softlink and it causse ane error : no such file  
                abs_SrcFileName = os.path.realpath( os.path.join( src.getDir() , srcSafeFileName ) )
                try:
                    os.link(  abs_SrcFileName , abs_DestFileName )
                except OSError :
                    #if the src and dest are not on the same device
                    #an OSError: [Errno 18] Invalid cross-device link , is raised
                    try:
                        shutil.copy( abs_SrcFileName , abs_DestFileName )
                    except IOError ,err:
                        # je ne connais - ni le service (s'il existe)
                        #               - ni le l' ID du job ou de la session
                        # donc je laisse le soin au Job ou la session a logger l'erreur
    
                        msg = "can't copy data from %s to %s : %s" %( abs_SrcFileName ,
                                                                      abs_DestFileName ,
                                                                      err )
                        
                        raise MobyleError , "can't copy data : "+ str(err)
     
            else: #src is a job , jobState , MobyleJOb instance ( session is Local )
                
                data = src.getOutputFile( srcFileName )
                try:
                    f = open( abs_DestFileName , 'wb' )
                    f.write( data )
                    f.close()
                except IOError ,err:
                    pass
  
        else:
            try:
                fh = open( abs_DestFileName , "wb" )
                fh.write( data )
                fh.close()
            except IOError , err:
                # je ne connais - ni le service (s'il existe)
                #               - ni le l' ID du job ou de la session
                # donc je laisse le soin au Job ou la session a logger l'erreur

                msg = "error occur when creating file %s: %s" %( os.path.basename( abs_DestFileName ) ,  err )

                raise MobyleError , msg
            
            
        size  = os.path.getsize( abs_DestFileName )
        return os.path.basename( abs_DestFileName ) , size
    
    
    @staticmethod
    def convert( value , param ):
        """
        Do the generals control and cast the value in the right type.
        if a control or the casting fail a MobyleError is raised

        @param value: the value provide by the User for this parameter
        @type value: String
        @return: the fileName ( basename ) of the binary file
        """
        if param.isout():
            raise UserValueError( parameter = param , msg = "out parameter can't be modify by users" )       
        #if param.isout():
        #    if value is None :
        #        raise UserValueError( parameter = param , msg = msg )
        #    else:
        #        safeFileName = Mobyle.Utils.safeFileName( value )
        #        return safeFileName

        if value is None:
            return None
 
        elif len( value ) == 5 :
            data , dest , destFileName , src , srcFileName = value
        else:
            raise MobyleError ,"value must be a tuple of 5 elements: ( data , dest , destFileName , src , srcFileName )"

        if destFileName is None:
            return None
        
        if dest is None:
            raise MobyleError, "the destination is mandatory"

        if  data and src :
            raise MobyleError, "you cannot specify data and src in the same times"

        if not data and ( not dest or not src ) :
            raise MobyleError , "if data is not specify, dest and src must be defined"

        if src and not srcFileName :
            raise MobyleError , "if src is specify , srcFileName must be also specify"

        if param.isInfile():
            #param._toFile return an absolute path
            fileName , size = BinaryDataType._toFile( param , data , dest , destFileName , src , srcFileName )

        else:
            #safeFileName return a basename
            fileName = Mobyle.Utils.safeFileName( destFileName )

        try:
            jobstate = dest.jobState
        except AttributeError :
            jobstate = None

        if jobstate is not None:
            lang = param.cfg.lang()
            if param.promptHas_lang( lang ):
                prompt = param.getPrompt( lang = lang )
            else:
                prompt = None
                lang   = None
            
            jobstate.setInputDataFile( param.getName() ,
                                       ( prompt , lang ) ,
                                       param.getType() ,
                                       ( fileName , str( size ) , None )
                                       )
            jobstate.commit()

        return fileName



    @staticmethod
    def validate( param ):
        """
        @todo: il faudrait avoir value = None et verifier file names
        """
        if param.isout():
            value = param.getValue()
            if value is None :
                return True
            else:
                return False
        else:        
            return os.path.exists( param.getValue() )
        


class FilenameDataType( DataType ):
 
    @staticmethod
    def convert( value ,param ):
        if value is None:
            return None
            #raise UserValueError( parameter = param , msg= " this parameter must be a String" )
 
        fileName = Mobyle.Utils.safeFileName( value )
        return fileName
   
    @staticmethod
    def validate( param ):
        value = param.getValue()
        if value is None :
            return True
        safefileName = Mobyle.Utils.safeFileName( value )
        if safefileName != value:
            msg = "invalid value: %s :the followings characters \:/ ;` {} are not allowed" %( value )
            raise UserValueError( parameter = param , msg = msg )
        else:
            return True

    
        


