#! /usr/bin/env python
#############################################################
#                                                           #
#   Author: Bertrand Neron                                  #
#   Organization:'Biological Software and Databases' Group, #
#                Institut Pasteur, Paris.                   #
#   Distributed under GPLv2 Licence. Please refer to the    #
#   COPYING.LIB document.                                   #
#                                                           #
#############################################################
"""
programsInstaller.py

This script is used to update the list of installed and imported programs
on the mobyle server, as well as the various index files.
"""
import sys, os
try:
    print >> sys.stderr, "Settings:"
    print >> sys.stderr, "MOBYLEHOME=%s" % os.environ['MOBYLEHOME']
except KeyError, ke:
    sys.exit("%s must be defined in your environment." % ke)

if ( os.path.join( os.environ[ 'MOBYLEHOME' ] ) ) not in sys.path:
    sys.path.append( os.environ[ 'MOBYLEHOME' ] )
if ( os.path.join( os.environ['MOBYLEHOME'] , 'Src' ) ) not in sys.path:    
    sys.path.append( os.path.join( os.environ['MOBYLEHOME'] , 'Src' ) )

import logging
import shutil
import glob
from Ft.Xml.Domlette import NoExtDtdReader, Print
from Ft.Lib import UriException

from Mobyle import MobyleLogger
MobyleLogger.MLogger()
r_log = logging.getLogger('mobyle.registry' )
console = logging.StreamHandler(sys.stderr)
r_log.setLevel(logging.INFO)
formatter = logging.Formatter('%(levelname)-8s %(message)s')
# tell the handler to use this format
console.setFormatter(formatter)
# add the handler to the root logger
r_log.addHandler(console)

from Mobyle.Registry import Registry, registry
from Mobyle.SearchIndex import SearchIndex
from Mobyle.ClassificationIndex import ClassificationIndex
from Mobyle.DataInputsIndex import DataInputsIndex
from Mobyle.DescriptionsIndex import DescriptionsIndex


import Mobyle.ConfigManager

class ProgramInstaller:
    """
    @author: Bertrand Neron
    @organization: Institut Pasteur
    @contact:mobyle@pasteur.fr
    """

    def __init__( self ):
        self.cfg = Mobyle.ConfigManager.Config()
        self.customPrefix =  os.path.join ( os.environ['MOBYLEHOME'], "Local" , "Programs")
        self.publicPrefix = os.path.join ( os.environ['MOBYLEHOME'], "Programs")
               
        
    def clean( self , program = 'all' , server = 'all'):
        """
        clean/remove a specific program or every program on every server
        @param program: program name
        @type program: string
        @param server: server name
        @type server: string
        """
        r_log.info("Published programs: cleaning %s/%s" % (server, program))
        if program == 'all':
            if server == 'local':
                directories = [self.cfg.programs_path()]
            elif server == 'all':
                directories = [self.cfg.programs_path(), self.cfg.programs_cache_path()]
            else:
                directories = [self.cfg.programs_cache_path()]
            programPaths = []
            for directory in directories:
                programPaths.extend(glob.glob( os.path.join( directory , '*.xml' ) ))
        else:
            programPaths = [registry.getProgramPath(program, server)]
        for path in programPaths:
            r_log.info("Published programs: removing %s" % path)
            try:
                os.unlink(path)
            except OSError , err:
                msg = "Program remove failed for path %s : %s" % (path, err)
                r_log.error( msg )



    def install(self , program = 'all'  , server = 'all' , force = False ):
        """
        install or import a specific program or every program from every server
        @param program: program name
        @type program: string
        @param server: server name
        @type server: string
        """
        if program is None :
            program = 'all' 
        if server is None :
            server = 'all'

        if server == 'all':
            if program == 'all':
                self.clean()
                self._localInstall()
                self._remoteInstall()
            else:
                self.clean( program = program )
                self._localInstall( program = program )
                self._remoteInstall( program = program )
                #msg = "when program is specified, server must be specified (\"all\" is not allowed )"
                #r_log.error( msg )
                #raise Exception, "error in argument value: "+ msg
        elif server == 'local':
            if program == 'all':
                self.clean(server='local')
                self._localInstall()
            else:
                #install uniquement ce prg du server local
                self._localInstall( program , force = force )
        else:
            if program == 'all':
                self.clean( server = server )
                self._remoteInstall( server = server )
            else:
                self.clean( server = server  , program = program )
                self._remoteInstall( server = server , program = program ) 
    
    
    def index(self):
        """
        generate index files corresponding to the installed program
        """
        r_log.info("Regenerating indexes:")
        r_log.info("Reloading the registry...")
        registry.load()
        r_log.info("Regenerating search index...")
        SearchIndex.generate()
        r_log.info("Regenerating classification index...")
        ClassificationIndex.generate()
        r_log.info("Regenerating data inputs index...")
        DataInputsIndex.generate()
        r_log.info("Regenerating descriptions index...")
        DescriptionsIndex.generate()
        pass
    
    
    
    ###################################################
    #                                                 #
    #   private methods for local server operations   #
    #                                                 #
    ###################################################

    def _localInstall(self , program = 'all' , force = False ):
        """
        install one or all of the local programs
        @param program: program name
        @type program: string
        """
        if program == 'all' :
            r_log.info( "Installing all local programs" )
            programs = self._localPrograms2install()
        
        else:
            if self._isInConfig( program ):
                r_log.info( "Installing local program: %s" %program )
            else:
                if force :
                    r_log.info( "Force Installing local program: %s" %program )
                else:
                    r_log.info( "The program you want to install is not in Config. Add it in config or use force option to install it" )
                    sys.exit( 0 )
                
            customPath = os.path.join(self.customPrefix, program + '.xml')
            if os.path.exists( customPath ):
                programs = [ customPath ]
            else:
                publicPath = os.path.join(self.publicPrefix, program + '.xml')
                if os.path.exists( publicPath ):
                    programs = [ publicPath ]
                else:
                    r_log.error( "Local program install failed, path not found: %s" % program)
                    sys.exit(1)
                    
        
        
        for path in programs:
            destPath = os.path.join(self.cfg.programs_path(), os.path.basename( path ) )
            r_log.info( "Installing local programs: copying %s to %s "% (path, destPath))
            try:
                shutil.copy(path, destPath )
                os.chmod(destPath, 0644 )
            except OSError , err:
                msg = "Program install failed for path %s : %s" % ( path , err )
                r_log.error( msg )
                
        ################## FIX ME ##################
        # probleme des entites
        # je recopie systematiquement toutes les entites
        
        
        for Dir in [ 'Env' , 'Entities' ]:
            destDirPath = os.path.join( self.cfg.programs_path() , Dir )
            if not os.path.exists( destDirPath ) :
                try:
                    os.makedirs( destDirPath , 0744 )
                except IOError ,err:
                    print 
                    msg = "can't create dir %s : %s" %( destDirPath , err )
                    err.message = msg
                    r_log.error( msg )
                    raise err
                
            toCopy = self._uniq( glob.glob( os.path.join( self.publicPrefix, Dir , "*.xml") ) , 
                                glob.glob( os.path.join( self.customPrefix,  Dir , "*.xml")) )
            for path in toCopy:
                destPath = os.path.join( destDirPath , os.path.basename( path ) )
                r_log.info("Installing entities and environment: copying %s to %s " % (path , destPath))
                try:
                    shutil.copy( path , destPath )
                except IOError ,err :
                    msg = "can't copy entities and environment : %s" %( err )
                    err.message = msg
                    r_log.error( msg )
                    raise err
                
                os.chmod( destPath , 0644 )

        ################## FIN  FIX ME ##################



    def _uniq(self , publicPath , customPath ):
        """
        @return: a list containing one xml path per service with the priority to the custom xml   
        """
        result = {}
        for path in publicPath:
                result[ os.path.basename( path )[:-4] ] = path
        for path in customPath:
            result[ os.path.basename( path )[:-4] ] = path
        return result.values()
    
    

    def _localPrograms2install(self ):
        """
        @return: the list of local (from this server) program file paths to install
        @rtype: list of strings
        """
        programs = {}
        for method in self.cfg.programs_installation_order():
            getattr( self, '_' + method )( programs )
        
        return programs.values()
    
    
    def _isInConfig( self , programName ):
        """
        @param programName: the name of a program
        @type programName: string
        @return: True if the programName is in the program define in Config.py, False otherwise
        @rtype: boolean
        """
        programs = {}
        for method in self.cfg.programs_installation_order():
            getattr( self, '_' + method )( programs )
            
        if programs.has_key( programName ):
            return True
        else:
            return False
        
        
    def _include( self , programs ):
        """
        @return: the list of local (from this server) program file paths include in install
        @rtype: list of strings
        """
        for mask in self.cfg.programs_installation_include():
            for path in self._uniq( glob.glob( os.path.join( self.publicPrefix , mask + '.xml' )) , 
                                    glob.glob( os.path.join( self.customPrefix , mask + '.xml' ))
                                    ):
                programs[ os.path.basename( path )[:-4] ] = path
            
        return programs

        
    def _exclude( self , programs ):
        """
        @return: the list of local (from this server) program file paths to exclude from install
        @rtype: list of strings
        """
        for mask in self.cfg.programs_installation_exclude():
            for path in self._uniq( glob.glob( os.path.join( self.publicPrefix , mask + '.xml' )) ,
                                    glob.glob( os.path.join( self.customPrefix , mask + '.xml' ))
                                    ):
                try:
                    del( programs[ os.path.basename( path )[:-4] ] )
                except KeyError:
                    pass
        return programs    
    
    
    
    ###################################################
    #                                                 #
    #  private methods for remote server operations   #
    #                                                 #
    ###################################################
    
    def _remoteInstall(self , server='all', program = 'all' ):
        """
        install one or all of the remote programs
        @param server: program name
        @type server: string
        @param program: program name
        @type program: string
        """
        r_log.info( "Installing remote programs: %s/%s" % (server, program))
        if server=='all':
            remoteServers = registry.serversByName
            del remoteServers['local']
            serverNames = remoteServers.keys()
        else:
            serverNames = [server]
        if program == 'all' :
            programs = [(p.server.name, p.name, p.url) for p in registry.programs if p.server.name in serverNames]
        else:
            programs = [(p.server.name, p.name, p.url) for p in registry.programs if p.name==program and p.server.name in serverNames]

        for server, program, url in programs:
            destPath = registry.getProgramPath(program, server)
            r_log.info( "Installing remote programs: copying %s to %s "  % (url , destPath))
            try:
                doc = NoExtDtdReader.parseUri(url)
                destFile = open(destPath, 'w')
                Print(doc, stream=destFile)
                destFile.close()
                os.chmod( destPath , 0644 )
            except  UriException, err:
                msg = "Program install failed for path %s : %s" % (url, err)
                r_log.error( msg )


    

if __name__ == '__main__':
    import getopt

    def usage():
        return """usage programInstaller <options> command
    commands
           install : install program in    available options [-p , -s ]
           clean   : clean the repository  available options [-p , -s ]
           index   : generate indexes      no options
    options
           -p --program programName: apply the command only on this program 
           -s --server MobyleServerName : apply the command on programs from 
                this server. default is 'all' ( if -p is specified -s must be specified too and not 'all' )
           -f --force : force to install a program even it is not in the config.py
           -h --help : print this
    examples:
           programInstaller.py install 
               install all programs defined in configuration ( local and remote )
           
           programInstaller.py  --program squizz --server 'local' install
               install squizz interface only
           
           programInstaller.py  --program squizz --server 'local' --force install
               install squizz interface only even squizz is not in config.py
           
           programInstaller.py --server marygay install
               install all programs imported from Mobyle server named "marygay"
                       
           programInstaller clean
               clean program repository from all interfaces
                
           programInstaller -p squizz clean
               remove only squizz interface
               
           programmInstaller index
           
    """   

    try:
        opts, cmds = getopt.getopt(sys.argv[1:], "hfp:s:", ["help", "force" , "program=", "server="])
    except getopt.GetoptError , err :
        r_log.error(err)
        print >> sys.stderr, usage()
        sys.exit( 1 )
    
    if len( cmds ) != 1 :
        print >> sys.stderr, usage()
        sys.exit( 1 )
    else:
        command = cmds[0]
        installer = ProgramInstaller()
        
        server = 'all'
        program = 'all'
        force = False
        for opt , value in opts:
            if opt == "-s" or opt == "--server" :
                server = value
            elif opt == "-p" or opt == "--program":
                program = value
            elif opt == "-f" or opt == "--force":
                force = True 
            if opt == "-h" or opt == "--help":
                print >> sys.stderr, usage()
                sys.exit( 0 )

        if program != 'all' and server == 'all':
            print >> sys.stderr , "when program is specified, server must be specified (\"all\" is not allowed )"
            print >> sys.stderr, usage()
            sys.exit( 1 )
        if command == "install":
            installer.install( program = program , server = server , force = force )
            installer.index()
        elif command  == "clean":
            installer.clean( program = program , server = server )
            installer.index()
        elif command == "index":
            if  opts:
                print >> sys.stderr, usage()
                sys.exit( 1 )
            else:
                installer.index()
        else:
            print >> sys.stderr, usage()
            sys.exit( 1 )
        sys.exit( 0 )