#! /usr/bin/env python

import sys, time, signal
import getopt
import os
import subprocess


import Mobyle.ConfigManager
import Mobyle.Session

# ======================================
#
#            TmpCleaner  CLASS
#
# ======================================

class Cleaner :
    """
    clean Results or Session directory of Mobyle.

    It's supposed that Results tree is:

       results_path = '/pasteur/mobyle/htdocs/Mobyle/Results'
       jobName_path = results_path + '/jobName'
       jobKey_path = jobName_path+ '/jobKey'
       file_path = jobKey_path + '/file'

       exemple:

       jobName_path = '/pasteur/mobyle/htdocs/Mobyle/Results/cap3'
       jobKey_path = '/pasteur/mobyle/htdocs/Mobyle/Results/cap3/A063675248410'
       file_path = '/pasteur/mobyle/htdocs/Mobyle/Results/cap3/A063675248410/dat'

    It's supposed that Session tree is:

       anonymous_session_path = '/pasteur/mobyle/htdocs/Mobyle/Sessions/anonymous
    """


    def __init__( self, config = Mobyle.ConfigManager.Config(), force = False, verbose=False, session = False ):
        """
        force = True => cleaner without verification of process status
        force = False =>   ..   with         ..            ..     ..
        """

        self.config = config

        self.zipSuffix = ['tar', 'gz', 'zip', 'tgz']

        self.force = force
        self.verbose = verbose
        if session:
            self.session = Mobyle.Session.SessionFactory( self.config )
        # current time in seconds ( float )
        self.current_time = int( time.time() )


    def isForce( self ):
        """
        Test if force rmdir or not  --> boolean
        """
        return self.force


    def isJobNameToClean( self, jobName ):
        """
        Test if jobName is a directory to deal with --> boolean

        Directory which are not jobName:
        - special directory like: the administrative directory (ADMINDIR) or the pipeline directory (PIPELINE)
        """
        dirName = os.path.basename( jobName )
        if dirName == 'ADMINDIR':
            return False
        return os.path.isdir( jobName )


    def isJobKeyToClean( self, jobKey ):
        """
        Test if jobKey is a directory to deal with --> boolean
        """
        return os.path.isdir( jobKey )


    def isFileToClean( self, file ):
        """
        Test if file is a file to deal with --> boolean

        file which are not looked as file are:
        - a directory ( Normely, do not exist )
        - special file like: .[filename]
        """
        filename = os.path.basename( file )
        if filename[0] == '.':
            return False
        return os.path.isfile( file )

    def isOldJobKey( self, jobKey , jobTime ):
        """
        Test if the last modification time of jobKey directory is older than
        jobTime --> boolean
        """
        mtime = os.path.getmtime( jobKey )
        return self.current_time - mtime - jobTime > 0


    def isBigFile( self, file , fileSize ):
        """
        Test if file is biggest than fileSize --> boolean
        """
        return os.path.getsize( file ) >= fileSize

    def isOldZipFile( self, file , zipTime ):
        """
        Test if last modification time of file is older than zipTime --> boolean
        """
        suffix = file.split( '.' )[-1]
        mtime = os.path.getmtime( file )
        return suffix in self.zipSuffix and ( self.current_time - mtime - zipTime > 0 )


    def isRunningJobKey( self, jobName, jobKey ):
        """
        Test existence of jobKey in ADMINDIR --> boolean

        A jobKey is still active if a file like 'jobName.jobKey' exist in
        ADMINDIR.

        exemple:
        /pasteur/mobyle/htdocs/Mobyle/Results/ADMINDIR/blast2.B06713905771017

        """

        file = '%s.%s' %( jobName, jobKey )

        return os.path.isfile( '%s/ADMINDIR/%s' % ( self.config.results_path() , file ) )


    def jobKeyCleaner( self, jobTime = 0 ):
        """
        Remove jobKey directories older than jobTime.

        jobTime is a number of day.
        """
        if not jobTime:
            return

        if jobTime < 0:
            print >>sys.stderr, "File size must be positive"
            sys.exit()

        jobTime = self._dayToScd( jobTime )


        # Proceed all jobName in the results_path directory
        for jobName in os.listdir( self.config.results_path() ):

            jobName_path = os.path.join( self.config.results_path(), jobName )

            # Skip special directory or files
            if not self.isJobNameToClean( jobName_path ):
                continue

            # Proceed all jobKey directories from a job directory
            for jobKey in os.listdir( jobName_path ):

                jobKey_path = os.path.join( jobName_path, jobKey )

                # Skip special directory or files
                if not self.isJobKeyToClean( jobKey_path ):
                    continue

                # Delete old jobKey directory
                if self.isOldJobKey( jobKey_path , jobTime ):

                    # Check Process Status
                    if self.isRunningJobKey ( jobName, jobKey ):
                            continue
                    else:
                        if self.isForce():
                            cmd = "rm -rf %s" % jobKey_path
                            msg = "%s: REMOVE with force " % jobKey_path
                        else:
                            cmd = "rm -r %s" % jobKey_path
                            msg = "%s: REMOVE" % jobKey_path
                        try:
                            os.system( cmd )
                            if self.verbose:
                                print >>sys.stdout, msg
                        except IOError, err:
                            print >>sys.stderr, "%s" % ( err )


    def jobFileCleaner( self , fileSize=0, zipTime=0 ):
        """
        Remove zip file in jobKey directories older than ZipTime or/and
        file biggest than fileSize

        fileSize and zipTime are respectively Go and Hours.
        """
        if not fileSize and not zipTime:
            return

        if fileSize < 0:
            print >>sys.stderr, "File size must be positive"
            sys.exit()

        if zipTime < 0:
            print >>sys.stderr, "Zip hour must be positive"
            sys.exit()


        fileSize = self._GoToOctet( fileSize )
        zipTime = self._hourToScd( zipTime )

        # Proceed all job directories from the tmp directory
        for jobName in os.listdir( self.config.results_path() ):

            jobName_path = os.path.join( self.config.results_path(), jobName )

            # Skip special directory or files
            if not self.isJobNameToClean( jobName_path ):
                continue

            # Proceed all jobKey directories from a job directory
            for jobKey in os.listdir( jobName_path ):

                jobKey_path = os.path.join( jobName_path ,jobKey )

                # Skip special directory or files
                if not self.isJobKeyToClean( jobKey_path ):
                    continue

                # Proceed all files from a jobKey directory
                for file in os.listdir( jobKey_path ):

                    file_path = os.path.join( jobKey_path ,file )

                    # Skip special files
                    if not self.isFileToClean( file_path ):
                        continue

                    # Delete a big file or an archive file
                    if ( fileSize > 0 and self.isBigFile( file_path, fileSize )) or ( zipTime > 0 and self.isOldZipFile( file_path, zipTime )):
                        try:
                           os.unlink( file_path )
                           msg = "%s: UNLINK" % file_path
                           if self.verbose:
                               print >>sys.stdout, msg
                        except IOError, err:
                           print >>sys.stderr, "%s" % ( err )


    def isSessionActive( self, session ):
        try:
            job = session.getJobs()
            return job
        except Exception ,err :
            print err
            return 'SessionActiveError'
    
    def isSessionEmpty( self, session, sessionKey_path ):
        """
        return True if the .session.xml is empty
        """
        filename = session.FILENAME
        filename_path = os.path.join( sessionKey_path, filename )
        if os.path.exists(filename_path):
            size = os.path.getsize( filename_path )
            return size == 0
        else:
            return True

    def anonymousSessionCleaner( self, jobTime = 0 ):
        """
        Remove  anonymous session directories older than jobTime.

        jobTime is a number of day.
        """
        if not jobTime:
            return

        if jobTime < 0:
            print >>sys.stderr, "File size must be positive"
            sys.exit()

        jobTime = self._dayToScd( jobTime )
        anonymous_path = os.path.join( self.config.user_sessions_path(), 'anonymous')
        # Proceed all jobName in the results_path directory
        for sessionKey in os.listdir( anonymous_path ):

            anonymous = self.session.getAnonymousSession( sessionKey )
            sessionKey_path = os.path.join( anonymous_path, sessionKey )
            
            # Delete old sessionKey directory
            if self.isOldJobKey( sessionKey_path, jobTime ):
                # Check Process Status
                try:
                    jobs = anonymous.getJobs()
                except Exception ,err :
                    sessionEmpty = self.isSessionEmpty( anonymous, sessionKey_path )
                    if sessionEmpty:
                        print >> sys.stdout, "Warning: %s/%s was empty." % (sessionKey_path, anonymous.FILENAME )
                    else:
                        print >> sys.stdout,"error in session : ", err
                    
                    continue

                if jobs:
                    if self.verbose:
                        print >> sys.stdout, '%s: Session active' % sessionKey_path
                    continue
                else:
                    if self.isForce():
                        cmd = "rm -rf %s" % sessionKey_path
                        msg = "%s: REMOVE with force " % sessionKey_path
                    else:
                        cmd = "rm -r %s" % sessionKey_path
                        msg = "%s: REMOVE" % sessionKey_path
                    
                    try:
                        os.system( cmd )
                        if self.verbose:
                            print >>sys.stdout, msg
                    except IOError, err:
                        print >>sys.stderr, "%s" % ( err )                   



    def _GoToOctet( self, fileSize ):
        return fileSize * 1024 * 1024 * 1024

    def _dayToScd( self, jobTime ):
        return jobTime * 24 * 60 * 60

    def _hourToScd( self, zipTime ):
        return zipTime * 24 * 60 * 60
