/**
 * Author: Herve Menager
 * Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.
 * Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.
 */

/**
 * @author Herve Menager
 * @organization Institut Pasteur
 * @contact mobyle@pasteur.fr
 */
var Mobyle = new Object();
/**
 * tabMethods is the set of methods that manage tabbed panels manipulation
 */
Mobyle.tabsMethods = {

	/**
	 * insert a tabs container inside any given container
	 * 
	 * @param {Element}
	 *            element container
	 */
	insertTabsContainer : function(element) {
		$(element).insert("<ul class='handlesList' />");
		$(element).insert("<div class='panelsList' />");
	},

	/**
	 * insert a tab in a given tabs container.
	 * 
	 * @param {Element}
	 *            element the element containing the tabs (handles+panels)
	 * @param {String}
	 *            id the id of the tab to be created
	 * @param {String}
	 *            label the label of the tab to be created (displayed in the
	 *            handle)
	 * @param {boolean}
	 *            close indicates wether or not a close control should be
	 *            created
	 */
	insertTab : function(element, id, label, close) {
		if (!($(element).down('ul.handlesList') && $(element).down(
				'div.panelsList'))) {
			throw Error("element " + $(element).id
					+ " is not a valid tabs container");
		}
		// create tab html if it does not exist
		if (!$(id + '_tabPanel')) {
			// insert handle
			$(element).down('ul.handlesList').__insertHandle(id, label, close);
			// insert panel
			$(element).down('div.panelsList').__insertPanel(id);
		}
		// set show on click behaviour
		Event.observe($(id + '_tabHandle'), 'click', function(event) {
			Event.element(event).showTab();
			Event.stop(event);
		});
		// set close behaviour
		if (close) {
			Event.observe($(id + '_tabHandle').up('tr').down('.closeLink'),
					'click', function(event) {
						Event.element(event).up('tr').down('.tabHandle')
								.removeTab();
						Event.stop(event);
					});
		}
		$(id + '_tabHandle').showTab();
	},

	/**
	 * remove a tab
	 * 
	 * @param {Element}
	 *            element the handle link of the element that should be removed
	 */
	removeTab : function(element) {
		var panelId = $(element).href.gsub('(.*)\#', '');
		if ($(element.up('li')).siblings().last())
			var nextTab = $(element.up('li')).siblings().last().down(
					'.tabHandle');
		$(element.up('li')).remove();
		$(panelId).remove();
		// show previous tab
		if (nextTab)
			nextTab.showTab();
	},

	/**
	 * insert the html for a tabbed panel
	 * 
	 * @param {Element}
	 *            element the tab panels container
	 * @param {String}
	 *            id the id of the created tab
	 */
	__insertPanel : function(element, id) {
		var panelTemplate = new Template(
				"<div class='tabPanel' id='#{id}_tabPanel'></div>");
		$(element).insert(panelTemplate.evaluate( {
			'id' :id
		}));
	},

	/**
	 * insert the html for a tab handle
	 * 
	 * @param {Element}
	 *            element the tab handles container
	 * @param {String}
	 *            id the id of the tab to be created
	 * @param {String}
	 *            label the label of the tab to be created (displayed in the
	 *            handle)
	 * @param {boolean}
	 *            close indicates wether or not a close control should be
	 *            created
	 */
	__insertHandle : function(element, id, label, close) {
		if (close) {
			var closeLink = "<td><a href='#' class='closeLink' title='close this tab'>x</a></td>";
		}
		var handleTemplate = new Template(
				"<li><table><tr><td><a href='##{id}_tabPanel' class='tabHandle' id='#{id}_tabHandle'>#{label}</a></td>#{closeLink}</tr></table></li>");
		$(element).insert(handleTemplate.evaluate( {
			'id' :id,
			'label' :label,
			'closeLink' :closeLink
		}));
	},

	/**
	 * display a tab
	 * 
	 * @param {Element}
	 *            element the handle link or the panel of the element that
	 *            should be displayed
	 */
	showTab : function(element) {
		if ($(element).hasClassName('tabPanel')) {
			var handleId = $(element).id.sub('_tabPanel', '_tabHandle');
			var element = $(handleId);
		} else if (!$(element).hasClassName('tabHandle')) {
			throw Error("this element is not a tab handle or panel");
		}
		var liEl = $(element.up('li'));
		liEl.siblings().each( function(el) {
			if (el.down('a.tabHandle')) {
				el.removeClassName('selected');
				var panelId = el.down('a').href.gsub('(.*)\#', '');
				$(panelId).hide();
			}
		});
		liEl.addClassName('selected');
		var panelId = liEl.down('a').href.gsub('(.*)\#', '');
		$(panelId).show();
		// recursively show the parent tabs of the shown tab
	if ($(element).up('.tabPanel')) {
		$(element).up('.tabPanel').showTab();
	}
},

/**
 * insert / show a job tab
 * 
 * @param {Element}
 *            element the tab handles container
 * @param {String}
 *            id the job id
 */
getJobTab : function(element, id) {
	var tabId = id + '_tabHandle';
	// if the job tab exists
	if ($(tabId)) {
		$(tabId).showTab();
	} else {
		// otherwise we just create it
		$(element).insertTab(id, "loading job<br />&nbsp;", true);
		var p = $H( {
			jobid :id
		});
		var op = $H( {
			el :$(id + '_tabPanel')
		});
		new MobyleJob(p, op);
	}
},

getFormTab : function(element, programUrl, paramName, fileName) {
	var tabId = programUrl + '_tabHandle';
	// if the form exists
	if ($(tabId)) {
		$(tabId).showTab();
		// update the databox if needed
		if (paramName && fileName) { // update the databox if this is part of
										// a pipe
			var updatedDatabox = portal.forms.get(programUrl).databoxes
					.get(paramName);
			updatedDatabox.setInputField(updatedDatabox.databoxFields
					.get('result'));
			updatedDatabox.historyFields.get('result').dataSelectData
					.selectEntry(fileName);
		}
	} else {
		// otherwise create the form
		$(element).insertTab(programUrl, "loading program", true);
		var p = $H( {
			programUrl :programUrl
		});
		var otherpars = $H( {
			el :$(programUrl + '_tabPanel')
		});
		if (paramName && fileName) {
			otherpars.update( {
				paramName :paramName,
				fileName :fileName
			})
		}
		portal.forms.set(programUrl, new MobyleForm(p, otherpars));
	}
},

getTutorialTab : function(element, href, label) {
	var tabId = href + '_tabHandle';
	// if the tutorial exists
	if ($(tabId)) {
		$(tabId).showTab();
	} else {
		// otherwise create the tutorial tab and fill it
		$(element).insertTab(href, label, true);
		new Ajax.Updater($(href + '_tabPanel'), href, {
			method :'GET'
		});
	}

},

/**
 * hides an element this method overrides prototypejs', because display:none
 * only sometimes is not always sufficient to hide a panel in IE
 * 
 * @param {Element}
 *            element the element to hide
 */
hide : function(element) {
	$(element).style.display = 'none';
	$(element).style.visibility = 'hidden';
},

/**
 * show an element this method overrides prototypejs', because display:none only
 * sometimes is not always sufficient to hide a panel in IE
 * 
 * @param {Element}
 *            element the element to show
 */
show : function(element) {
	$(element).style.display = '';
	$(element).style.visibility = '';
}

}

Element.addMethods(Mobyle.tabsMethods);