/**
 * Author: Herve Menager
 * Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.
 * Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.
 */

/**
 * @class MobyleProgramsList is a MobyleWidget class that represents the list of
 *        available programs on the server (and potentially elsewhere). It
 *        includes a search field that lets users filter the programs using a
 *        search string. It includes:
 *        <ul>
 *        <li>cgi call to get the list of programs</li>
 *        <li>Tree behaviour initialization</li>
 *        </ul>
 * 
 * @author Herve Menager
 * @organization Institut Pasteur
 * @contact mobyle@pasteur.fr
 * @extends MobyleWidget
 */
MobyleProgramsList = Class.create();
MobyleProgramsList.prototype = Object.extend( {

	el :'programs_list_channel',

	cgiurl :'programs_list.py',

	initAjaxParams : function() {
	},

	onCompleteFunction : function() {

		// initializing search form values and backing them up
		if (this.cgiparams && this.cgiparams.searchString) {
			$("searchString").value = this.cgiparams.searchString;
			this.previousSearchString = this.cgiparams.searchString;
		} else {
			this.previousSearchString = '';
		}
		if (this.cgiparams && this.cgiparams.fullSearch) {
			var fsEl = $("fullSearch");
			for (i = 0; i < fsEl.options.length; i++) {
				if (fsEl.options[i].value.strip() == this.cgiparams.fullSearch
						.strip()) {
					fsEl.selectedIndex = i;
				}
			}
			this.previousFullSearch = this.cgiparams.fullSearch;
		} else {
			this.previousFullSearch = "on";
		}
		this.initProgramNodes();
		this.initCategoryNodes();
		Event.observe("programSearch", 'submit', function(e) {
			$("programSearch").disable();
			if ($("programsList")) {
				$("programsList").disable();
			}
			this.submitSearch();
			Event.stop(e);
		}.bindAsEventListener(this));
		if ($("programsList")) {
			Event.observe("programsList", 'click', function(e) {
				$("programSearch").disable();
				if ($("programsList")) {
					$("programsList").disable();
				}
				this.displayAll();
				Event.stop(e);
			}.bindAsEventListener(this));
		}

	},

	initProgramNodes : function() {

		var mpl = this;
		this.el.select(".tree_program").each( function(element) {
			new MobyleTreeProgramNode(element, mpl);
		});

	},

	initCategoryNodes : function() {

		this.el.select(".tree_category,.tree_server").each( function(element) {
			new MobyleTreeNode(element);
		});

	},

	submitSearch : function() {

		// before to set the timeout, we check if the search parameters
	// have really changed
	p = new Object();
	p.searchString = $F("searchString");
	this.cgiparams = p;
	this.build();

},

displayAll : function() {

	var p = new Object();
	this.cgiparams = p;
	this.build();

}

}, MobyleWidget.prototype);

treeNodeMethods = {

	showTreeNode : function(element) {
		element = $(element);
		element.ancestors().each( function(ancestorEl) {
			if (ancestorEl.hasClassName("tree_category")) {
				Element.removeClassName(ancestorEl, 'tree_close');
			}
		});
	},

	hideTreeNode : function(element) {
		element = $(element);
		element.ancestors().each( function(ancestorEl) {
			if (ancestorEl.hasClassName("tree_category")) {
				Element.addClassName(ancestorEl, 'tree_close');
			}
		});
	}

}

Element.addMethods(treeNodeMethods);

/**
 * @class MobyleTreeProgramNode initializes the behaviour of a program link in
 *        the MobyleProgramsList
 * 
 * @author Herve Menager
 * @organization Institut Pasteur
 * @contact mobyle@pasteur.fr
 * @extends MobyleWidget
 */
MobyleTreeProgramNode = Class.create();
MobyleTreeProgramNode.prototype = {
	initialize : function(el) {

		this.el = el;
		this.initializeBehaviour();
		Object.extend(this, this.el);

	},

	initializeBehaviour : function(el) {

		this.linkEl = this.el.select(".program_link").first(); // N.B.: There
																// is only one
																// link for a
																// category
		Event.observe(this.linkEl, 'click', function(e) {
			Event.stop(e);
			this.showService();
		}.bindAsEventListener(this));

	},

	showTooltip : function() {

		if (this.timeOutTooltipId) {
			clearTimeout(this.timeOutTooltipId);
		}
		this.timeOutTooltipId = setTimeout(( function() {
			Element.show(this.tooltipEl)
		}).bind(this), 500);

	},

	hideTooltip : function() {

		if (this.timeOutTooltipId) {
			clearTimeout(this.timeOutTooltipId);
		}
		Element.hide(this.tooltipEl);

	},

	moveTooltip : function(e) {

		Element.setStyle(this.tooltipEl, {
			position :"absolute",
			top :(Event.pointerY(e) - e.currentTarget.offsetParent.offsetTop)
					+ "px",
			left :(Event.pointerX(e) - e.currentTarget.offsetParent.offsetLeft)
					+ "px"
		});

	},

	showService : function() {

		// get the program name corresponding to the form that has to be
		// generated
	// create a form tab by invoking the tab's id (program name here)
	var programUrl = this.linkEl.href;
	var programName = this.linkEl.innerHTML;
	$('services_tabPanel').getFormTab(programUrl, programName);

}
},

/**
 * @class MobyleTreeNode initializes the behaviour of a category node in the
 *        MobyleProgramsList
 * 
 * @author Herve Menager
 * @organization Institut Pasteur
 * @contact mobyle@pasteur.fr
 * @extends MobyleWidget
 */
MobyleTreeNode = Class.create();
MobyleTreeNode.prototype = {
	initialize : function(el) {

		this.el = el;
		this.initializeBehaviour();

	},

	initializeBehaviour : function(el) {

		this.linkEl = this.el.select(".tree_node_link").first(); // N.B.:
																	// There is
																	// only one
																	// link for
																	// a
																	// category
	Event.observe(this.linkEl, 'click', function(e) {
		this.toggleTreeStyle();
		Event.stop(e);
	}.bindAsEventListener(this));
	if ($("searchString").value == '') {
		this.toggleTreeStyle();
	}

},

toggleTreeStyle : function() {

	if (!Element.hasClassName(this.el, 'tree_close')) {
		Element.addClassName(this.el, 'tree_close');
	} else {
		Element.removeClassName(this.el, 'tree_close');
	}
	return false;

}
}
