/**
 * Author: Herve Menager
 * Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.
 * Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.
 */

/**
 * @class MobyleJSONList is a class that downloads JSON Data from a given URL
 *        and stores it. Users can get/filter Data from this class using the
 *        getData method It is the equivalent to a Model in a classical MVC
 * 
 * @extends Enumerable
 */
MobyleJSONList = Class.create();
Object.extend(MobyleJSONList.prototype, Enumerable);
Object.extend(MobyleJSONList.prototype, {

	initialize : function(cgiUrl) {

		this.cgiUrl = cgiUrl;
		this.views = $A();
		this.build();
		// here we register an event handler that takes care of removing a view
	// from the listeners if its HTML is removed from the DOM
	Event.observe(window, 'DOMNodeRemoved', function(e) {
		el = e.target;
		this.views.each( function(view) {
			if (view.viewEl.descendantOf(el)) {
				view.remove();
			}
		});
	}.bind(this));

},

build : function(callbackMethod, callbackObject) {

	this.items = $A();
	this.views.each( function(view) {
		view.setLoading();
	});
	// N.B: adding the date below is necessary for browsers such as IE, which do
	// not seem to actually load/refresh the data
	new Ajax.Request(this.cgiUrl, {
		method :'get',
		onSuccess : function(transport) {
			this.items = transport.responseText.evalJSON();
			// update each view of this model
		this.views.each( function(view) {
			view.build();
		});
	}.bind(this),
	onComplete : function(transport) {
		this.views.each( function(view) {
			view.unsetLoading();
		});
		if (callbackMethod != null) {
			callbackMethod(callbackObject);
		}
	}.bind(this)
	});

},

_each : function(iterator) {
	return this.items.each();
},

getData : function(filter) {

	var fitems = this.items;
	var index = 0;
	if (filter) {
		fitems = fitems.findAll( function(item) {
			if (filter(item) == true) {
				return item;
			}
		}.bind(this));
	}
	return fitems;

}

});

/**
 * @class MobyleJSONSelect is a View of the MobyleJSONList class It displays
 *        filtered elements of a list as an HTML select input It is the
 *        equivalent to a View in a classical MVC
 * 
 */
MobyleJSONSelect = Class.create();
Object
		.extend(
				MobyleJSONSelect.prototype,
				{
					initialize : function(selectEl, filter, optionValue,
							optionTitle, optionText, ajaxData, submitEl,
							sortFunction, onBuiltFunction) {

						this.selectEl = selectEl;
						this.filter = filter;
						this.optionValue = new Template(optionValue);
						this.optionTitle = new Template(optionTitle);
						this.optionText = new Template(optionText);
						if (onBuiltFunction) {
							this.onBuiltFunction = onBuiltFunction.bind(this);
						}
						if (sortFunction) {
							this.sortFunction = sortFunction;
						}
						if (submitEl) {
							this.submitEl = submitEl;
						}
						this.ajaxData = ajaxData;
						this.ajaxData.views[this.ajaxData.views.size()] = this;
						this.viewEl = this.selectEl;
						this.build();

					},

					remove : function() {

						// unregister this select as a view when it is destroyed
						this.ajaxData.views = this.ajaxData.views.without(this);

					},

					setLoading : function() {
					},

					unsetLoading : function() {
					},

					build : function() {

						this.data = this.ajaxData.getData(this.filter); // loading
																		// the
																		// data
																		// from
																		// the
																		// model
					if (this.sortFunction) {
						this.data = this.data.sortBy(this.sortFunction);
					}
					if (this.selectEl.selectedIndex != -1) {
						var selEntryBackup = this.selectEl.options[this.selectEl.selectedIndex]; // backing
																									// up
																									// current
																									// selected
																									// entry
																									// before
																									// rebuilding
																									// it,
																									// to
																									// avoid
																									// reinitialization
																									// side
																									// effects
					}
					// display/hide the select element, label and button
					// depending on data availability
					if (this.data.size() == 0) {
						this.selectEl.addClassName('empty_select');
						if ($$('*[for=' + this.selectEl.id + ']').first()) {
							$$('*[for=' + this.selectEl.id + ']').first()
									.addClassName('empty_label');
						}
						if (this.submitEl) {
							this.submitEl.addClassName('empty_submit');
							;
						}
						this.selectEl.disable();
					} else {
						this.selectEl.removeClassName('empty_select');
						if ($$('*[for=' + this.selectEl.id + ']').first()) {
							$$('*[for=' + this.selectEl.id + ']').first()
									.removeClassName('empty_label');
						}
						if (this.submitEl) {
							this.submitEl.removeClassName('empty_submit');
							;
						}
						this.selectEl.enable();
					}
					// remove previous options
					this.selectEl.immediateDescendants().invoke('remove');
					// default (blank) option
					var option = document.createElement("option");
					option.setAttribute("value", '');
					option.appendChild(document.createTextNode("< >"));
					this.selectEl.appendChild(option);
					// add html options
					this.data.each( function(item) {
						var option = document.createElement("option");
						option.setAttribute("value", this.optionValue
								.evaluate(item));
						option.setAttribute("title", this.optionTitle
								.evaluate(item));
						option
								.appendChild(document
										.createTextNode(this.optionText
												.evaluate(item)));
						this.selectEl.appendChild(option);
					}.bind(this));
					if (selEntryBackup && !selEntryBackup.value.blank()) { // only
																			// restore
																			// an
																			// entry
																			// if
																			// one
																			// had
																			// been
																			// selected...
						this.selectEntry(selEntryBackup.value, true);
					}
					if (this.onBuiltFunction) {
						this.onBuiltFunction();
					}

				},

				selectEntry : function(value, silentChange) {

					$A(this.selectEl.options).each( function(option) {
						if (option.value == value) {
							option.selected = true;
							if (!silentChange) {
								if (this.selectEl && this.selectEl.onchange) {
									this.selectEl.onchange(); // trigger the
																// onchange
																// events since
																// we just
																// changed its
																// value
						}
						if (this.submitEl && this.submitEl.onclick) {
							this.submitEl.onclick();
						}
					}
				}
			}.bind(this));

				}

				});

/**
 * @class MobyleJSONUl is a View of the MobyleJSONList class It displays
 *        filtered elements of a list as an HTML unordered list It is the
 *        equivalent to a View in a classical MVC
 * 
 */
MobyleJSONUl = Class.create();
Object.extend(MobyleJSONUl.prototype, {
	initialize : function() {

		this.initParameters();
		this.ajaxData.views[this.ajaxData.views.size()] = this;
		this.viewEl = this.ulistEl;
		this.build();

	},

	remove : function() {

		// unregister this select as a view when it is destroyed
		ajaxData.views = ajaxData.views.without(this);

	},

	setLoading : function() {
		this.ulistEl.addClassName('loading');
	},

	unsetLoading : function() {
		this.ulistEl.removeClassName('loading');
	},

	build : function() {

		this.setLoading();
		this.data = this.ajaxData.getData(this.filter); // loading the data from
														// the model
		// remove previous options
		this.ulistEl.immediateDescendants().invoke('remove');
		// add html list items
		this.data
				.each( function(item) {
					var li = document.createElement("li");
					li.setAttribute("id", this.liIdT.evaluate(item));
					li.setAttribute("title", this.liTitleT.evaluate(item));
					li.setAttribute("class", this.liClassT.evaluate(item));
					li.appendChild(document.createTextNode(this.liTextT
							.evaluate(item)));
					if (this.onLiClick) {
						li.onclick = this.onLiClick.bindAsEventListener(this,
								this.liIdT.evaluate(item), this.liTextT
										.evaluate(item));
					}
					this.ulistEl.appendChild(li);
				}.bind(this));
		this.unsetLoading();

	}

});

/**
 * @class MobyleJSONRefresh is a Control of the MobyleJSONList class It triggers
 *        the update of this list from the server using AJAX It is attached to a
 *        clickable HTML element (button, link...) It is the equivalent to a
 *        Control in a classical MVC
 * 
 */
MobyleJSONRefresh = Class.create();
Object.extend(MobyleJSONRefresh.prototype, {
	initialize : function(buttonEl, ajaxData) {

		this.buttonEl = buttonEl;
		this.ajaxData = ajaxData;
		Event.observe(this.buttonEl, 'click', function(e) {
			this.ajaxData.build();
			Event.stop(e);
		}.bindAsEventListener(this));
		// this.buttonEl.onclick =
		// this.ajaxData.build.bindAsEventListener(this.ajaxData);

	}
});

/**
 * @class MobyleJobsUl is a View of the list of jobs It displays the list of
 *        jobs in an HTML unordered list
 * 
 * @extends MobyleJSONUl
 */
MobyleJobsUl = Class.create();
MobyleJobsUl.prototype = Object.extend( {

	initParameters : function() {

		this.ulistEl = $('jobsListUl');
		this.filter = null;
		this.liIdT = new Template('#{id}');
		this.liTitleT = new Template('status: #{status}');
		this.liClassT = new Template('job_link, jobStatus#{status}');
		this.liTextT = new Template('#{programName} - #{jobDate}');
		this.ajaxData = portal.jobs;

	},

	onLiClick : function(event, id, text) {

		$('jobs_tabPanel').getJobTab(id);

	}

}, MobyleJSONUl.prototype);

/**
 * @class MobyleFilesUl is a View of the list of files It displays the list of
 *        files in an HTML unordered list
 * 
 * @extends MobyleJSONUl
 */
MobyleFilesUl = Class.create();
MobyleFilesUl.prototype = Object.extend( {

	initParameters : function() {

		this.ulistEl = $('filesListUl');
		this.filter = null;
		this.liIdT = new Template('#{id}');
		this.liTitleT = new Template('#{title}');
		this.liClassT = new Template('');
		this.liTextT = new Template('#{bioTypes} #{dataType}: #{userName}');
		this.ajaxData = portal.data;

	}

}, MobyleJSONUl.prototype);

/**
 * @class MobyleJSONTable is a View of the MobyleJSONList class It displays
 *        filtered elements of a list as an HTML table It is the equivalent to a
 *        View in a classical MVC
 * 
 */
MobyleJSONTable = Class.create();
Object.extend(MobyleJSONTable.prototype, {
	initialize : function() {

		this.initParameters();
		this.ajaxData.views[this.ajaxData.views.size()] = this;
		this.viewEl = this.tableEl;
		this.build();

	},

	remove : function() {

		// unregister this select as a view when it is destroyed
	ajaxData.views = ajaxData.views.without(this);

},

setLoading : function() {
	this.tableEl.addClassName('loading');
},

unsetLoading : function() {
	if (this.registerBehaviours) {
		this.registerBehaviours();
	}
	this.tableEl.removeClassName('loading');
},

build : function() {

	this.setLoading();
	this.data = this.ajaxData.getData(this.filter); // loading the data from the
													// model
	// remove previous options
	this.tableEl.childElements().invoke('remove');
	// table headers
	var tr = document.createElement("tr");
	this.columns.each( function(column) {
		var th = document.createElement("th");
		th.innerHTML = column.key;
		tr.appendChild(th);
	}.bind(this));
	this.tableEl.appendChild(tr);
	// table body (data)
	this.data.each( function(item) {
		var tr = document.createElement("tr");
		this.columns.each( function(column) {
			var td = document.createElement("td");
			td.innerHTML = column.value.evaluate(item);
			tr.appendChild(td);
		}.bind(this));
		this.tableEl.appendChild(tr);
	}.bind(this));
	this.unsetLoading();

}

});

/**
 * @class MobyleDataManagement is a view of the list of files stored in a
 *        MobyleSession. Additionally, it provides control (removal, renaming)
 *        over these data
 * @extends MobyleJSONTable
 */
MobyleDataManagement = Class.create();

MobyleDataManagement.prototype = Object
		.extend(
				{

					initParameters : function() {

						this.tableEl = $('dataManagement');
						this.filter = null;
						this.columns = $H( {
							'Name' :new Template('#{userName}'),
							'DataType' :new Template('#{dataType}'),
							'BioTypes' :new Template('#{bioTypes}'),
							'Size' :new Template('#{size} KiB'),
							'Data begin' :new Template('#{title}'),
							'Actions' :new Template(
									'<input type="button" class="remove_data_button" id="dataBookmark_#{id}" value="Remove">')
						});
						this.ajaxData = portal.data;

					},

					registerBehaviours : function() {

						this.tableEl.select(".remove_data_button").each(
								function(buttonEl) {
									new MobyleRemoveDataButton(buttonEl,
											this.ajaxData);
								}.bind(this));

					}

				}, MobyleJSONTable.prototype);

/**
 * @class MobyleRemoveDataButton is a Control of the portal data It triggers the
 *        removal of a data bookmark from the server using AJAX It is attached
 *        to an HTML input It is the equivalent to a Control in a classical MVC
 */
MobyleRemoveDataButton = Class.create();

MobyleRemoveDataButton.prototype = Object.extend( {

	initialize : function(buttonEl, ajaxData) {

		this.buttonEl = buttonEl;
		this.ajaxData = ajaxData;
		this.buttonEl.onclick = this.remove.bindAsEventListener(this);

	},

	remove : function() {

		this.p = new Object();
		this.buttonEl.disable();
		var id = this.buttonEl.id;
		id = id.substr(id.indexOf('_') + 1);
		this.p['id'] = id;
		new Ajax.Request('data_remove.py', {
			method :'post',
			postBody :$H(this.p).toQueryString(),
			onComplete : function() {
				// reload user data
			portal.data.build();
		}.bind(this)
		});

	}

});

MobyleSessionGauge = Class.create();

MobyleSessionGauge.prototype = Object.extend( {

	initialize : function(ajaxData) {

		this.gaugeEl = $('sessionUsage');
		this.ajaxData = ajaxData;
		this.ajaxData.views[this.ajaxData.views.size()] = this;
		this.viewEl = this.gaugeEl;
		this.gaugeSize = parseFloat($('sessionLimit').value);
		this.build();

	},

	remove : function() {

		// unregister this select as a view when it is destroyed
	ajaxData.views = ajaxData.views.without(this);

},

setLoading : function() {
},

unsetLoading : function() {
	if (this.registerBehaviours) {
		this.registerBehaviours();
	}
},

build : function() {

	this.setLoading();
	this.data = this.ajaxData.getData(this.filter); // loading the data from the
													// model

	this.gaugeLevel = 0.0;
	this.data.each( function(item) {
		this.gaugeLevel = this.gaugeLevel + parseFloat(item.size);
	}.bind(this));
	this.usage = this.gaugeLevel / this.gaugeSize * 100;
	this.gaugeEl.style.width = this.usage + "%";
	this.unsetLoading();

}

});
