########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.        #
#                                                                                      #
########################################################################################


import random , os , os.path , string
import re
import md5
import fcntl
import types
import glob
import Ft.Xml , Ft.Xml.Domlette
from Ft.Xml import EMPTY_NAMESPACE
from time import strftime , strptime , time , sleep 
import urllib2 , urlparse
from httplib import HTTPConnection

import Mobyle.Utils
import Mobyle.JobState
from Mobyle.Parser import ServiceParser , DataTypeFactory
from Mobyle.MobyleError import MobyleError , UserValueError , SessionError , AuthenticationError , NoSpaceLeftError , ParserError
from JobFacade import JobFacade
from Mobyle.Registry import registry

import logging 

s_log = logging.getLogger('mobyle.session')



import sys


       ############################################################
       #                                                          #
       #                    SessionFactory                        #
       #                                                          #
       ############################################################


class SessionFactory( object ):
    """
    This class defines a session, that stores all the information
    about a user that should be persistent on the server
    @organization: Institut Pasteur
    @contact:mobyle@pasteur.fr
    """
    __ref = None


    def __new__( cls , cfg ):
        if cls.__ref is  None:
            self = super( SessionFactory , cls ).__new__( cls )
            self.cfg = cfg
            self.__sessions = {}
            cls.__ref = self

        return cls.__ref


    def getAuthenticatedSession( self , email , passwd ):
        """
        @return: an already existing authenticated session.
        @param email: the user email
        @type email: a Mobyle.Net.Email instance
        @param passwd: the session pass word 
        @type passwd: string
        @raise AuthenticationError: if the passwd doesn't match the session passwd
        @raise AuthenticationError: the session doesn't already exists
        """
       
        mymd5 = md5.new()
        mymd5.update( email.To )
        key = mymd5.hexdigest()

        try:
            session = self.__sessions[ key ]
            if session.checkPasswd( passwd ):
                return session
            else:
                raise AuthenticationError , "There is no user with this email and password"
       
        except KeyError: 
            sessionDir = os.path.normpath( os.path.join( self.cfg.user_sessions_path() , "authentified" , key  ) )
          
            if os.path.exists( sessionDir ):
                session = AuthenticatedSession( self.cfg , email , passwd )
                self.__sessions[ session.getKey() ] = session  
                return session
            else: 
                raise AuthenticationError , "There is no user with this email" 



    def getAnonymousSession( self , key = None ):
        """
        @return: an anonymous session. If key is None create a new anonymous session
        @rtype: Session object
        @param key: the key to identfy a anonymous session
        @type key: string
        @raise SessionError: if key is specified and doesn't match any session.
        """
        anonymousSessionAllowed = self.cfg.anonymousSession()
        s_log.debug( "SessionFactory.getAnonymousSession( key = %s ) " % key  )
        if anonymousSessionAllowed== 'no':
            s_log.error("SessionFactory/can't create anonymous session ANONYMOUS_SESSION is set to \"no\" in Local/Config/Config.py")          
            raise MobyleError , "can't create anonymous session: permission denied"
      
        try:
            return self.__sessions[ key ]
        except KeyError: 
            session = None

        if key :
            sessionDir = self.__getSessionDir( key )
            
            if os.path.exists( sessionDir ):
                s_log.debug( "SessionFactory.getAnonymousSession( key= %s )/ the dir exist I 'll return this session" % key)
                session = AnonymousSession( self.cfg , key )
            else: 
                s_log.error( "can't retrieve anonymous session, the Key: %s doesn't match with any Session" % key )
                raise SessionError , "wrong Key: %s" % key
               
        else: #new session
            session = AnonymousSession( self.cfg )
            s_log.debug( "SessionFactory.getAnonymousSession( key= %s ) / a new anonymous has been created . I 'll return this session" % key)
            
        self.__sessions[session.getKey()] = session  
        s_log.debug( "SessionFactory.getAnonymousSession( key= %s ) I return this anonymous session :key="+str(session.getKey() )) 
        return session
   

    def createAuthenticatedSession( self , email , passwd ):
        """
        create an authenticated session whith email as login and passwd as pass word
        @param email: the user email
        @type email: a Mobyle.Net.Email object
        @param passwd: the user password
        @type passwd: string
        @return: a new autheticated session
        @rtype: session instance
        @raise AuthenticationError: if there is already a session with this email, or the email is not allowed on this server
        """
        authenticatedSessionAllowed = self.cfg.authenticatedSession()
      
        if authenticatedSessionAllowed == 'no':
            s_log.error("can't create  session AUTHENTICATED_SESSION is set to \"no\" in Local/Config/Config.py")          
            raise MobyleError , "can't create  authenticated session: permission denied"

   
        mymd5 = md5.new()
        mymd5.update( email.To )
        key = mymd5.hexdigest()
       

        if self.__sessions.has_key( key ) : 
            msg = "Try to create a new Session with email, this Session ( %s ) already exist" % key
            s_log.error( msg )        
            raise AuthenticationError , "user with the email you specify already exist" 
       
       
        else:  
            sessionDir = os.path.normpath( os.path.join( self.cfg.user_sessions_path() , "authentified" , key  ) )
          
            if os.path.exists( sessionDir ):
                msg = "Try to create a new Session with email, this Session ( %s ) already exist" % key
                s_log.error( msg )
                raise AuthenticationError , "user with the email you specify already exist" 
                
            session = AuthenticatedSession( self.cfg , email , passwd )
            self.__sessions[ session.getKey() ] = session   
            return session
  

    def removeSession( self , key ):
        sessionDir = self.__getSessionDir(  key )
       
        for File in os.listdir( sessionDir ):
            os.unlink( os.path.join( sessionDir , File ) )
        os.rmdir( sessionDir )
        del self.__sessions[ key ]


    def __getSessionDir( self , key ) :
 
        if  len ( key ) == 32  :
            #a md5 value is always encode in 32 char
            return os.path.normpath( os.path.join( self.cfg.user_sessions_path() , "authentified" , key  ) )
        else:
            ##the anonymous key have 15 char length
            return os.path.normpath( os.path.join( self.cfg.user_sessions_path() , "anonymous" , key ) )
 
 
    def __getSessionFilePath( self , key , email = None) :
        return os.path.normpath( os.path.join( self.__getSessionDir( key ) , Session.FILENAME  ) )
    



       ############################################################
       #                                                          #
       #                        Session                           #
       #                                                          #
       ############################################################



class Session( object ):
    """
    This class defines a session, that stores all the information
    about a user that should be persistent on the server
    @author: Bertrand Neron
    @organization: Institut Pasteur
    @contact:mobyle@pasteur.fr
    """

    FILENAME = '.session.xml'
    WRITE  = fcntl.LOCK_EX
    READ   = fcntl.LOCK_SH


    def __init__( self , Dir , key , cfg ):

        self.cfg = cfg
        """the maximun size of a session ( in bytes )"""
        self.sessionLimit = self.cfg.sessionlimit()
        self.key = str( key )
        """ the user/session  key"""
        self.Dir = Dir
        """ the absolute path to this session directory """
        self.__transactionFile = None
        self.__transactionLockType = None
        self._modifiedTransaction = False   


    def isLocal(self):
        """
        we cannot instanciate a distant session
        """
        return True
   
    def getDir( self ):
        """
        @return: the absolute path to this session
        @rtype: string
        """
        return self.Dir

    def getKey( self ):
        """
        @return: the session key
        @rtype: string
        """
        return  self.key 

    def getUrl( self ):
        """
        @return: the url to the session
        @rtype: string
        """
        return self.__path2url( self.Dir )

    
    def __path2url( self, Dir ):
        """
        translate a directory absolute path in mobyle tree into an url in Mobyle 
        @param dir: the directory to translate this dir should be in Mobyle tree. Doesn't check if the path exist
        @type dir: string
        @return: an url http coresponding to the directory dir
        @rtype: string
        @raise MobyleError: -L{MobyleError} 
        """
        cfg_dirs = self.cfg.dirs()
        sessionDir = os.path.normpath( cfg_dirs['user_sessions_path'] ) #remove the ending slash
        begin = Dir.find( sessionDir )

        if begin != -1:
            end = begin + len( sessionDir )
            extrapath = Dir[end:]

            ## if the 2nd argument of os.path.join begin by "/"
            ## it return only the second arg
            for i in range( 0 , len( extrapath )):
                if extrapath[i] == os.path.sep :
                    continue
                else:
                    break
        else:
            msg = str( Dir ) + " is not Mobyle directory compliant "
            s_log.error( msg )
            raise MobyleError , msg

        extrapath = os.path.join( cfg_dirs['sessions_url'] , extrapath[i:] )
        return urlparse.urljoin(  cfg_dirs['root_url'] , extrapath )



    def __url2path( self, url ):
        """
        translate an url ( in Mobyle ) in a dir into Mobyle tree
        @param url: the url to translate this url should be in Mobyle 
        @type url : string
        @return : the absolute path corresponding to url
        @rtype: string
        @raise MobyleError : -L{MobyleError}
        """
        protocol , host , path , a,b,c = urlparse.urlparse( url )
        cfg_dirs = self.cfg.dirs()
        if protocol +"://"+host  != cfg_dirs['root_url'] :
            msg =  "url2path: " + str( url ) + " is not Mobyle url compliant "
            s_log.error( msg )
            raise MobyleError, msg

        session_url = cfg_dirs['sessions_url']
        match = re.search( tmpurl , path)
        if match:
            begin , end = match.span()
            extrapath = path[ end : ]
            if extrapath[0] == os.sep:
                extrapath = extrapath[ 1 : ]

            return os.path.normpath( os.path.join( cfg_dirs['user_sessions_path'] , extrapath ))
        else:
            msg = "url2path: " + str( url ) + " is not Mobyle url compliant "
            s_log.error( msg )
            raise MobyleError , msg



    def _getTransaction( self , Type ):
        """
        @return: the transaction of this session
        @rtype: a L{Transaction} object
        @raise SessionError: if can't acces to the session
        @raise IOError: if the transaction is stay locked after 3 attemps. 
        """
        import sys
        fileName = os.path.normpath( os.path.join( self.Dir , Session.FILENAME  ) )
        try:
            if Type == self.READ :
                self.__transactionFile = open( fileName , 'r' )
            elif Type == self.WRITE :
                self.__transactionFile = open( fileName , 'r+' )
            else:
                raise MobyleError
        except IOError , err:
            msg = "can't open session %s : %s" %( self.getKey() , err )
            s_log.critical( msg )
            raise SessionError , msg
      
        IGotALock = False

        if Type == 1 :
            logType = 'READ'
        elif Type == 2 :
            logType = 'WRITE'
        else:
            logType = 'UNKNOWN LOCK( ' + str( self.__transactionLockType ) +' )'
        s_log.debug( "%f : %s : _getTransaction Type= %s ( call by= %s )"  %( time() ,
                                                                              self.getKey() ,
                                                                              ( 'UNKNOWN LOCK', 'READ' , 'WRITE' )[ Type ] ,
                                                                              os.path.basename( sys.argv[0] ) ,
                                                                              ))
      
        for attempt in range( 4 ):
            try:             
                fcntl.lockf( self.__transactionFile , Type | fcntl.LOCK_NB )
                IGotALock  = True
                s_log.debug( "%f : %s : _getTransaction IGotALock = True" %(time() , self.getKey() ))
                break
            except IOError , err:
                s_log.debug( "%f : %s : _getTransaction IGotALock = False" %(time() , self.getKey() ))
                sleep( 0.2 )

        if not IGotALock :
            self.__transactionFile.close()
            self.__transactionFile = None
            self.__transactionLockType = None
            s_log.error( "%s : %s" %( self.getKey() , err ) )

            s_log.debug( "%f : %s : _getTransaction Type= %s ( call by= %s )"  %( time() ,
                                                                                  self.getKey() ,
                                                                                  Type ,
                                                                                  os.path.basename( sys.argv[0] ) 
                                                                                  ))
               
            raise IOError , err

        self.__transactionLockType = Type
        try:
            transaction= None
            ##################################################
            #transaction = pickle.load( self.__transactionFile )
            transaction = self._xml2Struct(  )
            ###################################################
        except NotImplementedError , err:
            import shutil
            shutil.copy2( self.__transactionFile.name , "%s.copy.%d" %( self.__transactionFile.name , int( time() ) ))
          
            msg = """
an error occured in _getTransaction : 
session id = %s/%s 
Type =  %s
IGotALock = %s
transaction = %s
                """ %( self.Dir ,
                       self.key ,
                       self.__transactionLockType ,
                       IGotALock,
                       transaction
                      )
            s_log.debug( msg  , exc_info = True )
            s_log.error( msg , exc_info = True )      
            raise SessionError , err
      
        return transaction

    
    def _commit( self , transaction ):
        """
        """
        import sys
        if self.__transactionLockType == 1 :
            logType = 'READ'
        elif self.__transactionLockType == 2 :
            logType = 'WRITE'
        else:
            logType = 'UNKNOWN LOCK( ' + str( self.__transactionLockType ) +' )'

        s_log.debug( "%f : %s : _commit Type= %s ( call by= %s )"  %( time(),
                                                                      self.getKey() ,
                                                                      logType ,
                                                                      os.path.basename( sys.argv[0] ) 
                                                                      ))
        try:
            if self.__transactionLockType == self.WRITE and self._modifiedTransaction :
                try:
                    tmpFile = open( "%s.%d" %(self.__transactionFile.name , os.getpid() )  , 'w' )
                    ##########################################################
                    doc = self._struct2xml( transaction )
                    Ft.Xml.Domlette.PrettyPrint( doc , tmpFile )
                    #pickle.dump( transaction , tmpFile  )
                    ##########################################################
                    tmpFile.close()
                    os.rename( tmpFile.name , self.__transactionFile.name )
                except IOError , err :
                    msg = "can't commit this transaction: " + str( err )
                    s_log.error( msg )
                    raise SessionError , msg
             
                except Exception ,err:
                    s_log.error( "session/%s self.__transactionFile = %s" %( self.getKey(), self.__transactionFile ) , 
                                 exc_info = True )
                    raise err
        finally:
            try:
                s_log.debug("%f : %s : _commit UNLOCK type= %s modifiedTransaction= %s" %(time() ,
                                                                                          self.getKey() ,
                                                                                          logType, 
                                                                                          self._modifiedTransaction
                                                                                          ))   
                fcntl.lockf( self.__transactionFile , fcntl.LOCK_UN  )
                self.__transactionFile.close()
                self.__transactionFile = None
                self.__transactionLockType = None
                self._modifiedTransaction = False #reset _modifiedTransaction to false for the next transaction
            except IOError , err :
                s_log.error( "session/%s : cannot UNLOCK  transaction type= %s modifiedTransaction= %s: " %(self.getKey(),
                                                                                                            logType,
                                                                                                            self._modifiedTransaction,
                                                                                                            err ) ) 
                pass
                #TODO 


    def _xml2Struct( self ):
        doc = Ft.Xml.Domlette.NoExtDtdReader.parseStream( self.__transactionFile , uri = "file://%s" % os.path.abspath( self.__transactionFile.name )) 
        root = doc.documentElement
        transaction = Transaction()
        try:
            authenticated = str( root.xpath( 'authenticated/text()' )[0].data )
            if authenticated == 'true':
                authenticated = True
        except IndexError:
            pass
        try:
           activated = str( root.xpath( 'activated/text()' )[0].data )
           if activated == 'true':
               transaction.activated = True
        except IndexError:
            pass
        try:
            activatingKey = str( root.xpath( 'activatingKey/text()' )[0].data )
            transaction.activatingKey = activatingKey
        except IndexError:
            pass
        try:           
            transaction.email = str( root.xpath( 'email/text()' )[0].data )
        except IndexError:
            pass
        try:           
            transaction.passwd = str(  root.xpath( 'passwd/text()' )[0].data )
        except IndexError:
            pass
        try:
            transaction.current_captcha_sol = root.xpath( 'captchaSolution/text()' )[0].data
        except IndexError:
            pass
        
        try:
            dataListNode = root.xpath( 'dataList/data')
        except IndexError:
            pass
        else:
            dtf = DataTypeFactory()
            for dataNode in dataListNode:
                data = {}
                dataID = str( dataNode.xpath( '@id' )[0].value )
                if transaction.datas.has_key( dataID ):
                    raise MobyleError ,"there is 2 data with the same id"
                try:
                    data[ 'userName' ] = str( dataNode.xpath( 'userName/text()' )[0].data )
                except IndexError:
                    pass
                
                #type
                try:
                    data[ 'Type' ] = ServiceParser.parseType( dataNode.xpath( 'type' )[0] , dataTypeFactory = dtf )
                except ( ParserError , MobyleError ) ,err :
                    pass
                try:
                    data[ 'dataBeg' ] = str( dataNode.xpath( 'headOfData/text()' )[0].data )
                except IndexError:
                    data[ 'dataBeg' ] = ''
                data[ 'inputModes' ] = [ str( inputmode.xpath( 'text()' )[0].data )  for inputmode  in dataNode.xpath( 'inputModes/inputMode' ) ]
                data[ 'producedBy' ] =  [ str( jobID.value ) for jobID in dataNode.xpath( 'producedBy/@ref' ) ]
                data[ 'usedBy' ] =  [ str( jobID.value ) for jobID in dataNode.xpath( 'usedBy/@ref' ) ]
                try:
                    data[ 'format' ] = ( str( dataNode.xpath('type/dataFormat/text()')[0].data ) ,
                                         str( dataNode.xpath( '@count' )[0].value ),
                                         'to fill the api (unused)' , 
                                         )
                except IndexError:
                    data[ 'format' ] = None
                data[ 'size' ] = dataNode.xpath( '@size' )[0].value
                transaction.datas[ dataID ] = data 
               
        try:
            jobListNode = root.xpath( 'jobList/job')
        except IndexError:
            pass
        else:
            for jobNode in jobListNode:
                job = {}
                jobID = str( jobNode.xpath( '@id' )[0].value )
                if transaction.datas.has_key( jobID ):
                    pass
                try:
                    job[ 'userName' ] = jobNode.xpath( 'userName/text()' )[0].data
                except IndexError:
                    pass
                try:
                    job[ 'programName' ] =  jobNode.xpath( 'programName/text()' )[0].data 
                except ( ParserError , MobyleError ) ,err :
                    pass
                try:
                    statusString = jobNode.xpath( 'status/text()' )[0].data
                except IndexError :
                    s_log.error( "the job %s in session %s has no status" %( jobID , self.getKey( )) )
                    raise MobyleError 
                else:
                    try:
                        job[ 'status' ] = Mobyle.Utils.status2code[ statusString ]
                    except KeyError:
                        s_log.error( "error in status %s for job %s in session %s" %( statusString ,
                                                                                     jobID , 
                                                                                     self.getKey() )
                        )
                        raise MobyleError 
                try:
                    job[ 'date' ] = strptime( jobNode.xpath( 'date/text()' )[0].data , "%x  %X")
                except IndexError:
                    pass  
              
                job[ 'dataProduced' ] =  [ str( dataID.value ) for dataID in jobNode.xpath( 'dataProduced/@ref' ) ]
                job[ 'dataUsed' ] =  [ str( dataID.value ) for dataID in jobNode.xpath( 'dataUsed/@ref' ) ]
                
                transaction.jobs[ jobID ] = job 
       
        return transaction 
               
                
    def _addTextNode( self, doc , node , nodeName , content , attr = None):
        """ 
        add a text node named nodeName with content to the node node
        @param node: the node on which the new node will append 
        @type node: node element
        @param nodeName: the name of the new node
        @type nodeName: string
        @param content: the content of the new node
        @type content: string
        """

        newNode = doc.createElementNS( EMPTY_NAMESPACE , nodeName )
        if attr:
            for attrName in attr.keys():
                newNode.setAttributeNS( EMPTY_NAMESPACE , attrName , str( attr[ attrName ] ) )
        
        text_node = doc.createTextNode( str( content ) )
        newNode.appendChild( text_node )
        node.appendChild( newNode )


    def _struct2xml( self , transaction ):
        import StringIO
        doc = Ft.Xml.Domlette.implementation.createDocument( Ft.Xml.EMPTY_NAMESPACE , "userSpace" , None )
        root = doc.documentElement
       
        root.setAttributeNS( EMPTY_NAMESPACE , 'id' , self.getKey() ) 
       # False, True are not valid in xsd space ( valid values are  0 , 1 , false , true )
        self._addTextNode( doc , root , "authenticated" ,  str( self.isAuthenticated() ).lower() )
        self._addTextNode( doc , root , "activated"     ,  str( transaction.activated ).lower() ) 
        if transaction.email :
            self._addTextNode( doc , root , "email" ,  transaction.email )
        if transaction.activatingKey :
            self._addTextNode( doc , root , "activatingKey" ,  transaction.activatingKey )
        if transaction.passwd:  
            self._addTextNode( doc , root , "passwd" ,  transaction.passwd ) 
        if transaction.current_captcha_sol :
            self._addTextNode( doc , root , "captchaSolution" ,  transaction.current_captcha_sol )
       
        if transaction.datas:
            dataListNode = doc.createElementNS( EMPTY_NAMESPACE , 'dataList' )
            root.appendChild( dataListNode )
            
            for dataId in transaction.datas.keys():
                dataNode = doc.createElementNS( EMPTY_NAMESPACE , 'data' )
                dataNode.setAttributeNS( EMPTY_NAMESPACE , 'id' , dataId )
                data = transaction.datas[ dataId ]
                dataNode.setAttributeNS( EMPTY_NAMESPACE , 'size' , str( data[ 'size' ] ) )
               
                if data[ 'format' ]:
                    dataFormat =  str( data[ 'format' ][0] )
                    count =  str( data[ 'format' ][1] )
                else:
                    dataFormat = None
                    count = None
                   
                if count:
                    dataNode.setAttributeNS( EMPTY_NAMESPACE , 'count' , count ) 
               
                self._addTextNode( doc , dataNode , "userName" ,  data[ 'userName' ] )
                #type
                typeNode = data['Type'].toDom()
                dataNode.appendChild( typeNode )
               
                if dataFormat:
                    self._addTextNode( doc , typeNode , "dataFormat" ,  dataFormat )
                    dataNode.appendChild( typeNode)
               
                self._addTextNode( doc , dataNode , "headOfData" ,  data[ 'dataBeg' ] )
               
                inputModeNode = doc.createElementNS( EMPTY_NAMESPACE , 'inputModes' )
                for inputMode in data[ 'inputModes' ]:
                    self._addTextNode( doc , inputModeNode , "inputMode" ,  inputMode )
                dataNode.appendChild( inputModeNode )

                for jobId in data['producedBy']:
                    producedByNode = doc.createElementNS( EMPTY_NAMESPACE , "producedBy" )
                    producedByNode.setAttributeNS( EMPTY_NAMESPACE , 'ref' , jobId )
                    dataNode.appendChild( producedByNode )
                    
                for jobId in data['usedBy']:
                    usedByNode = doc.createElementNS( EMPTY_NAMESPACE , "usedBy" )
                    usedByNode.setAttributeNS( EMPTY_NAMESPACE , 'ref' , jobId )
                    dataNode.appendChild( usedByNode )
                   
                dataListNode.appendChild( dataNode )
       
        if transaction.jobs:
            jobListNode = doc.createElementNS( EMPTY_NAMESPACE , 'jobList' )
            root.appendChild( jobListNode )
            
            for jobId in transaction.jobs.keys():
                job = transaction.jobs[ jobId ]
                jobNode = doc.createElementNS( EMPTY_NAMESPACE , 'job' )
                jobNode.setAttributeNS( EMPTY_NAMESPACE , 'id' , jobId )
               
                self._addTextNode( doc, jobNode, 'userName' , job[ 'userName' ])
                self._addTextNode( doc, jobNode, 'programName' , job[ 'programName' ])
                self._addTextNode( doc, jobNode, 'date' , strftime( "%x  %X" , job[ 'date' ] ) )
                
                self._addTextNode( doc, jobNode, 'status' , Mobyle.Utils.code2status[ job[ 'status' ] ] ) 
               
                for dataId in job['dataUsed']:
                    dataUsedNode = doc.createElementNS( EMPTY_NAMESPACE , "dataUsed" )
                    dataUsedNode.setAttributeNS( EMPTY_NAMESPACE , 'ref' , dataId )
                    jobNode.appendChild( dataUsedNode )

                for dataId in job['dataProduced']:
                    dataProducedNode = doc.createElementNS( EMPTY_NAMESPACE , "dataProduced" )
                    dataProducedNode.setAttributeNS( EMPTY_NAMESPACE , 'ref' , dataId )
                    jobNode.appendChild( dataProducedNode )
               
                jobListNode.appendChild( jobNode )

        return doc
 
       
    def _close( self , transaction ):
        """
        """
        import sys
        if self.__transactionLockType == 1 :
            logType = 'READ'
        elif self.__transactionLockType == 2 :
            logType = 'WRITE'
        else:
            logType = 'UNKNOWN LOCK( ' + str( self.__transactionLockType ) +' )'

        s_log.debug( "%f : %s : _close Type= %s ( call by= %s )"  %( time(),
                                                                     self.getKey() ,
                                                                     logType ,
                                                                     os.path.basename( sys.argv[0] ) 
                                                                     ))
            
        try:
            fcntl.lockf( self.__transactionFile , fcntl.LOCK_UN  )
            self.__transactionFile.close()
            self.__transactionFile = None
            self.__transactionLockType = None
        except IOError , err :
            s_log.error( "session/%s : an error occured during closing transaction : " %( self.getKey() , err )) 
            pass
            # a gerer   
        except Exception , err :
            msg = "session/%s : an error occured during closing transaction : " %( self.getKey() , err )
            s_log.error( msg ) 
            s_log.debug( "%f : %s : self.__transactionFile = %s" %(time() , self.__transactionFile ))
            s_log.debug( "%f : %s : self.__transactionLockType = %s" %(time() , logType ))
            raise SessionError , msg


    def getBaseInfo( self ):
        """
        @return: 3 basic informations about this session , the email , isAuthenticated , isActivated
        @rtype: ( string , boolean , boolean )
        """
        s_log.debug( "%f : %s : baseInfo call by= %s" %( time(),
                                                         self.getKey() , 
                                                         os.path.basename( sys.argv[0] ) 
                                                         ))       
        transaction = self._getTransaction( self.READ  )
        response = ( self._getEmail(transaction) , self.isAuthenticated() , transaction.activated )
        self._close( transaction )
        s_log.debug( "%f : %s : baseInfo return : %s" %( time() ,
                                                         self.getKey() , 
                                                         response ) )
        return response
   
   
    def isActivated( self ) :
        s_log.debug( "%f : %s : isActivated call by= %s" %( time() , 
                                                            self.getKey() , 
                                                            os.path.basename( sys.argv[0] ) 
                                                            ))
        transaction = self._getTransaction( self.READ  )
        activated = transaction.activated
        self._close( transaction )      
        return activated
    
    def getEmail(self):
        """
        @return: the user email address
        @rtype: string
        """
        s_log.debug( "%f : %s : getEmail call by= %s" %( time() ,
                                                         self.getKey() , 
                                                         os.path.basename( sys.argv[0] )  
                                                         ))
        transaction = self._getTransaction( self.READ )
        email =  self._getEmail( transaction )
        self._close( transaction )
        return email
            
    
    def _getEmail( self , transaction ):

        try:
            email = transaction.email
        except AttributeError:
            return None
        return email


    def setEmail( self, email ):
        """
        set the user email in the session
        @param email: the user email
        @type email: a Mobyle.Net.Email instance
        """
        s_log.debug( "%f : %s : setEmail call by= %s || email = %s" %( time() ,
                                                                       self.getKey() , 
                                                                       os.path.basename( sys.argv[0] ) ,
                                                                       email.To ) )
      
        transaction = self._getTransaction( self.WRITE )
        if transaction.email == email.To :
            self._close( transaction )
        else:
            transaction.email = email.To
            self._modifiedTransaction = True
            self._commit( transaction )


    def addDatas( self , name  , mobyleType , content = None , producer = None , inputModes = [] , usedBy = None , producedBy = None):
        """
        @param name: the data name
        @type name: string
        @param content: the content of the data
        @type content: string
        @param Type: the type of the data
        @type Type: a MobyleType instance
        @param producer: where to find the data if content is None producer must be specify and vice & versa
        @type producer: a L{Job} , a L{MobyleJob} , or a L{Session} object
        @param inputModes: the source of data
        @type inputModes: a list of string among this values : 'db' , 'paste' , 'upload' , 'result'
        @param usedBy: jobID(s) which used this data as input
        @type usedBy: string or sequence of strings
        @param producedBy: jobID(s) which produced this data
        @type producedBy: string or sequence of strings   
        @return: the name of the data in this session
        @rtype: string
        """
        s_log.debug( "%f : %s : addDatas name = %s , mobyleType = %s , content = %s , producer = %s , inputModes = %s , usedBy = %s , producedBy = %s"%(  time() ,
                                                                                                                                                          self.getKey(),
                                                                                                                                                          name ,
                                                                                                                                                          mobyleType.getDataType().getName() ,
                                                                                                                                                          str( content )[0:10] ,
                                                                                                                                                          producer  , 
                                                                                                                                                          inputModes  , 
                                                                                                                                                          usedBy  ,
                                                                                                                                                          producedBy 
                                                                                                                                                          ) )
      
        if not mobyleType.isFile():
            raise MobyleError , "session/%s : Session could add only MobyleType which are file : %s" %( self.getKey() , mobyleType.getDataType().getName() )
        transaction = self._getTransaction( self.WRITE )
        try:
            s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@ addDatas @@@@@@@@@@@@@@@@@@@@@@" %( time() ,self.getKey() ))
            dataName = self._addDatas(transaction, name, mobyleType, content, producer, inputModes, usedBy, producedBy)
        except NoSpaceLeftError , err:
            #this error is already logged
            raise err
      
        except Exception , err:
            self._close(transaction)
          
            s_log.debug("%s : addDatas:" %(self.getKey() ) , exc_info = False  )
          
            #transaction = self._getTransaction( self.READ )
            #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@ etat de la transaction apres addDatas avec erreur @@@@@@@@@@@@@@@@@@@@@@\n" %( time() , self.getKey() ))
            #s_log.debug( "%f : %s : jobs = %s "%( time() , self.getKey() , transaction.jobs.keys() ))
            #s_log.debug( "%f : %s : datas = %s "%( time() , self.getKey() , transaction.datas.keys() ))
            #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@ etat du repertoire @@@@@@@@@@@@@@@@@@@@@@\n" %( time() , self.getKey() ))
            #s_log.debug( "%f : %s : %s" %( time() ,self.getKey() , os.listdir( self.Dir ) ))
            #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@ Fin addDatas @@@@@@@@@@@@@@@@@@@@@@" %(time() , self.getKey() ))
            #self._close(transaction)
          
            s_log.error( "session/%s addDatas: %s" %(self.getKey() , err ) )
            raise err

        #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@  etat de la transaction apres addData @@@@@@@@@@@@@@@@@@@@@@\n" %(time() , self.getKey() ))
        #s_log.debug( "%f : %s : jobs = %s "%( time() , self.getKey() , transaction.jobs.keys() ))
        #s_log.debug( "%f : %s : datas = %s "%(time() , self.getKey()  , transaction.datas.keys() ))      
        #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@ etat du repertoire @@@@@@@@@@@@@@@@@@@@@@\n " %(time() , self.getKey() ))
        #s_log.debug( "%f : %s : %s" %( time() , self.getKey() , os.listdir( self.Dir ) ))
        #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@ Fin addDatas @@@@@@@@@@@@@@@@@@@@@@" %(time() , self.getKey() )) 
      
        self._commit(transaction)
        return dataName
                  

    def _addDatas( self , transaction , name  , mobyleType , content = None , producer = None , inputModes = [] , usedBy = None , producedBy = None):
        s_log.debug(" _addDatas: transaction =%s , name=%s  , mobyleType= %s , content=  %s , producer= %s , inputModes= %s , usedBy= %s , producedBy= %s"%( transaction ,
                                                                                                                                                             name       , 
                                                                                                                                                             mobyleType.getDataType().getName() , 
                                                                                                                                                             str( content )[0:10]    , 
                                                                                                                                                             producer   , 
                                                                                                                                                             inputModes , 
                                                                                                                                                             usedBy     , 
                                                                                                                                                             producedBy 
                                                                                                                                                             ) )
        #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@    etat de la transaction  @@@@@@@@@@@@@@@@@@@@@@\n" %(time() , self.getKey() ))
        #s_log.debug( "%f : %s : jobs = %s "%( time() , self.getKey(), transaction.jobs.keys() ))
        #s_log.debug( "%f : %s : datas = %s "%( time() , self.getKey()  , transaction.datas.keys() ))
        #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@    etat du repertoire @@@@@@@@@@@@@@@@@@@@@@\n " %( time() , self.getKey() ))
        #s_log.debug( "%f : %s : %s" %( time() , self.getKey() , os.listdir( self.Dir ) ))    


        if content and producer :
            raise MobyleError , "you must specify either a content or a producer not the both"
        elif not ( content or producer ) :
            raise MobyleError , "you must specify either a content or a producer "

        userName = str( name )
        safeName = Mobyle.Utils.safeFileName( name )
        acceptedInputMode = ( 'db' , 'paste' , 'upload' , 'result' )
      
        if safeName == Session.FILENAME :
            #logger l'erreur dans adm et logs?
            raise MobyleError , "permission denied"
     
        if inputModes :   
            inputModesType = type( inputModes ) 
          
            if inputModesType == types.StringType:
                inputModes = [ inputModes ]
          
            for inputMode in inputModes :
                if inputMode not in acceptedInputMode :
                    raise MobyleError , "unknow source of data : " + inputMode
         
        if usedBy:
            jobType = type( usedBy )

            if  jobType== types.StringType :
                usedBy = [ usedBy ]
            elif jobType ==  types.ListType or jobType == types.TupleType:
                pass
            else:
                raise MobyleError
        else:
            usedBy = []
            
        if producedBy:
            jobType = type( producedBy )
            if  jobType == types.StringType :
                producedBy = [ producedBy ]
            elif jobType ==  types.ListType or jobType == types.TupleType:
                pass
            else:
                raise MobyleError
        else:
            producedBy = []      
       
        # to avoid md5 differences depending on the web client or platform
        # I clean the content prior to perform the md5
        # be careful the cleannig must be adapted to the parameter
        # it's not the same between text and binary parameter
 
        #################################################
        #  Attention Hack a fixer 
        #  si le MobyleType a un format alors convert ( appele par setValue )
        #  va convertir la donnee or on ne veux pas ca
        #  si il n'y a pas de format et qu'il n'y a pas de service convert ne convertit pas la donnee
        #  mais je suis oblige de faire une copy du Mt avant de mettre format a [] car c'est celui du paramettre du service
        # ( qui a ete passe par CGIjob) et donc sinon je modifie celui ci et il n' ya pas de conversion de sequence dans le job 
        #
        #  cela pose la question du Mt dans la sessionn que signifie accepted formats et card dans la session
        #  faut il les eliminer systematiquement?
        ###################################################       
       
        import copy
        mobyleType = copy.deepcopy( mobyleType )  
        param = Mobyle.Service.Parameter( mobyleType )
        param._mobyleType.acceptedFormats=[] #WARNING if acceptedFormats is not empty the data will be reformat by sequence converter
      
        if content :
            content = param.cleanData( content )
            newMd5 = md5.new()
            newMd5.update( content )
            newMd5.update( mobyleType.getDataType().getName() )
         
            nameInSession = newMd5.hexdigest()
         
            #the md5 is made on the content + the name of the datatype
            #this permit us to have the same content wit 2 datatype.
            #eg a fasta alignment couldbe used as alignment or sequence
         
            userExt = os.path.splitext( safeName )[1]
            dataMask = os.path.join( self.Dir , nameInSession + "*" )
            nameInSession = str( nameInSession + userExt )
            nameInProducer = None
            dataBegining = param.head( content )

        elif not isinstance( producer , Session ):
            #the producer is a jobState or MobyleJob ...
            #I must read the file to compute the md5

            #the name is Safe
            fh = producer.open( name )
            dataContent = ''
            l = fh.readline()
            while l:
                dataContent = dataContent + l
                l = fh.readline()

            fh.close()

            dataContent = param.cleanData( dataContent ) #dans les job la donne a deja ete nettoye
            dataBegining = param.head( dataContent )

            newMd5 = md5.new()
            newMd5.update( dataContent )
            newMd5.update( mobyleType.getDataType().getName() )
         
            nameInSession = newMd5.hexdigest()
            userExt = os.path.splitext( name )[1] #the last .ext
            dataMask = os.path.join( self.Dir , nameInSession + "*" )
            nameInSession = str( nameInSession + userExt )
            nameInProducer = name         
       
        elif producer:
            #producer is a Session
            #I copy the metadata about the data from the session source
            try:
                data = producer.getDatas( dataName = name )[0]
            except IndexError:
                raise MobyleError , "there is no data named : %s in session %s" %( name , producer.getKey() )
         
            nameInProducer = data[ 'dataName' ]
            safeName       = data[ 'userName' ]
            mobyleType     = data[ 'type' ]
            dataBegining   = data[ 'dataBegin' ]
            usedBy         = data[ 'usedBy' ]
            producedBy     = data[ 'producedBy' ]
            inputModes     = data[ 'inputModes' ]
            size           = data[ 'size' ]
            dataFormat     = data[ 'format' ]
            nameInSession  = str( nameInProducer )
         
            dataMask = os.path.join( self.Dir , nameInSession + "*" )
         
        #for all 
        absNameInSession = os.path.join( self.Dir , nameInSession )
        files_exists = glob.glob(  dataMask  )
        s_log.debug("%f : %s : files_exists( %s ) = %s" %( time() , self.getKey() , dataMask , files_exists ))        
       
        if files_exists : 
            #it's important to recalculate the name in session 
            #because the ext must be different ( upload / copypaste )
            #and the name return by file_exist is an basolute name and a local name in the data structure
            nameInSession = os.path.basename( files_exists[0] )
            try:       
                #transaction = self._getTransaction( self.WRITE )
                transaction.datas[ nameInSession ][ 'userName' ]
            except KeyError , err: #the data exist in directory but not in transaction object !
                msg = "inconsistency error between the session object and the directory"
                s_log.error("session/%s _addDatas: %s" %( self.getKey() , msg ) )
                s_log.error("session/%s _addDatas: %s" %( self.getKey() , err ))
                s_log.debug("%f : %s : _addDatas:" %( time() , self.getKey() ), exc_info = True )
                      
                #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@    etat de la transaction  @@@@@@@@@@@@@@@@@@@@@@\n" %( time() , self.getKey() ))    
                #s_log.debug( "%f : %s : jobs = %s "%( time() , self.getKey() , transaction.jobs.keys() ))
                #s_log.debug( "%f : %s : datas = %s "%(time() , self.getKey() , transaction.datas.keys() ))              
                #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@    etat du repertoire @@@@@@@@@@@@@@@@@@@@@@\n " %( time() , self.getKey() ))
                s_log.debug( "%f : %s : %s" %( time() , self.getKey() , os.listdir( self.Dir ) ))    
                try:
                    name2remove = os.path.join( self.Dir , nameInSession )
                    s_log.error("session/%s : remove the data %s before to add a new one " %( self.getKey() , name2remove ) )
                    os.unlink( name2remove )
                except IOError , err:
                    s_log.error("session/%s : the data already exist in session dir can't remove it before to add a new one : %s" %( self.getKey() , err ) )
                         
                #I add the data in transaction 
                self._createNewData( transaction ,
                                     param , 
                                     safeName , 
                                     content , 
                                     nameInSession , 
                                     producer , 
                                     nameInProducer , 
                                     mobyleType , 
                                     dataBegining , 
                                     producedBy = producedBy ,
                                     inputModes = inputModes )
 
                self._modifiedTransaction = True    
                return nameInSession 
                     
            #the data file exist in directory and in .session   
            if ( transaction.datas[ nameInSession ][ 'Type' ].getDataType().getName() != param.getDataType().getName() ):
                s_log.debug( "%f : %s : try to add data %s with DataType: %s but this data already exist with DataType =%s" %( time() ,
                                                                                                                               self.getKey() ,
                                                                                                                               param.getDataType().getName() ,
                                                                                                                               transaction.datas[ nameInSession ][ 'Type' ].getDataType().getName()
                                                                                                                               ) )
                raise MobyleError , "the data already exist with an other DataType"
              
              
            #mettre a jour la liste des jobs
            dataUsed2add = []
            dataProduced2add = []
            
            if usedBy :
                for jobID in usedBy:
                    if ( self.jobExists( jobID ) ) and ( jobID not in transaction.datas[ nameInSession ][ 'usedBy' ] ):
                        transaction.datas[ nameInSession ][ 'usedBy'].append( jobID )
                        dataUsed2add.append( nameInSession )
                        #jobID  , date = None , status = None , dataUsed = [] , dataProduced = [] 
            if producedBy :
                for jobID in producedBy :
                    if ( self.jobExists( jobID ) ) and ( jobID not in transaction.datas[ nameInSession ][ 'producedBy'] ):
                        transaction.datas[ nameInSession ][ 'producedBy'].append( jobID )
                        dataProduced2add.append( nameInSession )
            if inputModes:
                for inputMode in inputModes :
                    if inputMode not in transaction.datas[ nameInSession ][ 'inputModes' ]:
                        transaction.datas[ nameInSession ][ 'inputModes' ].append( inputMode )
                               
            #==================================================================
            # I can't include the update in previous loop 
            # because a transaction is open 
            # and updateJob open an other transaction
            #==================================================================
            if dataUsed2add: 
                self._updateJob( transaction , jobID , dataUsed = dataUsed2add )
          
            if dataProduced2add:
                self._updateJob( transaction , jobID , dataProduced = dataProduced2add )
           
            if not self._checkSpaceLeft():
                path = os.path.join( self.Dir , nameInSession )
                size = os.path.getsize( path )
                os.unlink( path )
                s_log.debug( "%f : %s : _addDatas unlink %s ( %d ) because there is no space in session"%( time() ,
                                                                                                           self.getKey(),
                                                                                                           nameInSession ,
                                                                                                           size
                                                                                                           ))
              
                s_log.error( "session/%s : the data %s ( %d ) cannot be added because the session size exceed the session limit ( %d )" %(
                                                                                                                                          self.getKey(),          
                                                                                                                                          nameInSession ,
                                                                                                                                          size ,
                                                                                                                                          self.sessionLimit
                                                                                                                                          ) )
             
                raise NoSpaceLeftError , "this data cannot be added to your bookmarks, because the resulting size exceed the limit ( %s )" %( Mobyle.Utils.sizeFormat( self.sessionLimit ) )
          
            self._modifiedTransaction =True
            return nameInSession 
     
        else: #this data doesn't exist in Session
            # the disk writing operation or link is made by setValue.
            # more precisely by _toFile called by convert called by setValue of this parameter

            self._createNewData( transaction ,
                                 param , 
                                 safeName , 
                                 content , 
                                 nameInSession , 
                                 producer , 
                                 nameInProducer , 
                                 mobyleType , 
                                 dataBegining , 
                                 usedBy = usedBy , 
                                 producedBy = producedBy ,
                                 inputModes = inputModes )
            
            self._modifiedTransaction =True    
            return nameInSession 
   
    def _createNewData(self , transaction , param , safeName , content , nameInSession , producer , nameInProducer , mobyleType , dataBegining , usedBy = None , producedBy = None ,  inputModes = None ):
        param.setName( safeName )
        s_log.debug( "param= %s ,safeName= %s , content= %s , nameInSession= %s , producer= %s , nameInProducer= %s , mobyleType= %s , dataBegining= %s , usedBy= %s , producedBy= %s , inputModes= %s " %( param , 
                                                                                                                                                                                                            safeName , 
                                                                                                                                                                                                            str( content )[0:10], 
                                                                                                                                                                                                            nameInSession , 
                                                                                                                                                                                                            producer , 
                                                                                                                                                                                                            nameInProducer , 
                                                                                                                                                                                                            mobyleType , 
                                                                                                                                                                                                            dataBegining , 
                                                                                                                                                                                                            usedBy  , 
                                                                                                                                                                                                            producedBy  ,  
                                                                                                                                                                                                            inputModes  )
        )
     
        param.setName( safeName )
        try:
            # the disk writing operation or link is made by setValue.
            # more precisely by _toFile called by convert called by setValue of this parameter

            param.setValue( ( content , self , nameInSession , producer , nameInProducer ) )
            dataFormat = param.getDataFormat()
        except UserValueError , err :
            file2remove = os.path.join( self.Dir , nameInSession )
            squizzDir = self.cfg.format_detector_cache_path()
            if squizzDir :
                try:
                    #dumper la donnee pour analyse 
                    os.link(   file2remove , os.path.join( squizzDir , mobyleType.getDataType().getName() + "_" + self.getKey() + "_" + nameInSession ) )
                except OSError , err2 : 
                    #s_log.error( "session/ %s / try to link %s in tmp/squizz/ : %s" %( self.getKey() , file2remove , err2  )  ) 
                    pass
            try:
                os.unlink( file2remove )
            except OSError , err3 :
                s_log.error( "session/%s : try to unlink %s from session : %s" %( self.getKey() ,
                                                                                  file2remove ,
                                                                                  err2 
                                                                                  )  )            
       
            s_log.debug( str( err ) )             
            raise UserValueError , err                         
       
        except MobyleError ,err : #qu'est ce qui reste comme erreur ?
            s_log.error("_createNewData : " , exc_info = True  )
        
            try:
                os.unlink( os.path.join( self.Dir , nameInSession ) )
            except OSError , err2 :
                pass

            raise MobyleError , err

        nameInSession = param.getValue( )
            
        if not self._checkSpaceLeft():
            path = os.path.join( self.Dir , nameInSession )
            size = os.path.getsize( path )
            os.unlink( path )
            s_log.debug( "%f : %s : _addDatas unlink %s ( %d ) because there is no space in session"%( time() ,
                                                                                                       self.getKey(),
                                                                                                       nameInSession ,
                                                                                                       size
                                                                                                       ))
            s_log.error( "session/%s : the data %s ( %d ) cannot be added because the session size exceed the session limit ( %d )" %(
                                                                                                                                      self.getKey() ,           
                                                                                                                                      nameInSession ,
                                                                                                                                      size ,
                                                                                                                                      self.sessionLimit
                                                                                                                                      ) )
         
            raise NoSpaceLeftError , "this data cannot be added to your bookmarks, because the resulting size exceed the limit ( %s )" % Mobyle.Utils.sizeFormat( self.sessionLimit )
                                                                                                                                 
     
        transaction.datas[ nameInSession ] = {}
        transaction.datas[ nameInSession ][ 'userName' ]   =  safeName
        transaction.datas[ nameInSession ][ 'Type' ]       =  mobyleType
        transaction.datas[ nameInSession ][ 'dataBeg' ]    =  dataBegining
        if usedBy:
            transaction.datas[ nameInSession ][ 'usedBy' ]  =  usedBy
        else:
            transaction.datas[ nameInSession ][ 'usedBy' ]   = []
            usedBy = []

        if producedBy:
            transaction.datas[ nameInSession ][ 'producedBy' ] = producedBy
        else:
            transaction.datas[ nameInSession ][ 'producedBy' ] = []
            producedBy = []
        
        if inputModes:
            try:
                transaction.datas[ nameInSession ][ 'inputModes' ] += inputModes
            except KeyError:
                transaction.datas[ nameInSession ][ 'inputModes' ] = inputModes
        else:
            transaction.datas[ nameInSession ][ 'inputModes' ] = []
            inputModes = []
        
        for jobID in usedBy:
            try:
                self._updateJob( transaction , jobID , dataUsed = nameInSession  )
            except KeyError :
                self._addJob( transaction , jobID, dataUsed = nameInSession )
        for jobID in producedBy:
            try:
                self._updateJob(transaction , jobID , dataProduced = nameInSession )
            except KeyError:
                self._addJob(transaction , jobID, dataUsed = nameInSession )
         
        transaction.datas[ nameInSession ][ 'size' ] = os.path.getsize( os.path.join( self.Dir , nameInSession ) )             
       
        if dataFormat:
            transaction.datas[ nameInSession ][ 'format' ] = dataFormat
        else:
            transaction.datas[ nameInSession ][ 'format' ] = None
             
        self._modifiedTransaction = True    
        return nameInSession        
       

    def _checkSpaceLeft( self ):
        sessionSize = 0
        for f in os.listdir( self.Dir ):
            sessionSize += os.path.getsize( os.path.join( self.Dir  , f ) )
            if sessionSize > self.sessionLimit:  
                s_log.debug( "%f : %s : _checkSpaceLeft call by= %s sizefound = %d" %( time() ,
                                                                                       self.getKey()  , 
                                                                                       os.path.basename( sys.argv[0] ) ,
                                                                                       sessionSize
                                                                                       )) 
                return False
        return True       
       
       
    def removeData( self , dataName ):
        """
        @param dataName: the name of one data in this session
        @type dataName:string
        """  
        s_log.debug( "%f : %s : removeDatas call by= %s || dataName = %s " %( time() ,
                                                                              self.getKey()  , 
                                                                              os.path.basename( sys.argv[0] ) ,
                                                                              dataName
                                                                              )) 
        
        transaction = self._getTransaction( self.WRITE ) 
        try:
            s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@  removeData  @@@@@@@@@@@@@@@@@@@@@@"%(time() , self.getKey() )) 
            self._removeData(transaction, dataName)
        except Exception , err:
            self._close(transaction)
          
            s_log.error("%s/removeData : %s" %(self.getKey() , err ) )
            s_log.debug("%s : removeData :" %(self.getKey()) , exc_info = False )
          
            #transaction = self._getTransaction( self.READ )
            #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@    etat de la transaction apres removeData avec erreur @@@@@@@@@@@@@@@@@@@@@@\n "%( time() , self.getKey() ))
            #s_log.debug( "%f : %s : jobs = %s "%( time() , self.getKey(), transaction.jobs.keys() ))
            #s_log.debug( "%f : %s : datas = %s "%( time(), self.getKey()  , transaction.datas.keys() ))              
            #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@    etat du repertoire @@@@@@@@@@@@@@@@@@@@@@\n " %( time() ,self.getKey() ))
            #s_log.debug( "%f : %s : %s" %( self.getKey() , os.listdir( self.Dir ) ) )         
            #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@    fin removeData  @@@@@@@@@@@@@@@@@@@@@@" %(time() , self.getKey() )) 
          
            #self._close(transaction)
            raise err

        #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@    etat de la transaction apres removeData @@@@@@@@@@@@@@@@@@@@@@\n " %(time(), self.getKey()) )
        #s_log.debug( "%f : %s : jobs = %s "%( time() , self.getKey(), transaction.jobs.keys() ) )
        #s_log.debug( "%f : %s : datas = %s "%(time() , self.getKey()  , transaction.datas.keys() ) ) 
        #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@    etat du repertoire @@@@@@@@@@@@@@@@@@@@@@\n " %(time() , self.getKey() ))
        #s_log.debug( "%f : %s : %s" %( time(),self.getKey() , os.listdir( self.Dir ) ) )     
        #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@    fin removeData  @@@@@@@@@@@@@@@@@@@@@@" % ( time() , self.getKey() )) 
        self._commit(transaction)       
  
       
    def _removeData( self , transaction , dataName ):
        """
        @param dataName: the name of one data in this session
        @type dataName:string
        """
        #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@    etat de la transaction  @@@@@@@@@@@@@@@@@@@@@@\n " % (time(), self.getKey() ))
        #s_log.debug( "%f : %s : jobs = %s "%( time() , self.getKey(), transaction.jobs.keys() ) )
        #s_log.debug( "%f : %s : datas = %s "%(time() ,self.getKey()  , transaction.datas.keys() ) )      
        #s_log.debug( "%f : %s : @@@@@@@@@@@@@@@@@@@@@@    etat du repertoire @@@@@@@@@@@@@@@@@@@@@@\n " %( time(), self.getKey() ))
        #s_log.debug( "%f : %s : %s" %( time() , self.getKey() , os.listdir( self.Dir ) ) )
      
        if dataName == Session.FILENAME :
            s_log.error( "session/%s : can't remove file Session.FILENAME" %( self.getKey() ) )
            raise MobyleError, "permission denied"

        fileName = os.path.join( self.Dir , dataName  )

        if os.path.exists( fileName ):
            try:
                os.unlink( fileName )
            except OSError , err:
                msg = str( err )
                s_log.critical( "session/%s : can't remove data : %s" %( self.key , msg ) )
                raise MobyleError , msg

        try:
            del( transaction.datas[ dataName ] )
            self._modifiedTransaction = True
        except KeyError:
            pass       
       
       
       
    def renameData( self , dataName  , newUserName ):
        """
          @param dataName: the name of the data in the session ( md5 name )
          @type dataName: string
          @param newUserNAme: the new user name of this data
          @type newUserName: string
        """
        s_log.debug( "%f : %s : renameData call by= %s || dataName = %s newUserName = %s" %( time(),
                                                                                             self.getKey()  , 
                                                                                             os.path.basename( sys.argv[0] ) ,
                                                                                             dataName ,
                                                                                             newUserName
                                                                                             ))    
      
        transaction = self._getTransaction( self.WRITE )
        try:
            data = transaction.datas[ dataName ][ 'userName' ] = newUserName
            self._modifiedTransaction = True
        except KeyError:
            self._close( transaction )
            raise KeyError , "There is no Data with name %s in session %s" %( dataName , self.getKey() )

        self._commit( transaction )       
       
     
       
    def getContentData( self , dataName , forceFull = False ):
        """
          @param md5Name: the name of the data in the session.
          @type md5Name: string
          @return: the head or full content of the data
          @retype: tuple ( string 'HEAD'/'FULL' , string content )
          @raise MobyleError: if md5Name doesn't match any data in session
        """
        s_log.debug( "%f : %s : getContentData call by= %s || dataName = %s" %( time(),
                                                                                self.getKey()  , 
                                                                                os.path.basename( sys.argv[0] ) ,
                                                                                dataName
                                                                                ))    
      
        maxSize = self.cfg.previewDataLimit()
        filename = os.path.join( self.Dir , dataName  )
      
        if not os.path.exists( filename ):
            transaction = self._getTransaction( self.READ )
            try:
                data = transaction.datas[ dataName ]
                self._close( transaction )     
            except KeyError:
                #the data is neither in the session directory nor the session data structure 
                raise MobyleError, "%s  No such data in session: %s" %( self.getKey() , dataName )

            #the data is in the session directory But not in the session data structure
            s_log.critical( "session/%s is corrupted. The data %s is in directory But not in data structure" % ( self.getKey(),
                                                                                                                 dataName
                                                                                                                 )
                                                                                                                 )
            raise SessionError, "%s No such data in session: %s" %( self.getKey() , dataName )       
      
        fh = open( filename , 'r' )
        if forceFull:
            content = ''.join( fh.readlines() )
            flag = 'FULL'
        else:
            dataSize = os.path.getsize( filename )
            if dataSize > maxSize :
                content = fh.read( maxSize )
                flag = 'HEAD'
            else:
                content = ''.join( fh.readlines() )
                flag = 'FULL'            
      
        fh.close()
        return ( flag , content )
  
  
    def open( self , dataName ):
        """
        @param dataName: the name of the data in the session.
        @type dataName: string
        @return: a file object
        @retype: 
        @raise MobyleError: if dataName doesn't match any data in session
        """
        s_log.debug( "%f : %s  open dataName = %s" %( time(),
                                                      self.getKey() ,
                                                      dataName 
                                                      ))
        try:
            fh = open( os.path.join( self.Dir , dataName ), 'r' )
        except IOError, err:
            #verifier si la donnee est toujours dans la session
            s_log.error( "session/%s : open : %s" %( self.getKey() , err ) )
            raise MobyleError ,err  
  
  
    def getUserFileName( self, dataName ):
        """
        @param dataName: the file name of the data in the session ( md5 name)
        @type dataName: string
        @return: the user name of the data
        @rtype: string
        @raise: KeyError if dataName doesn't match with any data in session
        """
        s_log.debug( "%f : %s : getUserFileName call by= %s || dataName = %s" %( time(),
                                                                                 self.getKey()  , 
                                                                                 os.path.basename( sys.argv[0] ) ,
                                                                                 dataName
                                                                                 ))        

        try:
            transaction = self._getTransaction( self.READ )    
            datas = transaction.datas[ str( dataName ) ]      
            self._close( transaction )
            return  transaction.datas[ dataName ][ 'userName' ]  
        except KeyError , err:
            self._close( transaction )
            msg = "the Session %s has no data named : %s " %( self.getKey() , dataName )
            s_log.error( msg )
            raise KeyError , msg
  
  
    def getDataSize(self , dataName ):
        #I don't use the size sotred in transaction.datas[ 'size' ] 
        #to avoid to put a lock on the .session file
        dataPath =  os.path.join( self.Dir , dataName )
        if os.path.exists(  dataPath ):
            return os.path.getsize( dataPath )
        else:
            return None  
  
  
    def getDatas( self, dataName = None ):
        """
        @param dataName: the ID of the data in the session
        @type dataName: string or sequence of string
        @return: the data in the session which have dataName, if dataName is None return all data in session 
        @rtype: list of dict
                 [   {  'dataName'  : string ,
                        'userName'  : string ,                
                        'type'      : ( string  , string )
                        'dataBegin' : string , 
                        'inputModes': list of strings ,
                        'producedBy': list of strings ,
                         'usedBy'   : list of strings 
                    } , ... 
                 ]   
        """
        s_log.debug( "%f : %s : getDatas call by= %s || dataName = %s " %( time() ,
                                                                           self.getKey() , 
                                                                           os.path.basename( sys.argv[0] ) , 
                                                                           dataName ,
                                                                        ))

 
        transaction = self._getTransaction( self.READ )
        datas = self._getDatas( transaction , dataName = dataName  )
        self._close(transaction)
        return datas  

  

    def _getDatas( self, transaction , dataName = None ):
        """
        @param dataName: the ID of the data in the session
        @type dataName: string or sequence of string
        @return: the data in the session which have dataName, if dataName is None return all data in session 
        @rtype: list of dict
             [   {  'dataName'  : string ,
                    'userName'  : string ,                
                    'type'      : ( string  , string )
                    'dataBegin' : string , 
                    'inputModes': list of strings ,
                    'producedBy': list of strings ,
                     'usedBy'   : list of strings 
                } , ... 
             ]   
        """
        results= []
        dataNameType = type( dataName )

        if dataName is None:
            dataNames = transaction.datas.keys()
        elif dataNameType == types.StringType:
            dataNames = [ dataName ]
        elif dataNameType == types.ListType  or dataNameType == types.TupleType:
            pass
        else:
            raise Error
        for dataName in dataNames:
            try:
                results.append( { 'dataName'  : dataName ,
                                 'userName'  : transaction.datas[ dataName ][ 'userName' ] ,
                                 'type'      : transaction.datas[ dataName ][ 'Type' ] ,
                                 'dataBegin' : transaction.datas[ dataName ][ 'dataBeg' ] ,
                                 'usedBy'    : transaction.datas[ dataName ][ 'usedBy' ] ,
                                 'producedBy': transaction.datas[ dataName ][ 'producedBy' ] ,
                                 'inputModes': transaction.datas[ dataName ][ 'inputModes' ] ,
                                 'format'    : transaction.datas[ dataName ][ 'format' ],
                                 'size'      : transaction.datas[ dataName ][ 'size'] 
                                 } 
                                 )
            except KeyError :
                pass
        return results       
  

  
  

    def __extractServiceName( self ,job ):
        """
        @param job: 
        @type job: string
        @return: the service name for a job
        @rtype string
        """
        s = job.split('/')
        if s[-1] == '' or s[-1] == 'index.xml':
            return s[ -3  ]
        else:
            return s[ -2 ]
        
        
    def hasJob(self , jobID):
        s_log.debug( "%f : %s : hasJobcall by= %s || jobID = %s " %( time() ,
                                                                       self.getKey() ,
                                                                       os.path.basename( sys.argv[0] ) ,
                                                                       jobID 
                                                                       ) )
        transaction = self._getTransaction( self.READ )
        res = transaction.jobs.has_key( jobID )
        self._close( transaction) 
        return res
    
    def getJobs( self , jobID = None ):
        """
        @return: the list of jobs 
        @rtype: list of dictionary = [  { 'jobID'       : string ,
                                          'userName'    : string , 
                                          'programName' : string ,
                                          'date'        : time.time_struct ,
                                          'status'      : string ,
                                          'dataUsed'    : [ md5name ] ,
                                          'dataProduced : [ md5name ] ,
                                        } , ...
                                    ]
        """
        s_log.debug( "%f : %s : getJobs call by= %s || jobID = %s " %( time() ,
                                                                       self.getKey() ,
                                                                       os.path.basename( sys.argv[0] ) ,
                                                                       jobID 
                                                                       ) )
        results = []
        transaction = self._getTransaction( self.WRITE )

        self._updateJobStatus( transaction , jobID = jobID )
      
        if jobID is None:
            jobIDs = transaction.jobs.keys()
        else:
            jobIDs = [ jobID ]    
        
        for jobID in jobIDs:
         
            if self.jobExists( jobID ):
                try:
                    userName     = transaction.jobs[ jobID ][ 'userName' ]
                    programName  = transaction.jobs[ jobID ][ 'programName' ]
                    date         = transaction.jobs[ jobID ][ 'date' ]
                    status       = transaction.jobs[ jobID ][ 'status' ]
                    dataUsed     = transaction.jobs[ jobID ][ 'dataUsed' ]
                    dataProduced = transaction.jobs[ jobID ][ 'dataProduced' ]
                except KeyError :
                    raise KeyError , "the session %s , have no job with ID : %s" %( self.getKey() , jobID ) 

            if status is not None:
                status = Mobyle.Utils.code2status[ status ]
                results.append(  { 'jobID'       :  jobID ,
                                  'userName'    :  userName ,
                                  'programName' :  programName ,
                                  'date'        :  date , 
                                  'status'      :  status , 
                                  'dataUsed'    :  dataUsed ,
                                  'dataProduced':  dataProduced 
                                  }
                                  )
            
            else: #when a job (directory) doesn't exists anymore I remove it's refs from session
                self._removeJob( transaction , jobID )
                self._modifiedTransaction = True
                results.append( { 'jobID'        : jobID ,
                                  'userName'    : None  ,
                                  'programName' : None  ,
                                  'date'        : None  , 
                                  'status'      : None  , 
                                  'dataUsed'    : None  ,
                                  'dataProduced': None  
                                  } ) 
            
        self._commit( transaction )
   
        return results                


    
    def renameJob( self , jobID  , newUserName ):
        """
        @param jobID: the name of the job in the session (url)
        @type jobID: string
        @param newUserNAme: the new user name of this job
        @type newUserName: string
        """
        s_log.debug( "%f : %s : renameJob call by= %s || dataName = %s newUserName = %s" %( time() ,
                                                                                            self.getKey()  , 
                                                                                            os.path.basename( sys.argv[0] ) ,
                                                                                            dataName ,
                                                                                            newUserName
                                                                                            ))    
      
        transaction = self._getTransaction( self.WRITE )
        try:
            data = transaction.jobss[ dataName ][ 'userName' ] = newUserName
            self._modifiedTransaction = True
        except KeyError:
            self._close( transaction )
            raise KeyError , "There is no job with ID %s in session %s" %( jobID , self.getKey() )

        self._commit( transaction )    
   
    
    def addJob( self, jobID  , userName = None , date = None , status = None , dataUsed = [] , dataProduced = [] ):
        """
        add a job in session if jobID already exist update its status and data used by this job
        if job doesn't already exist it's set the sessionKey in job index and job .admin to this session key 
        @param jobID: the ID (url) of a job 
        @type jobID: string
        @param date: the submission date of this job
        @type date: time.struct_time
        @param status: the status code of the job (status code are defined in {Mobyle.Utils})
        @type status: int
        @param dataUsed: the name of data used by this job ( md5Name )
        @type dataUsed: string or sequence of strings
        @param dataProduced: the name of data produced by this job ( md5Name )
        @param dataProduced: string or sequence of strings
        """
        s_log.debug( "%f : %s : addJob call by= %s || jobID = %s , date = %s ,status = %s , dataUsed = %s , dataProduced = %s " %( time(),
                                                                                                                                self.getKey()  , 
                                                                                                                                os.path.basename( sys.argv[0] ) ,
                                                                                                                                jobID ,
                                                                                                                                date ,
                                                                                                                                status ,
                                                                                                                                dataUsed ,
                                                                                                                                dataProduced 
                                                                                                                                ))

        transaction = self._getTransaction( self.WRITE )
        try:
            self._addJob( transaction , jobID  , date = date , status = status , dataUsed = dataUsed , dataProduced = dataProduced )
        except MobyleError ,err :
            self._close(transaction)
            raise err
         
        self._commit( transaction )        
    
    
    
    
    def _addJob( self, transaction , jobID  , userName = None , date = None , status = None , dataUsed = [] , dataProduced = [] ):
        """
        add a job in session if jobID already exist update its status and data used by this job
        if job doesn't already exist it's set the sessionKey in job index and job .admin to this session key 
        @param jobID: the ID (url) of a job 
        @type jobID: string
        @param date: the submission date of this job
        @type date: time.struct_time
        @param status: the status code of the job (status code are defined in {Mobyle.Utils})
        @type status: int
        @param dataUsed: the name of data used by this job ( md5Name )
        @type dataUsed: string or sequence of strings
        @param dataProduced: the name of data produced by this job ( md5Name )
        @param dataProduced: string or sequence of strings
        """
     
        dataUsedType = type( dataUsed )
     
        if dataUsedType == types.ListType or  dataUsedType == types.TupleType :
            pass
        elif dataUsedType == types.StringType :
            dataUsed = [ dataUsed ]
        else:
            raise MobyleError , "dataUsed must be a sequence of string or a string"
    
        dataProducedType = type( dataProduced )
     
        if  dataProducedType==  types.ListType or  dataProducedType == types.TupleType:
            pass
        elif  dataProducedType == types.StringType :
            dataProduced = [ dataProduced ]
        else:
            raise MobyleError , "dataProduced must be a sequence of string or a string"
        
        if isinstance( status ,  types.StringTypes ):#StringTypes test string and unicode !
            status = Mobyle.Utils.status2code[ status ]

        programName = self.__extractServiceName( jobID )

        #transaction = self._getTransaction( self.WRITE )
     
        if transaction.jobs.has_key( jobID ): #update an already existing job
            if userName:
                transaction.jobs[ jobID ][ 'userName' ] = userName
                self._modifiedTransaction = True
                
            if status :
                transaction.jobs[ jobID ][ 'status' ] = status
                self._modifiedTransaction = True
            if date :
                transaction.jobs[ jobID ][ 'date' ] = date
                self._modifiedTransaction = True
             
            for data in dataUsed:
                if data not in transaction.jobs[ jobID ][ 'dataUsed' ] :                  
                    transaction.jobs[ jobID ][ 'dataUsed' ].append( data )
                    self._modifiedTransaction = True
                 
            for data in dataProduced:
                if data not in transaction.jobs[ jobID ][ 'dataProduced' ] :                  
                    transaction.jobs[ jobID ][ 'dataProduced' ].append( data )
                    self._modifiedTransaction = True 
                    
        else: #create a new job
            state = Mobyle.JobState.JobState( jobID )
            if date is None:  
                try:
                    date = state.getDate() 
                except MobyleError:
                    date = None
                date = strptime(  date , "%x  %X")     
         
            if userName is None :
                userName = jobID
                 
            transaction.jobs[ jobID ] = {
                                         'userName'    : userName ,  
                                         'programName' : programName , 
                                         'date'        : date ,
                                         'status'      : status ,
                                         'dataUsed'    : dataUsed ,
                                         'dataProduced': dataProduced
                                         }
            self._modifiedTransaction = True
         
            sessionKey = self.getKey()                      
            #state.setSessionKey( sessionKey )
            #state.commit()
            #jobpath = Mobyle.JobState.url2path( jobID )               
            #admin = Mobyle.Utils.Admin( jobpath )
            #admin.setSession( sessionKey )
            #admin.commit()    

        for data in dataUsed: #update  self.__datas for internal coherence
            jobs = transaction.datas[ data ][ 'usedBy' ]  
        
            if jobID not  in jobs:
                jobs.append( jobID )
                self._modifiedTransaction =True
           
        for data in dataProduced:
            jobs = transaction.datas[ data ] [ 'producedBy']
            self._modifiedTransaction =True
        
            if jobID not in jobs:
                jobs.append( jobID )
                self._modifiedTransaction =True
            
    
    def updateJob( self, jobID  , userName = None , date = None , status = None , dataUsed = [] , dataProduced = [] ):
        """
        update an already existinig job in this session. 
        @param jobID: the ID (url) of a job 
        @type jobID: string
        @param date: the date of job submission
        @type date: time.time_struct
        @param status: the status code of the job (status code are defined in {Mobyle.Utils})
        @type status: int
        @param dataUsed: the name(s) of data used by this job ( md5Name )
        @type dataUsed: string or sequence of strings
        @param dataProduced: the name(s) of data produced by this job ( md5Name )
        @type dataProduced: string or sequence of strings
        """
        s_log.debug( "%f : %s : updateJob call by= %s || jobID = %s , date = %s ,status = %s , dataUsed = %s , dataProduced = %s " %( time(),
                                                                                                                                      self.getKey()  , 
                                                                                                                                      os.path.basename( sys.argv[0] ) ,
                                                                                                                                      jobID ,
                                                                                                                                      date ,
                                                                                                                                      status ,
                                                                                                                                      dataUsed ,
                                                                                                                                      dataProduced ,
                                                                                                                                      ))
     
     
        transaction = self._getTransaction( self.WRITE )
     
        try:
            self._updateJob( transaction , jobID , userName = userName , date = date , status = status , dataUsed = dataUsed , dataProduced = dataProduced )
        except KeyError, err :
            self._close( transaction )
            raise KeyError , "this session %s have no job with ID: %s" %( self.getKey() , jobID )
        except MobyleError, err :
            self._close( transaction )
            raise err
     
        self._commit( transaction )
    
    
    def _updateJob( self, transaction , jobID , userName = None , date = None , status = None , dataUsed = [] , dataProduced = [] ):
        """
        update an already existinig job in this session. 
        @param jobID: the ID (url) of a job 
        @type jobID: string
        @param date: the date of job submission
        @type date: time.time_struct
        @param status: the status code of the job (status code are defined in {Mobyle.Utils})
        @type status: int
        @param dataUsed: the name(s) of data used by this job ( md5Name )
        @type dataUsed: string or sequence of strings
        @param dataProduced: the name(s) of data produced by this job ( md5Name )
        @type dataProduced: string or sequence of strings
        """

        if isinstance( dataUsed , types.ListType) or  isinstance( dataUsed , types.TupleType ):
            pass
        elif isinstance( dataUsed , types.StringTypes ):
            dataUsed = [ dataUsed ]
        else:
            raise MobyleError , "dataUsed must be a sequence of string or a string"

        dataProducedType = type( dataProduced )
     
        if isinstance( dataProduced , types.ListType ) or isinstance( dataProduced , types.TupleType ) :
            pass
        elif isinstance( dataProduced ,types.StringTypes ) :
            dataProduced = [ dataProduced ]
        else:
            raise MobyleError , "dataProduced must be a sequence of string or a string"

        if isinstance( status , types.StringTypes ) :
            status = Mobyle.Utils.status2code[ status ]
    
        if transaction.jobs.has_key( jobID ):
            #programName , oldDate , oldstatus , datasInJob = transaction.jobs[ jobID ]
            if userName and userName != transaction.jobs[ jobID ][ 'userName' ]:
                transaction.jobs[ jobID ][ 'userName' ] = userName
                self._modifiedTransaction = True
            
            if date and date != transaction.jobs[ jobID ][ 'date' ]:
                transaction.jobs[ jobID ][ 'date' ] = date
                self._modifiedTransaction = True
           
            if status and status != transaction.jobs[ jobID ][ 'status' ]:
                transaction.jobs[ jobID ][ 'status'] = status
                self._modifiedTransaction = True
           
            for oneDataName in dataUsed :
                if oneDataName not in  transaction.jobs[ jobID ][ 'dataUsed' ]:
                    transaction.jobs[ jobID ][ 'dataUsed' ].append( oneDataName )
                    self._modifiedTransaction = True
              
                jobsInDatas = transaction.datas[ oneDataName ][ 'usedBy' ]  

                if jobID not in jobsInDatas:
                    jobsInDatas.append( jobID )
                    self._modifiedTransaction = True
                 
            for oneDataName in dataProduced :
                if oneDataName not in  transaction.jobs[ jobID ][ 'dataProduced' ]:
                    transaction.jobs[ jobID ][ 'dataProduced' ].append( oneDataName )
                    self._modifiedTransaction = True
              
                jobsInDatas = transaction.datas[ oneDataName ][ 'producedBy' ]  

                if jobID not in jobsInDatas:
                    jobsInDatas.append( jobID )
                    self._modifiedTransaction = True                                 
        else:    
            raise KeyError , "this session %s have no job with ID: %s" %( self.getKey() , jobID )


    
    def updateJobStatus( self , jobID = None , status = None ):
        """
        update the status of a job. If jobID is not specify, update all job recorded in session with status. if status is not specified request dynamically the status of the job 
        param jobID:the Id(s) of a job (url)
        type jobID: string or sequence of strings
        param status: the status of a job. If status is not specify update status according the batch used
        type status: string
        """
        s_log.debug( "%f : %s : updateJobStatus call by= %s  || jobID = %s, status = %s " %( time(),
                                                                                             self.getKey()  , 
                                                                                             os.path.basename( sys.argv[0] ) ,
                                                                                             jobID ,
                                                                                             status ,
                                                                                             ))
     
      
        transaction = self._getTransaction( self.WRITE )
        try:
            self._updateJobStatus(  transaction , jobID = jobID , status = status )
        except ( TypeError , KeyError , MobyleError ) , err:
            self._close( transaction )
            s_log.error( "session/%s : error during updateJobStatus : %s" %( self.getKey() , err) )
            raise err
        self._commit( transaction )
    
    
    
    def _updateJobStatus( self , transaction , jobID = None , status = None ):
        """
        update the status of a job. If jobID is not specify, update all job recorded in session with status. if status is not specified request dynamically the status of the job 
        param jobID:the Id(s) of a job (url)
        type jobID: string or sequence of strings
        param status: the status of a job. If status is not specify update status according the batch used
        type status: string
        """
        jobIDType = type( jobID )
      
        if jobID is None:
            jobIDs = transaction.jobs.keys()
        elif jobIDType == types.StringType :
            jobIDs= [ jobID ]
        elif jobIDType == types.ListType or jobIDType == types.TupleType :
            jobIDs = jobID
        else:
            raise TypeError, "updateJobStatus accept None , string or sequence as jobID : provide %s , %s"%( jobID , jobIDType )

        newStatus = status

        for jobID in jobIDs :    
            try:
                if status is None :
                    if self.jobExists( jobID ):
                        if transaction.jobs[jobID]['status'] not in [4,5,6]: 
                            try:
                                s_log.debug("updating job status for %s" % jobID )
                                j = JobFacade.getFromJobId(jobID)
                                result = j.getStatus()
                                newStatus  , msg = result['status'], result['msg']
                            except KeyError, ke:
                                s_log.error("Could not update status for job %s" % (jobID))
                                newStatus = transaction.jobs[jobID]['status']
                            except (MobyleError , NotImplementedError):
                                newStatus = transaction.jobs[jobID]['status']
                        else: # if job is finished, in error, or killed, we do not
                            # try to update its status
                            newStatus = transaction.jobs[jobID]['status']                      
                    else:    
                        self._removeJob( transaction , jobID  )
                        self._modifiedTransaction =True
                        continue
                        # retourne une erreur ??
    
                if type( newStatus ) in types.StringTypes :
                    try:
                        newStatus = Mobyle.Utils.status2code[ newStatus ]
                    except KeyError:
                        raise MobyleError , "invalid status value : " + str( newStatus )
                elif type( newStatus ) == types.IntType :
                    if not Mobyle.Utils.code2status.has_key( newStatus ) :
                        raise MobyleError , "invalid status value : " + str( newStatus )
                else:
                    raise MobyleError , "invalid status value : " + str( newStatus ) + " for job " + jobID
           
                if transaction.jobs[ jobID ][ 'status' ] != newStatus:
                    transaction.jobs[ jobID ][ 'status' ] =  newStatus      
                    self._modifiedTransaction =True 
            
            except (urllib2.HTTPError, urllib2.URLError):
                # should be modified: maybe a new status (unreachable) could be applied if justified
                # 404 => the job has been removed
                # otherwise => its hosting server is unreachable
                pass    
    

    def removeJob(self , jobID ):
        """
        remove job from Session (in jobs and datas )and ONLY in session
        @param job:
        @type job:
        @todo: to be tested
        """
        s_log.debug( "%f : %s : removeJob call by= %s || jobID = %s" %( time(),
                                                                        self.getKey()  , 
                                                                        os.path.basename( sys.argv[0] ) ,
                                                                        jobID
                                                                        ))        
        protocol , host , path , a,b,c = urlparse.urlparse( jobID  )
        if protocol != "http" :
            jobID = Mobyle.JobState.path2url( jobID )
        transaction = self._getTransaction( self.WRITE )    
        self._removeJob ( transaction , jobID  )  
        self._commit( transaction )
 
 
           
    def _removeJob( self , transaction , jobID ):
        """
        remove job from Session (in jobs and datas )and ONLY in session
        @param job:
        @type job:
        @todo: to be tested
        """

        protocol , host , path , a,b,c = urlparse.urlparse( jobID  )
        if protocol != "http" :
            jobID = Mobyle.JobState.path2url( jobID )
        
        try:
            dataUsed = transaction.jobs[ jobID ][ 'dataUsed' ]
        except KeyError:
            dataUsed = []
     
        try:
            dataProduced = transaction.jobs[ jobID ][ 'dataProduced' ]
        except KeyError:
            dataProduced = []
     
        #status = transaction.jobs[ jobID ][ 'status' ]
        if self.jobExists( jobID ):
            try:
                status = Mobyle.Utils.getStatus( jobID )[0]
            except NotImplementedError  , err :
                s_log.error( "session/%s removeJob can't kill distant job = %s : %s"%( self.getKey()  ,
                                                                                       jobID ,
                                                                                       err        
                                                                                       ) )    
    
            except MobyleError , err:
                s_log.error( "session/%s removeJob can't kill job = %s : %s"%( self.getKey()  ,
                                                                               jobID ,
                                                                                err
                                                                                ) )
            else:
                if status not in [ 'killed' , 'finished' , 'error' , 'building' ]:
                    try:
                        Mobyle.Utils.killJob( jobID )
                    except MobyleError , err:
                        s_log.error( "session/%s removeJob can't kill job = %s : %s"%( self.getKey()  ,
                                                                                       jobID ,
                                                                                        err
                                                                                        ) )
        else:
            pass   
    
        try:    
            del( transaction.jobs[ jobID ] )
            self._modifiedTransaction =True 
        except KeyError:
            pass
     
        #remove the jobID from the transaction.datas
        for dataName in dataUsed:
            try: 
                jobsInDatas = transaction.datas[ dataName ][ 'usedBy' ]
            except KeyError:
                s_log.debug( "%f : %s : _removeJob la data (%s) reference dans un job n'existe pas dans data" %( time(),
                                                                                                                 self.getKey() ,
                                                                                                                 dataName 
                                                                                                                 ) )
                if os.path.exists( os.path.join( self.Dir , dataName ) ):
                    s_log.debug( "%f : %s : _removeJob cette data existe dans la session"%(time(),
                                                                                           self.getKey() 
                                                                                           ))
                else:
                    s_log.debug( "%f : %s : _removeJob cette data  n'existe pas dans la session"%( time(),
                                                                                                   self.getKey() 
                                                                                                   ) )
                continue            

            for i in range( len( jobsInDatas ) ) :
                if jobsInDatas[ i ] == jobID:
                    del( jobsInDatas[ i ] ) #could be used like that, just once !
                    self._modifiedTransaction =True
                break 
    
        for dataName in dataProduced:
            try: 
                jobsInDatas = transaction.datas[ dataName ][ 'producedBy' ]
            except KeyError:
                s_log.debug( "%f : %s : _removeJob la data (%s) reference dans un job n'existe pas dans data" %( time(),
                                                                                                                 self.getKey() ,
                                                                                                                 dataName 
                                                                                                                 ) )
                if os.path.exists( os.path.join( self.Dir , dataName ) ):
                    s_log.debug( "%f : %s : _removeJob cette data existe dans la session"%(time(),
                                                                                           self.getKey() 
                                                                                           ))
                else:
                    s_log.debug( "%f : %s : _removeJob cette data  n'existe pas dans la session"%( time() ,
                                                                                                   self.getKey() 
                                                                                                   ))
                continue
            
            for i in range( len( jobsInDatas ) ) :
                if jobsInDatas[ i ] == jobID:
                    del( jobsInDatas[ i ] ) #could be used like that, just once !
                    self._modifiedTransaction =True
                    break    
    
    def jobExists( self , jobID ):
        server = registry.getServerByJobId(jobID)
        if server is None :
            return False
        if server.name=='local':
            try:
                jobPath = Mobyle.JobState.url2path( jobID )
            except MobyleError:
                return False
            return os.path.exists( jobPath )
        else: #this jobID correspond to a distant job
            try:
                protocol , host , path , a,b,c = urlparse.urlparse( jobID )
                cn = HTTPConnection(host)
                cn.connect()
                cn.request( "HEAD" , jobID + "/index.xml")
                return cn.getresponse().status == 200
            except Exception, e:
                s_log.error(e)
                return False    
   
   
   
    
class AuthenticatedSession( Session ):
   
    def __init__( self , cfg , email , passwd ):
        self.cfg = cfg
        """the maximun size of a session ( in bytes )"""
        self.sessionLimit = self.cfg.sessionlimit()
        self.__email = email.To
        authenticatedSessionAllowed = self.cfg.authenticatedSession()        
       
        if authenticatedSessionAllowed == "no" :
            s_log.error("can't create  session AUTHENTICATED_SESSION is set to \"no\" in Local/Config/Config.py")          
            raise SessionError , "can't create  authenticated session: permission denied"
              
        key = self.__newSessionKey()
        sessionDir = os.path.normpath( os.path.join( self.cfg.user_sessions_path() , "authentified" , key  ) )
        self.key = key 
        """ the user/session  key"""
        self.Dir = sessionDir
        """ the path to this session directory """
        self._modifiedTransaction = False
       
        if os.path.exists( sessionDir ): #the session already exist
            if not self.checkPasswd( passwd ):
                s_log.info( "authentified/%s : Authentication Failure "%( self.getKey() ) )
                raise AuthenticationError , "Authentication Failure"
               
        else: #creation of new Session
            chk = email.check()
            if not chk:
                msg = email.getMessage()
                s_log.error( "authenticated session creation for \"%s\" aborted: %s" %( email.To , msg ) )
                raise SessionError , "%s not allowed on this server" % email.To
           
            os.makedirs( sessionDir , 0755 ) #create parent directory 
            transaction = Transaction()
            transaction.email = self.__email
            
            mymd5 = md5.new()
            mymd5.update( passwd )
            cryptPasswd = mymd5.hexdigest()
            transaction.passwd = cryptPasswd        
            
            authenticatedSessionAllowed = self.cfg.authenticatedSession()        

            if authenticatedSessionAllowed == "yes":            
                transaction.activated = True 
    
            elif authenticatedSessionAllowed == "email" :
                transaction.activatingKey = self.__newActivatingKey()
                serverName = self.cfg.root_url()
                cgi_url = self.cfg.cgi_url()
                msg = """You have requested an account on the %s Mobyle server.
Your activation key is %s. To activate this account please click on the
following link (or paste its URL in your favourite browser):
%s/%s/portal.py?actkey=%s""" %( serverName ,
                                transaction.activatingKey ,
                                serverName , 
                                cgi_url ,
                                transaction.activatingKey 
                                )
                   
                subject =  "[ Mobyle server ( %s ) ] - new user confirmation" % serverName
               
                try:
                    email.sendMsg( subject , msg ) #from mailHelp
                    #transaction.activated = False   by default 
                except MobyleError , err :
                    msg = "can't send an activation email to \"%s\". session creation aborted: %s" % ( email.To , err)
                    s_log.error( msg )
                    os.rmdir( self.Dir )
                    raise SessionError , msg
               
            transactionFileName = os.path.normpath( os.path.join( self.Dir , Session.FILENAME  ) )
            transactionFile = open( transactionFileName , 'w' )
            #############################################################
            doc = self._struct2xml( transaction )
            Ft.Xml.Domlette.PrettyPrint( doc , transactionFile )
            #pickle.dump( transaction , transactionFile )
            #############################################################
            transactionFile.close()
            s_log.debug( "%f : %s  create a new session call by= %s " %( time(),
                                                                         self.getKey()  , 
                                                                         os.path.basename( sys.argv[0] ) ,
                                                                         ))
         
    def isAuthenticated( self ):
        return True   
    
    def setPasswd( self , passwd ):
        """
        set the pass word for this session
        @param passwd: the pass word to this session
        @type passwd: string
        """
        s_log.debug( "%f : %s setPasswd call by= %s" %( time() ,
                                                        self.getKey()  , 
                                                        os.path.basename( sys.argv[0] ) ,
                                                        ))         
        transaction = self._getTransaction( Session.WRITE )
        newMd5 = md5.new()
        newMd5.update( passwd )
        transaction.passwd = newMd5.hexdigest()
        self._commit( transaction )    
    
    def checkPasswd( self , passwd ):
        """
        check if passwd is the passwd of this session
        @param passwd: the session pass word
        @type passwd: string
        """ 
        s_log.debug( "%f : %s : checkPasswd call by= %s" %( time(),
                                                          self.getKey()  , 
                                                          os.path.basename( sys.argv[0] ) ,
                                                          ))        
        transaction = self._getTransaction( Session.READ )
        realPasswd = transaction.passwd
        self._close( transaction )
        
        newMd5 = md5.new()
        newMd5.update( passwd )
        passwd = newMd5.hexdigest()
        if passwd == realPasswd :
            return True
        else:
            return False    
  
    
    def confirmEmail( self , activatingKey ):
        """
        if the activatingkey match the session activatingkey the session is activated
        """ 
        s_log.debug( "%f : %s confirmEmail call by= %s" %( time() ,
                                                           self.getKey()  , 
                                                           os.path.basename( sys.argv[0] ) ,
                                                          ))              
        transaction = self._getTransaction( Session.WRITE )
        
        if transaction.activatingKey == activatingKey:
            transaction.activated = True
            self._modifiedTransaction = True
            self._commit( transaction )
               
        else:
            self._close( transaction )
            s_log.info("authentified/%s : wrong key : %s" %( self.getKey() , activatingKey ) )
            raise AuthenticationError , "wrong key : %s" %activatingKey
        
        s_log.debug("%f : %s : confirmEmail succeed : %s" %( time(), self.getKey() , activatingKey ) )     
    
    
    def __newSessionKey( self ):
        """
        @return: a unique id for the session
        @rtype: string
        """
        newMd5 = md5.new()
        newMd5.update( self.__email )
        return newMd5.hexdigest()    
    
    
    def __newActivatingKey(self):
        """Build a new Session uniq activating key"""
        t1 = time()
        sleep( random.random() )
        t2 = time()
        base = md5.new( str(t1 +t2) )
        sid = base.hexdigest()
        return sid    
    
    
    def mergeWith( self , anonymousSession ):
        """
        merge the anonymous session to this authenticated session
        @param anonymousSession: the session we add this session
        @type anonymousSession: L{AnonymousSession} object
        """
        s_log.debug( "%f : %s mergeWith AnonymousSession %s call by= %s" %( time() ,
                                                                            self.getKey()  , 
                                                                            anonymousSession.getKey() ,
                                                                            os.path.basename( sys.argv[0] ) ,
                                                                            ))  
        if anonymousSession == self:
            s_log.error( "authentified/%s try to merge with myself" %self.getKey() )
            raise SessionError , "try to merge with myself"   
          
        anonymousSessionKey = anonymousSession.getKey()
        transaction = self._getTransaction( Session.WRITE )
        try:
            for data in anonymousSession.getDatas():
                self._addDatas( transaction ,
                                data['dataName'] ,  
                                data['type'] , 
                                producer   = anonymousSession , 
                                inputModes = data['inputModes'] , 
                                usedBy     = data['usedBy'] , 
                                producedBy = data['producedBy']
                                )
           
            for job in anonymousSession.getJobs():
                self._addJob( transaction ,
                              job[ 'jobID' ] ,
                              date         = job[ 'date' ] ,
                              status       = job[ 'status' ] ,
                              dataUsed     = job[ 'dataUsed' ] ,
                              dataProduced = job[ 'dataProduced' ] 
                              )
        except Exception , err :
            self._close( transaction )
            s_log.error( "authentified/%s : error during mergeWith : %s"%( self.getKey() , err ) )
            s_log.debug("%f : %s : error during mergeWith :" %( time(), self.getKey() ) , exc_info = True )
            raise err
        self._modifiedTransaction = True
        self._commit( transaction )
 
      
      
      
class AnonymousSession( Session ):
  
    
    def __init__( self ,  cfg  , key = None):
        self.cfg = cfg
        """the maximun size of a session ( in bytes )"""
        self.sessionLimit = self.cfg.sessionlimit()
        anonymousSessionAllowed = self.cfg.anonymousSession()
        self._modifiedTransaction = False
       
        if anonymousSessionAllowed== 'no':
            s_log.error("can't create anonymous session ANONYMOUS_SESSION is set to \"no\" in Local/Config/Config.py")          
            raise MobyleError , "can't create anonymous session: permission denied"

        if key :
            self.Dir = os.path.normpath( os.path.join( self.cfg.user_sessions_path() , "anonymous" , key ) )
            if not os.path.exists( self.Dir ):
                s_log.error( "can't retrieve anonymous session, the Key: %s doesn't match with any Session" % key )
                raise SessionError , "wrong key : %s" % key

            self.key = key
            s_log.debug( "%f : %s return new annonymousSession based on old dir call by= %s" %( time() ,
                                                                                                self.getKey()  , 
                                                                                                os.path.basename( sys.argv[0] ) ,
                                                                                                ))
              
        else: #create a new session
            self.key = self.__newSessionKey( )
            """ the user/session  key"""
            self.Dir = os.path.normpath( os.path.join( self.cfg.user_sessions_path() , "anonymous" , self.key ) )
           
            if os.path.exists( self.Dir ):
                msg = "Try to make a new anonymous session with key: %s. This directory already exist" % self.key
                s_log.critical( msg )
                raise SessionError , "can't create new anonymous session"

            os.makedirs( self.Dir , 0755 ) #create parent directory
            """ the path to this session directory """
            transaction = Transaction()

            if anonymousSessionAllowed == 'yes':
                transaction.activated = True
                
            transactionFileName = os.path.normpath(  os.path.join( self.Dir , Session.FILENAME  )  )
            transactionFile = open( transactionFileName , 'w' )
            ##########################################################
            doc = self._struct2xml( transaction )
            Ft.Xml.Domlette.PrettyPrint( doc , transactionFile )
            #pickle.dump( transaction , transactionFile )
            ##########################################################
            transactionFile.close()
           
            s_log.debug( "%f : %s  create a new session call by= %s" %( time() ,
                                                                        self.getKey()  , 
                                                                        os.path.basename( sys.argv[0] ) ,
                                                                        ))
                
    def isAuthenticated( self ):
        return False
   
    def getCaptchaProblem( self ):
        """
        @return: a png image wich is a captcha
        @rtype:
        """
        s_log.debug( "%f : %s getCaptchaProblem call by= %s" %( time() ,
                                                                self.getKey()  , 
                                                                os.path.basename( sys.argv[0] ) ,
                                                                ))       
        from Captcha.Visual.Tests import PseudoGimpy
        import StringIO
        captcha = PseudoGimpy()
       
        transaction = self._getTransaction( Session.WRITE )
        transaction.current_captcha_sol = captcha.solutions[0]
        self._modifiedTransaction = True
        self._commit( transaction )
        pf = StringIO.StringIO()
        captcha.render().save( pf , "PNG" )
       
        return pf.getvalue()
  
             
    def checkCaptchaSolution( self , solution ):
        """
        check if solution is the solution to the current problem
        @param solution: the solution to the current problem
        @type solution: string
        @return: True if solution is the solution of the current captcha problem, False otherwise
        @rtype: boolean
        """
        s_log.debug( "%f : %s getCaptchaProblem call by= %s" %( time(),
                                                                self.getKey()  , 
                                                                os.path.basename( sys.argv[0] ) ,
                                                                ))   
        transaction = self._getTransaction( Session.WRITE )
        current_captcha_sol = transaction.current_captcha_sol
       
        if current_captcha_sol is None :
            raise SessionError , "getCaptchaProblem before checkCaptchaSolution"
       
        if solution == current_captcha_sol :
            transaction.activated = True
            self._modifiedTransaction = True
            self._commit( transaction )
            return True
        else:
            transaction.activated = False
            self._modifiedTransaction = False
            self._close( transaction )
            return False
     
     
    def __newSessionKey( self ):
        """
        @return: a unique id for the session
        @rtype: string
        """
        letter = string.ascii_uppercase[ random.randrange( 0 , 26 ) ]
        strTime = "%.9f" % time()
        strTime = strTime[-9:]
        strPid = "%05d" %os.getpid()
        strPid = strPid.replace( '.' , '' )[ -5 : ]
        return letter + strPid + strTime


        ############################################################
        #                                                          #
        #                      Transaction                         #
        #                                                          #
        ############################################################
    
    

class Transaction( object ):
    """
    This class defines a session, that stores all the information
    about a user that should be persistent on the server
    @author: Herve Menager
    @organization: Institut Pasteur
    @contact:mobyle@pasteur.fr
    """
   
    def __init__( self ):
        self.email = ''
        self.passwd = None
        self.current_captcha_sol = False
        self.activatingKey = None
        self.activated = False
        self.datas = { }
        #               'userName'   : None ,    # the name of the data given to(/by) the user
        #               'Type'       : None ,    # the Mobyle Type of the data,it's a mobyleType instance
        #               'dataBeg'    : None ,    # the 50 first characters of the Data (if it's a Text)
        #               'usedBy'     : []   ,    # the list of jobs in wich this data as used as input
        #               'producedBy' : []   ,    # he list of jobs wich produce this data
        #               'inputModes' : []   ,    # the source of the data.
                                                 # The allowed values are : 'db' , 'paste' , 'upload' , 'result'
        #               'size'       : None ,    # the size of the data in bytes
        #               'format'     : None ,    # a tuple of 3 elements ( 'format name' ,'nb of item in this data' , 'prgr which has detected this format' )
     
        self.jobs = {}                                     
        #===============================================================================
        #    { jobID : {
        #               'userName'     ,  string
        #               'programName'  ,  string
        #               'date'         ,  time.struct_time
        #               'status        ,  int
        #               'dataUsed      ,  [ md5 ]
        #               'dataProduced  ,  [ md5 ]
        #              }
        #     }
        #===============================================================================
    
    
    
    
          
