########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.        #
#                                                                                      #
########################################################################################

"""
Run the command , do the choice of a synchron asynchron process.
in the case of asynchron process, manage the creation of the child process and the synchronization with it.
"""


import os ,os.path
import signal , siginterrupt
import sys
import time
import cPickle

import logging
rf_log = logging.getLogger('mobyle.runnerfather')

import Mobyle.Service
import Mobyle.CommandBuilder 
import Mobyle.JobState 
import Mobyle.ConfigManager
import Mobyle.Classes.DataType

from Mobyle.MobyleError import MobyleError , ServiceError , EmailError

__extra_epydoc_fields__ = [ ('call', 'Calledby','Called by') ]

#_cfg = Mobyle.ConfigManager.Config()


class CommandRunner:
    """

    """


    def __init__( self, job , jobState = None , synchron = False  ):
        """
        instanciate a L{CommandBuilder} and use it to build the CommandLine, then run it
        @param service:
        @type job: a L{Job} instance
        @param jobState:
        @type jobState:
        @param synchron: True if we want a synchron job execution, False for an asynchronus execution (default = False)
        @type synchron: Boolean
        @call: l{Job.run}
        @todo: implementation pour le warper de cgi ou WS a faire
        """
        self.job = job
        self._service = self.job.getService()
        self._adm = Mobyle.Utils.Admin( self.job.getDir() )
        self.synchron = synchron
        if jobState is None:
            self._jobState = Mobyle.JobState.JobState( os.getcwd() )
        else:
            self._jobState = jobState
        
        commandBuilder = Mobyle.CommandBuilder.CommandBuilder()
        
        method = self._service.getCommand()[1].upper()

        if method == '' or method == 'LOCAL':

            try:
                self._commandLine  , self._xmlEnv = commandBuilder.buildLocalCommand( self.job )
            except Exception ,err :
                msg = "an error occured during the command line building: " 

                self._logError( admMsg = msg + str( err ),
                                userMsg = "Mobyle Internal Server Error",
                                logMsg = None #the error log is filled by the rf_log.critical
                                )
                #this error is already log in build.log
                msg = "%s/%s : %s" %( self._service.getName() ,
                                                   self.job.getKey() , 
                                                   msg , 
                                                   )
                
                if self.job.cfg.debug( self._service.getName() ) == 0:
                    rf_log.critical( msg  , exc_info = True) # send an email
                else:
                    rf_log.error( msg , exc_info = True) 
                 
                raise MobyleError , "Mobyle Internal Server Error"

        elif method == "GET" or method == "POST" or method == "POSTM":
            raise NotImplementedError ,"cgi wrapping is not yet implemented"
        elif method == "SOAP" or method == "XML-RPC":
            raise NotImplementedError ,"cgi wrapping is not yet implemented"
        else:
            raise MobyleError, "unknown method of command : "+str( method )
        
        try:
            commandFile = open( ".command" , 'w' )
            commandFile.write( self._commandLine + "\n" )
            commandFile.close()
        except IOError , err:
            msg = "an error occured during the .command file creation: " + str( err )
            self._logError( admMsg = msg ,
                            userMsg= "Mobyle Internal Server error" ,
                            logMsg = msg
                            )
            raise MobyleError , err

        self._jobState.setCommandLine( self._commandLine )
        self._jobState.commit()


    def run(self):
        """
        instanciate a L{SynchronRunner} or an L{AsynchronRunner} in function of the attribute synchron.
        """
        if self.synchron:
            return SynchronRunner( self._commandLine , os.getcwd() , self.job , jobState = self._jobState , xmlEnv = self._xmlEnv )
        else:
            return AsynchronRunner( self._commandLine , os.getcwd() , self.job , jobState = self._jobState , xmlEnv = self._xmlEnv )

    def getCmdLine(self):
        return self._commandLine


    def _logError( self , admMsg = None , userMsg = None , logMsg = None ):

        if  admMsg :
            self._adm.setStatus( 5 , admMsg )
            self._adm.commit()


        if userMsg :
            self._jobState.setStatus( 5 , userMsg )
            self._jobState.commit()

        if logMsg :
            rf_log.error( "%s/%s : %s" %( self._service.getName() ,
                                          self.job.getKey() ,
                                          logMsg
                                          )
                          )



class SynchronRunner:
    """
    @todo: not yet implemented
    """
    
    def __init__( self, commandLine , dirPath , job , jobState = None , cfg = None ):
        """
        TODO
        """
        if cfg :
            self._cfg = cfg
        else:
            import Mobyle.ConfigManager
            self._cfg = Mobyle.ConfigManager.Config()
        
        self._command = commandLine
        self._dir = dirPath
        self._job = job
        raise  NotImplementedError , "SynchronRunner not yet impelemented"

 



class AsynchronRunner:
    """
    manage the child process creation
    manage the synchronisation with his child
    """
    

    def __init__( self, commandLine , dirPath , job , jobState = None , cfg = None , xmlEnv = None ):
        """
        the fathter process
        @param commandLine: the command to be executed
        @type commandLine: String
        @param dirPath: the absolute path to directory where the job will be executed (nomarly we are already in)
        @type dirPath: String
        @param service:
        @type service: A L{Service} instance
        @param jobState:
        @type jobState: a L{JobState} instance
        @type cfg: L{ConfigManager.Config} instance
        @call: L{CommandRunner.run}
        """
        if cfg :
            self._cfg = cfg
        else:
            import Mobyle.ConfigManager
            self._cfg = Mobyle.ConfigManager.Config()

        self._command = commandLine
        self._xmlEnv = xmlEnv
        self._dir = dirPath
        self._job = job
        self._service = self._job.getService()
        self._child_pid = 0
        self.child_done = False

        cwd = os.getcwd()
        try:
            self.timeout = self._cfg.timeout()
        except AttributeError:
            self.timeout = 60        

        if jobState is None:
            self.jobState = Mobyle.JobState.JobState( cwd )
        else:
            self.jobState = jobState

        self._adm = Mobyle.Utils.Admin( cwd )
                        
        try:
            self._child_pid = os.fork()
        except Exception , err:
            msg = "Can't fork : " + str( err )
            self._logError( admMsg = "AsynchronRunner.__init__ : " + msg ,
                            logMsg = msg ,
                            userMsg = "Mobyle Internal server error"
                            )

            raise MobyleError , msg
        
        
        if self._child_pid > 0 : ####### FATHER #######
            signal.signal(signal.SIGUSR1, self.sigFromChild)
            
            # encapsulated module to siginterrupt system.
            # see Mobyle/Utils/Siginterrupt/siginterrupt.c
            
            siginterrupt.siginterrupt( signal.SIGUSR1 , False )
            
            time.sleep( self.timeout )
            
            if self.child_done:
                # my timeout was interupt
                # the child process is finished
                # I do what shall I do
                pass
                
            else:
                # my timeout is over, the child process is still alive
                # I send a signal to my child that I quit
                try:
                    os.kill( self._child_pid , signal.SIGUSR2 )
                except OSError , err :
                    msg = "Can't send a signal to my child process:" + str( err )
                    self._logError( admMsg = "AsynchronRunner.__init__ : " + msg ,
                                    logMsg = msg ,
                                    userMsg = "Mobyle Internal server error"
                                    )
                    raise MobyleError , msg
           

                email = self._job.getEmail()
                jobName = self._service.getName()
                jobKey = self._job.getKey()
                jobID = self.jobState.getID()
                portalUrl = "%s/%s/portal.py" %( self._cfg.root_url() ,
                                                 self._cfg.cgi_url()
                                                 )
                job_url_in_portal = "%s?jobs=%s" %( portalUrl ,
                                                    jobID
                                                    )
                if email is not None:
                    subject = 'your job %s/%s is running' %( jobName , jobKey )
                    msg = """\n
    Your %s/%s job is running on %s server.
    You'll receive the results by email.
    you could access to the results or check the job status at the following adress:
    %s""" %( jobName , 
             jobKey,
             portalUrl ,
             job_url_in_portal
             )
                    try:
                        email.sendMsg( subject , msg )
                    except EmailError , err :
                        rf_log.error( "%s/%s : can not sent job notification : %s" %( jobName ,
                                                                                      jobKey ,
                                                                                      err
                                                                                      )
                                                                                          )                       
                
        elif self._child_pid == 0 : ####### CHILD #######
            os.setsid()
            devnull = os.open( "/dev/null" , os.O_RDWR )
            os.dup2( devnull , sys.stdin.fileno() )
            
            serviceName = self._service.getName()
            
            try:
                #logfile = os.open(  os.path.join( self._cfg.log_dir(), 'debug' ) , os.O_CREAT | os.O_WRONLY | os.O_TRUNC )                
                logfile = os.open( os.path.join( self._cfg.log_dir(), 'child_log' ) , os.O_APPEND | os.O_WRONLY | os.O_CREAT )                
                            
                os.dup2( logfile , sys.stdout.fileno() )
                os.dup2( logfile , sys.stderr.fileno() )
                os.close( logfile )

                
            except ( IOError , OSError ) , err :
                rf_log.critical( "error in redirecting stderr or stdout to debug : ", exc_info = True ) 
                
                self._logError( admMsg = "AsynchronRunner.__init__ : error in redirecting stderr or stdout in child process: " ,
                                logMsg = None ,
                                userMsg = "Mobyle Internal server error"
                                )
                
                try :
                    os.dup2( devnull , sys.stdout.fileno() )
                    os.dup2( devnull , sys.stderr.fileno() )
                except (IOError , OSError ) , err:
                    rf_log.critical( "error in redirecting stderr or stdout to /dev/null : " , exc_info = True ) 
                    
                
            childName = os.path.join( os.environ[ 'MOBYLEHOME' ] ,
                                      'Src' ,
                                      'Mobyle' ,
                                      'RunnerChild.py'
                                      )


            email = self._job.getEmail()
            
            if email is None:
                email_to = None
            else:
                email_to = email.To

            try:
                paramsOut = self._service.getAllOutParameter(  )
                results_mask ={}
                evaluator = self._service.getEvaluator()
                stdout = False
                allClasses = dir( Mobyle.Classes )

                for paramName in paramsOut :
                    try:
                        lang = self._cfg.lang()
                        prompt = self._service.getPrompt( paramName , lang = lang )
                        paramPrompt = ( prompt , lang )
                    except ServiceError:
                        paramPrompt = ( None , None )
                        
                    if self._service.precondHas_proglang( paramName , 'python' ):
                        preconds = self._service.getPreconds( paramName , proglang='python' )
                        allPrecondTrue = True
                       
                        for precond in preconds:
                            
                            if not evaluator.eval( precond ) :
                                allPrecondTrue = False
                                break
                        
                        if not allPrecondTrue :
                            continue #next parameter

                    if self._service.isstdout( paramName ):
                        stdout = True

                    mobyleType = self._service.getType( paramName )
                    unixMasks = self._service.getFilenames( paramName , proglang = 'python' )
                    results_mask [ paramName ] = ( paramPrompt , mobyleType  , unixMasks )
                    
                dataTypeFactory = Mobyle.Classes.DataType.DataTypeFactory()
                serviceName = self._service.getName()
                
                if not stdout :
                    
                    stdoutDataType = dataTypeFactory.newDataType( 'Text' )
                    stdoutType = Mobyle.Service.MobyleType(  stdoutDataType )
                    results_mask [ 'stdout' ] = ( ( 'standard output' , None ) , stdoutType , [ serviceName + '.out' ] )
                
                stderrDataType = dataTypeFactory.newDataType( "Text" )
                stderrType = Mobyle.Service.MobyleType( stderrDataType )
                results_mask [ 'stderr' ] = ( ( 'errors and warnings from ' + serviceName , None ) , stderrType , [ serviceName + '.err' ] )
                
         
            
            
            except MobyleError, err:
                msg = "AsynchronRunner.__init__ : " + str( err )
                userMsg = "Mobyle Internal Server Error"
                self._logError( admMsg = msg ,
                                logMsg = msg ,
                                userMsg = "Mobyle Internal server error"
                                )
                raise MobyleError , msg
                
            forChild = { 'serviceName' : self._service.getName() ,
                         'email'       : email_to ,
                         'dirPath'     : self._dir,
                         'commandLine' : self._command ,
                         'resultsMask' : results_mask ,
                         'xmlEnv'      : self._xmlEnv
                         }

            try:
                fh = open(".forChild.dump",'w')
                cPickle.dump( forChild , fh )
                fh.close()
            except IOError , err:
                pass
            
            cmd = [ childName ]
            
            logging.shutdown() #close all loggers
            try:
                os.execv( cmd[0] , cmd )
            except Exception, err:
                MobyleLogger.MLogger() #I re-open the loggers to send this msg
                msg = "AsynchronRunner.__init__: exec child caught an error: " + str(err)

                self._logError( admMsg = msg ,
                                logMsg = msg ,
                                userMsg = "Mobyle Internal server error"
                                )
                
                raise MobyleError, msg
            
        
        
    def sigFromChild( self, signum ,frame ):
        """
        @call: when the child is terminated and send a SIGUSR1 to the father process
        """
        self.child_done = True
        signal.signal(signal.SIGUSR1, signal.SIG_IGN)





    def _logError( self , admMsg = None , userMsg = None , logMsg = None ):

        if  admMsg :
            self._adm.setStatus( 5 , admMsg )
            self._adm.commit()

        if userMsg :
            self.jobState.setStatus( 5 , userMsg )
            self.jobState.commit()

        if logMsg :
            rf_log.error( "%s/%s : %s" %( self._service.getName() ,
                                          self._job.getKey() ,
                                          logMsg
                                          )
                        )
    


