########################################################################################
#                                                                                      #
#   Author: Herve Menager                                                              #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.         #
#                                                                                      #
########################################################################################
"""
Mobyle.JobFacade

This module provides an simplified and seamless access to
 - local jobs and
 - remote jobs
"""
import time
import Mobyle.MobyleJob 
import Mobyle.Parser 
from Mobyle.MobyleError import *
from Mobyle.Registry import registry
import Mobyle.Utils
import urllib
import urllib2
import simplejson

import logging
from Mobyle import MobyleLogger
MobyleLogger.MLogger()
j_log = logging.getLogger('mobyle.jobfacade' )


class JobFacade(object):
    """
    JobFacade is an abstract class that is an access point to a Mobyle Job.
    """

    def addJobToSession(self, session, jobInfo):
        """
        links the job to a session by:
         - adding the job to the session
         - adding the inFile data to the session
         - getting user file name from the session
        @param session: the user "session"
        @type session: Mobyle.Session
        @param jobInfo: a dictionary containing job information such as id, date, etc.
        @type: dictionary
        """
        session.addJob( jobInfo['id'], date = \
                        time.strptime(jobInfo['date'], "%x %X" ), \
                        status=jobInfo['status'][0])
        dataUsed = []
        for param in [param for param in self.params.values() \
                      if (param.has_key('parameter') \
                      and param['parameter'].isInfile())]:
            if param['safeName'] and not(param['value']):
                param['userName'] = session.getUserFileName(param['safeName'])
                dataUsed.append(param['safeName'])
            if param['value'] and not(param['safeName']):
                param['safeName'] = \
                session.addDatas( param['userName'], \
                                  param['parameter'].getType(), \
                                  content = param['value'], \
                                  inputModes = [param['inputMode']])
        session.updateJob(jobID=jobInfo['id'],dataUsed=dataUsed)


    def __init__(self, serviceUrl, jobId=None):
        self.serviceUrl = serviceUrl
        self.serviceDef = registry.programsByUrl[self.serviceUrl]
        if jobId:
            self.jobId = jobId
        
    def _processForm(self):
        """
        process the cgi form:
         - preprocessing infiles, parsing user and safe file names
         - preprocessing boolean values, which have specific default values
         - create a parameter values dictionary which is used later
        """
        for paramName in self.service.getUserInputParameterByArgpos():
            param = self.service.getParameter(paramName)
            inputName = self.serviceUrl + '_' + paramName 
            # param names are prefixed with program name to avoid name 
            # collision between inputs that have the same parameter name
            value, userName, safeName, inputMode = None, None, None, None
            if param.isInfile():
                inputMode = self.request.getfirst('dataInputType_' + \
                                                 inputName,"upload")
                if self.request.has_key(inputName+"_sname"):
                    safeName = self.request.getfirst(inputName+"_sname")
                    userName = self.request.getfirst(inputName+"_uname")
                else:
                    value = self.request.getfirst(inputName+"_data")
                    userName = paramName + ".data"
            else:
                if (self.service.getDataType(paramName).getName()=='Boolean'): 
                    # for a boolean, no value = False...
                    value = self.request.getfirst(inputName, False)
                else:
                    value = self.request.getfirst(inputName, None) 
                    # for other parameter types, no value = set no value...
            if value is not None or safeName: # if the value is not null...
                self.params[paramName] = {'value': value, \
                                          'userName': userName , \
                                          'safeName': safeName, \
                                          'inputMode': inputMode, \
                                          'parameter': param}
  
    def create(self, request):
        """
        create sets up most of the class attributes TODO: check if this could be done in __init__?
        """
        self.request = request
        self.serviceUrl = self.request.getfirst('programName', None)
        parser = Mobyle.Parser.ServiceParser()
        self.service  = parser.parse(self.serviceUrl)
        self.email = self.request.getfirst('email')
        self.params = {}
        self._processForm()
  
    def getFromService(cls, serviceUrl):
        """
        create a new JobFacade from the service URL.
        @param serviceUrl: service URL
        @type session: string
        @return: the appropriate job facade
        @rtype: JobFacade
        """
        if registry.programsByUrl[serviceUrl].server.name=='local':
            return(LocalJobFacade(serviceUrl=serviceUrl))
        else:
            return(RemoteJobFacade(serviceUrl=serviceUrl))

    getFromService = classmethod(getFromService)
      
    def getFromJobId(cls, jobId):
        """
        create a JobFacade to access an existing job.
        @param jobId: the job identifier
        @type jobId: string
        @return: the appropriate job facade
        @rtype: JobFacade
        """
        mj = Mobyle.MobyleJob.MobyleJob(ID=jobId)
        programUrl = mj.jobState.getName()
        jobId = mj.getJobid() 
        # this id is identical to the one in parameter, 
        # except it has been normalized (may have removed
        # trailing index.xml from the id)
        if mj.isLocal():
            return(LocalJobFacade(serviceUrl=programUrl, jobId=jobId))
        else:
            return(RemoteJobFacade(serviceUrl=programUrl, jobId=jobId))
    getFromJobId = classmethod(getFromJobId)

class RemoteJobFacade(JobFacade):
    """
    RemoteJobFacade is a class that is an access point to a Mobyle Job on a remote server.
    """
  
    def submit(self, session=None):
        """
        submits the job on the remote server
        @param session: the session used to load infile values
        @type session: Mobyle.Session
        @return: job information as a dictionary
        @rtype: dictionary
        """
        endpointUrl = self.serviceDef.server.url + "/job_submit.py"
        values = self.request
        for param in [param for param in self.params.values() \
                      if (param.has_key('parameter') \
                          and param['parameter'].isInfile())]:
            if param['safeName'] and not(param['value']):
                param['value'] = session.getContentData(param['safeName'], \
                                                        forceFull = True)[1]
        requestDict = {}
        requestDict['programName'] = self.serviceUrl
        if self.email:
            requestDict['email'] = self.email
        for name, param in self.params.items():
            if (param.has_key('parameter') and param['parameter'].isInfile()):
                name += "_data"
            requestDict[self.serviceUrl + '_' + name] = param['value']
        try:
            response = self._remoteRequest(endpointUrl, requestDict)
        except:
            return {"errormsg":"A communication error happened during the \
                    submission of your job to the remote Portal"}
        return response
  
    def getStatus(self):
        """
        gets the job status on the remote server
        @return: job status information as a dictionary
        @rtype: dictionary
        """
        endpointUrl = self.serviceDef.server.url + "/job_status.py"
        requestDict = {}
        requestDict['jobId'] = self.jobId
        try:
            response = self._remoteRequest(endpointUrl, requestDict)
        except:
            return {"errormsg":"A communication error happened while \
                    asking job status to the remote Portal"}
        return response
    
    def _remoteRequest(self, url, requestDict):
        """
        encodes and executes the proper request on the remote server, with 
        proper error logging
        @param url: the target url
        @type url: string
        @param requestDict: the request parameters
        @type requestDict: dictionary
        @return: response as a dictionary
        @rtype: dictionary
        """
        data = urllib.urlencode(requestDict)
        req = urllib2.Request(url, data)
        try:
            handle = urllib2.urlopen(req)
        except (urllib2.HTTPError), e:
            j_log.error("Error during remote job communication for remote service %s \
             to %s, HTTP response code = %s" % (self.serviceUrl, url, e.code))
            raise e
        except (urllib2.URLError), e:
            j_log.error("Error during remote job communication for remote service %s \
             to %s, URL problem = %s" % (self.serviceUrl, url, e.reason))
            raise e
        try:
            jsonMap = simplejson.loads(handle.read())
        except ValueError, e:
            j_log.error("Error during remote job communication for remote service %s \
             to %s, bad response format: %s" % (self.serviceUrl, url, str(e)))
            raise e
        return jsonMap
        

class LocalJobFacade(JobFacade):
    """
    LocalJobFacade is a class that is an access point to a Mobyle Job on the local server.
    """

    def submit(self, session=None):
        """
        submits the job on the local server
        @param session: the session used to load infile values
        @type session: Mobyle.Session
        @return: job information as a dictionary
        @rtype: dictionary
        """
        try:
            if not(session) and not(self.serviceDef.isExported()): 
                return {'errormsg':"Unauthorized access to service %s, which \
                        is restricted to local use." % self.serviceUrl}
            self.job = Mobyle.MobyleJob.MobyleJob( service = self.service , \
                                                   email = self.email,
                                                   session = session)
            for name, param in self.params.items():
                if param['parameter'].isInfile():
                    if not(param['value']):
                        self.job.setValue(name, (param['userName'], None, \
                                                 session, param['safeName']))
                    else:
                        self.job.setValue(name, (name+'_data', param['value'], \
                                                 None, None))          
                else:
                    self.job.setValue(name, param['value'])
            # run Job
            self.job.run()
            return {
                    'id': str(self.job.getJobid()),
                    'date':str(time.strftime( "%x  %X", self.job.getDate())),
                    'status': self.job.getStatus(),
                    }
        except (UserValueError, MobyleError), e:
            if hasattr(e, 'message') and e.message:
                msg = {'errormsg':str(e.message)}
            else:
                msg = {'errormsg':str(e)}
            if hasattr(self, 'job') and self.job:
                jobId = self.job.getJobid()
                msg['id'] = str(jobId)
            if hasattr(e, 'param') and e.param:
                msg["errorparam"] = str(e.param)
            return msg            
  
    def getStatus(self):
        """
        gets the job status on the local server
        @return: job status information as a dictionary
        @rtype: dictionary
        """
        status  , msg = Mobyle.Utils.getStatus( self.jobId )
        return {
                'status': status,
                'msg': msg
                }