########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under LGPLv2 Licence. Please refer to the COPYING.LIB document.        #
#                                                                                      #
########################################################################################


import logging , logging.handlers
from string import upper

import sys , os , os.path
import re
import glob , copy 
import types

#append Mobyle Home to the search modules path
try:
    if ( os.path.join( os.environ[ 'MOBYLEHOME' ] ) ) not in sys.path:
        sys.path.append( os.environ[ 'MOBYLEHOME' ] )
    if ( os.path.join( os.environ['MOBYLEHOME'] , 'Src' ) ) not in sys.path:    
        sys.path.append( os.path.join( os.environ['MOBYLEHOME'] , 'Src' ) )

except KeyError:
    print >> sys.stderr , "the env vars MOBYLEHOME must be defined "



import Local.Config.Config
import Mobyle.MobyleError 

rc_log = logging.getLogger('mobyle.configManager')


class Config( object ):
    """
    this class is designed as the singleton pattern
    (ref: Programmtion Python , Tarek Ziade .p 483).
    there is only one instance at once.
    this class parse the file local/Config/Config.py
    raise error fix default values raise error if needed. 
    all other Mobyle classes must use this class to access
    to a configuration information. 
    """
    _ref = None

    def __new__( self ):
        
        if self._ref is None:
            self._ref = super( Config,self ).__new__( self )

            ###############################
            #
            #      logging
            #
            #############################


            self._dirs ={}           
            
            try:
                self._dirs['log'] = Local.Config.Config.LOGDIR
            except AttributeError:
                msg = "LOGDIR not found in  Local/Config/Config.py"
                self._dirs['log'] = '/var/log/Mobyle'
                self.log.warning( msg + ". It set to /var/log/Mobyle" )

            self.log = logging.getLogger('mobyle.config')
            self.log.propagate = False
            try:
                defaultHandler = logging.FileHandler(  os.path.join( self._dirs['log'] , 'error_log' ) ,
                                                       'a'
                                                       )
            except :
                print >> sys.stderr , " WARNING : can't access to logfile. Logs will redirect to /dev/null"
                defaultHandler = logging.FileHandler( '/dev/null' , 'a' )

            defaultFormatter = logging.Formatter(
                '%(name)-10s : %(levelname)-8s : L %(lineno)d : %(asctime)s : %(message)s' ,
                '%a, %d %b %Y %H:%M:%S'
                )

            defaultHandler.setLevel( logging.ERROR )
            defaultHandler.setFormatter( defaultFormatter )
            self.log.addHandler( defaultHandler )
    
            try:
                self._accounting = Local.Config.Config.ACCOUNTING
            except AttributeError:
                self.log.info( "ACCOUNTING not found in  Local/Config/Config.py, I set ACCOUNTING= False" )
                self._accounting = False


            #######################
            #
            #    debug
            #
            #######################

            try:
                self._debug = Local.Config.Config.DEBUG
            except AttributeError:
                self.log.info( "DEBUG not found in Local/Config/Config.py, I set DEBUG = 0" )
                self._debug = 0

            if self._debug < 0 or self._debug > 3:
                self.log.warning( "DEBUG must be >= 0 and <= 3 I found DEBUG =%i, I set DEBUG = 0" % self._debug ) 
                self._debug = 0

            try:
                self._particular_debug = Local.Config.Config.PARTICULAR_DEBUG
            except AttributeError:
                self.log.info( "PARTICULAR_DEBUG not found in Local/Config/Config.py" )
                self._particular_debug = {}


            #########################
            #
            #     dir
            #
            ########################

            try:
                #self._dirs ={}
                try:
                    self._dirs[ 'document_root' ] = os.environ[ 'DOCUMENT_ROOT' ]
                except KeyError:
                    try:
                        self._dirs[ 'document_root' ] = Local.Config.Config.DOCUMENT_ROOT
                    except KeyError:    
                        self._dirs[ 'document_root' ] = None
                        
                if os.path.exists( Local.Config.Config.PROGRAMS_PATH ): 
                    self._dirs[ 'programs_path' ] = os.path.realpath( Local.Config.Config.PROGRAMS_PATH )
                else:
                    msg = "PROGRAMS_PATH : " + str( Local.Config.Config.PROGRAMS_PATH ) + " no such directory."
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg                        

                if os.path.exists( Local.Config.Config.RESULTS_PATH ): 
                    self._dirs[ 'results_path' ] = os.path.realpath( Local.Config.Config.RESULTS_PATH )
                else:
                    msg = "RESULTS_PATH : " + str( Local.Config.Config.RESULTS_PATH ) + " no such directory."
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg

                if os.path.exists( Local.Config.Config.USER_SESSIONS_PATH ): 
                    self._dirs[ 'user_sessions_path' ] = os.path.realpath( Local.Config.Config.USER_SESSIONS_PATH )
                else:
                    msg = "USER_SESSIONS_PATH : " + str( Local.Config.Config.USER_SESSIONS_PATH ) + " no such directory."
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg

                if Local.Config.Config.PROGRAMS_URL[ 0 ] == '/' :    
                    self._dirs[ 'programs_url' ] = Local.Config.Config.PROGRAMS_URL[ 1 : ]
                else :
                    self._dirs[ 'programs_url' ] = Local.Config.Config.PROGRAMS_URL

                if Local.Config.Config.RESULTS_URL[ 0 ] == '/' :    
                    self._dirs[ 'results_url' ] = Local.Config.Config.RESULTS_URL[ 1 : ]
                else :
                    self._dirs[ 'results_url' ] = Local.Config.Config.RESULTS_URL

                if Local.Config.Config.SESSIONS_URL[ 0 ] == '/' :     
                    self._dirs[ 'sessions_url' ] = Local.Config.Config.SESSIONS_URL[ 1 : ]
                else:
                    self._dirs[ 'sessions_url' ] = Local.Config.Config.SESSIONS_URL


                self._dirs[ 'htdocs_url' ] = Local.Config.Config.MOBYLEROOT_HTDOCS_URL
                if len(self._dirs[ 'htdocs_url' ])>1: # For the / only case
                    if self._dirs[ 'htdocs_url' ][0] == os.path.sep :
                        self._dirs[ 'htdocs_url' ] = self._dirs[ 'htdocs_url' ][1:]
                    
                self._dirs[ 'cgi_url' ] = Local.Config.Config.MOBYLEROOT_CGI_URL
                if self._dirs[ 'cgi_url' ][0] == os.path.sep :
                    self._dirs[ 'cgi_url' ] = self._dirs[ '' ][1:]


            except AttributeError, err:
                msg = str(err).split()[-1] + "not found in Local/Config/Config.py"
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg

            try:
                
                if os.path.exists( Local.Config.Config.PROGRAMS_CACHE_PATH):
                    self._dirs[ 'programs_cache_path' ]= os.path.realpath( Local.Config.Config.PROGRAMS_CACHE_PATH )
                else:
                    msg = "PROGRAMS_CACHE_PATH : " + str( Local.Config.Config.PROGRAMS_CACHE_PATH ) + " no such directory."
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg
            except AttributeError, err:
                msg = "PROGRAMS_CACHE_PATH not found in Local/Config/Config.py /tmp will used."
                self._dirs[ 'programs_cache_path' ]= '/tmp'
                self.log.warning( msg )

            try:
                if not Local.Config.Config.ROOT_URL:
                    msg = 'ROOT_URL is found in Local/Config/Config.py but have no value'
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg    
                               
                if Local.Config.Config.ROOT_URL[ -1 ] == '/' :
                    self._dirs[ 'root_url' ] = Local.Config.Config.ROOT_URL[ 0 : -1 ]
                else:
                    self._dirs[ 'root_url' ] = Local.Config.Config.ROOT_URL
                if self._dirs[ 'root_url' ][ :7 ] != 'http://' :
                    self._dirs[ 'root_url' ] = 'http://'+ root_url
                    
            except ( AttributeError , NameError ) :
                try:
                    port = os.environ[ 'SERVER_PORT']
                    if port == '80':
                        port = ''
                    else:
                        port = ':' +  port 
                except KeyError:
                    port =''
                try:
                    serverName = os.environ[ 'SERVER_NAME']
                except KeyError:
                    msg = 'ROOT_URL not found in Local/Config/Config.py'
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg
                
                self._dirs[ 'root_url' ] = 'http://' + serverName + port

            try:
                binary_path = []
                for path in Local.Config.Config.BINARY_PATH:
                    if not os.path.exists( path ):
                       msg = "the path \"%s\" defined in BINARY_PATH doesn't exist"
                       self.log.warning( msg )
                    binary_path.append( path )
                       
                self._dirs[ 'binary_path' ] = binary_path
                
            except AttributeError:
                self.log.warning( "BINARY_PATH not found in Local/Config/Config.py the default web server path will be used" )

                self._dirs['binary_path' ] = []

            try:
                if os.path.exists( Local.Config.Config.FORMAT_DETECTOR_CACHE_PATH ):
                    self._dirs[ 'format_detector_cache_path' ] = os.path.realpath( Local.Config.Config.FORMAT_DETECTOR_CACHE_PATH )
                else:
                    self.log.warning("FORMAT_DETECTOR_CACHE_PATH defined in Local/Config/Config.py does not exist." )
                    self._dirs[ 'format_detector_cache_path' ] = None

            except AttributeError:
                self._dirs[ 'format_detector_cache_path' ] = None
                #self.log.info("FORMAT_DETECTOR_CACHE_PATH not found in Local/Config/Config.py." )                    

            try:
                self._databanks_config = Local.Config.Config.DATABANKS_CONFIG
            except AttributeError:
                self._databanks_config = []
                
            #######################
            #
            #  Mail
            #
            #######################

            try:
                if type( Local.Config.Config.MAINTAINER ) == types.ListType or type( Local.Config.Config.MAINTAINER ) == types.TupleType :
                    self._maintainer = Local.Config.Config.MAINTAINER
                else:
                    self._maintainer = [ Local.Config.Config.MAINTAINER ]
                    
                if type( Local.Config.Config.MAINTAINER ) == types.ListType or type( Local.Config.Config.MAINTAINER ) == types.TupleType :    
                    self._help = " , ".join( Local.Config.Config.MAINTAINER )
                else:
                    self._help = Local.Config.Config.HELP                    

                self._mailhost = Local.Config.Config.MAILHOST
            except AttributeError, err:
                msg = str(err).split()[-1] + "not found in Local/Config/Config.py"
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg

            try:
                self._sender = Local.Config.Config.SENDER
            except AttributeError, err:
                self._sender = self._help
                self.log.warning( "SENDER not found in  Local/Config/Config.py , I set SENDER = HELP ")
                
            try:
                self._dns_resolver = Local.Config.Config.DNS_RESOLVER
            except AttributeError:
                self.log.info( "DNS_RESOLVER not found in  Local/Config/Config.py, I set DNS_RESOLVER = False" )
                self._dns_resolver = False


            #######################
            #
            #     Authentication
            #
            #######################

            try:
                self._opt_email = Local.Config.Config.OPT_EMAIL
            except AttributeError:
                self.log.info( "OPT_EMAIL not found in  Local/Config/Config.py, I set OPT_EMAIL = True" )
                self._opt_email = True
                               
            try:
                self._particular_opt_email = Local.Config.Config.PARTICULAR_OPT_EMAIL
            except AttributeError:
                self.log.info( "PARTICULAR_OPT_EMAIL not found in  Local/Config/Config.py, use OPT_EMAIL for all services" )
                self._particular_opt_email = {}
                
            try:
                self._anonymous_session = Local.Config.Config.ANONYMOUS_SESSION
                
                try:
                    self._anonymous_session = self._anonymous_session.lower()
                except AttributeError:
                    self._anonymous_session = "captcha"
                    self.log.warning( "ANONYMOUS_SESSION have a wrong value: " + str( Local.Config.Config.ANONYMOUS_SESSION ) + " , set to \"captcha\"")
               
                if self._anonymous_session == "yes" or self._anonymous_session == "y":
                    self._anonymous_session = "yes"
                elif self._anonymous_session == "no" or self._anonymous_session == "n":
                    self._anonymous_session = "no"
                elif self._anonymous_session != "captcha":
                    self._anonymous_session = "captcha"
                    self.log.warning( "ANONYMOUS_SESSION have a wrong value: " + str( Local.Config.Config.ANONYMOUS_SESSION ) + " , set to \"captcha\"" )

            except AttributeError:
                self._anonymous_session = "captcha"
                self.log.info( "ANONYMOUS_SESSION not found in  Local/Config/Config.py, set to  default value:\"captcha\"" )
                
            
            try:
                self._authenticated_session = Local.Config.Config.AUTHENTICATED_SESSION

                try:
                    self._authenticated_session = self._authenticated_session.lower()
                except AttributeError:
                    self._authenticated_session = 'email'
                    self.log.warning( "AUTHENTICATED_SESSION have a wrong value: " + str( Local.Config.Config.AUTHENTICATED_SESSION ) + " , set to \"email\"" )
                    
                if self._authenticated_session not in ( "email" ,"no" ,"yes" ):
                    self._authenticated_session = "email"
                    self.log.warning( "AUTHENTICATED_SESSION have a wrong value: " + str( Local.Config.Config.AUTHENTICATED_SESSION ) + " , set to \"email\"" )
                        

                if self._anonymous_session == "no" and not self._authenticated_session :
                    msg = "anonymous session AND authenticated session are disabled. you can't disabled both"
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg
                
            except AttributeError , e :
                self._authenticated_session = 'email'
                self.log.info( "AUTHENTICATED_SESSION not found in  Local/Config/Config.py , set to default value: email")
                

            ########################
            #
            #      results
            #
            ########################

            try:
                self._dont_email_result = Local.Config.Config.DONT_EMAIL_RESULTS
            except AttributeError:
                self.log.info( "DONT_EMAIL_RESULTS not found in  Local/Config/Config.py, I set DONT_EMAIL_RESULTS = False")
                self._dont_email_result = False

            try:
                self._maxmailsize = Local.Config.Config.MAXMAILSIZE
            except AttributeError:
                if not self._dont_email_result :
                    self.log.info("MAXMAILSIZE not found in  Local/Config/Config.py but DONT_EMAIL_RESULTS is False, I set MAXMAILSIZE = 2 Mo")
                    self._maxmailsize = 2097152 
                else:
                    self._maxmailsize = None


            try:
                self._filelimit = int( Local.Config.Config.FILELIMIT )
            except AttributeError:
                self.log.info( "FILELIMIT not found in  Local/Config/Config.py, I set FILELIMIT = 2 Gib" )
                self._filelimit = 2147483648
            except ValueError:
                msg = "FILELIMIT have an invalid value : %s .\nIt must be an integer" % self._filelimit 
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg
 
            try:
                self._sessionlimit = int( Local.Config.Config.SESSIONLIMIT )
            except AttributeError:
                self.log.info( "SESSIONLIMIT not found in  Local/Config/Config.py, I set SESSIONLIMIT = 50 Mib" )
                self._sessionlimit = 52428800
            except ValueError:
                msg = "SESSIONLIMIT have an invalid value : %s .\nIt must be an integer" % self._sessionlimit 
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg

            try:
                self._previewDataLimit = int( Local.Config.Config.PREVIEW_DATA_LIMIT )
            except AttributeError:
                self.log.info( "PREVIEW_DATA_LIMIT not found in  Local/Config/Config.py, I set PREVIEW_DATA_LIMIT = 1 Mib" )
                self._previewDataLimit = 1048576
            except ValueError:
                msg = "PREVIEW_DATA_LIMIT have an invalid value : %s .\nIt must be an integer" % self._sessionlimit 
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg
              
            
            try:
                self._result_remain = int( Local.Config.Config.RESULT_REMAIN )
            except AttributeError:
                self.log.info( "RESULT_REMAIN not found in  Local/Config/Config.py, I set RESULT_REMAIN = 10 days" )
                self._result_remain = 10
            except ValueError:
                msg = "RESULT_REMAIN have an invalid value : %s .\nIt must be an integer" % Local.Config.Config.RESULT_REMAIN
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg



            #############################
            #
            #       SEQCONVERTER
            #
            #############################


            self._seqconverter = {}
            try:
                for conv in Local.Config.Config.SEQCONVERTER:
                    convUp = conv.upper()
                    if convUp not in ( 'SQUIZZ', 'READSEQ' ):
                        msg = self.log.error( conv + " is not an allowed value for SEQCONVERTER")
                        self.log.error( msg )
                        raise Mobyle.MobyleError.ConfigError , msg
                    else:
                        converter = Local.Config.Config.SEQCONVERTER[ conv ]
                        if os.path.exists( converter ):
                            self._seqconverter[ convUp ] = Local.Config.Config.SEQCONVERTER[ conv ]
                        else:
                            msg = str( converter ) + " : is not a path toward squizz or readseq"
                            self.log.error( msg )
                            raise Mobyle.MobyleError.ConfigError ,msg
            except AttributeError:
                msg = "SEQCONVERTER not found in  Local/Config/Config.py, it must be defined "
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg



            ##############################
            #
            #      Misc
            #
            ##############################


            try:
                self._lang = Local.Config.Config.LANG
            except AttributeError:
                self.log.info( "LANG not found in  Local/Config/Config.py, I set LANG=en" )
                self._lang = 'en'

            try:
                self._timeout = Local.Config.Config.TIMEOUT
            except AttributeError:
                self.log.info( "TIMEOUT not found in  Local/Config/Config.py, I set TIMEOUT=60" )
                self._timeout = 60


            ##############################
            #
            #      BATCH
            #
            ##############################

            try:
                self._batch = upper( str( Local.Config.Config.BATCH ) )
            except AttributeError:
                msg = "BATCH not found in  Local/Config/Config.py, it must be defined "
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError, msg

            if self._batch not in ('SYS', 'SGE', 'PBS'):
                msg = "BATCH can take only SYS , SGE , PBS values"
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg


            try:
                self._particular_batch = Local.Config.Config.PARTICULAR_BATCH
            except AttributeError:
                self.log.info( "PARTICULAR_BATCH not found in  Local/Config/Config.py" )
                self._particular_batch = {}


            try:
                self._default_Q = Local.Config.Config.DEFAULT_Q
            except AttributeError:
                if not self._batch == 'SYS':
                    msg = "%s required to defined DEFAULT_Q in Local/Config/Config.py" % self._batch
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg
                else:
                   self._default_Q = None 


            try:
                self._particular_Q  = Local.Config.Config.PARTICULAR_Q
            except AttributeError:
                self.log.info( "PARTICULAR_Q not found in  Local/Config/Config.py" )
                self._particular_Q = {}

            try:
                self._Qproperties = Local.Config.Config.Q_PROPERTIES
            except AttributeError:
                self.log.info( "Q_PROPERTIES not found in  Local/Config/Config.py" )
                self._Qproperties = {}
                
            try:
                self._sge_root = Local.Config.Config.SGE_ROOT
                self._sge_cell = Local.Config.Config.SGE_CELL

            except AttributeError ,err:
                if self._batch == "SGE" or "SGE" in self._particular_batch.values() :

                    msg = str( err ).split()[ -1 ]+" not found in Local/Config/Config.py\nIf you use SGE queueinq system you must defined SGE_ROOT and SGE_CELL"
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError ,msg
                else:
                    self._sge_root = None
                    self._sge_cell = None

            try:
                self._pbs_root = Local.Config.Config.PBS_ROOT
                self._pbs_wait_time = Local.Config.Config.DEFAULT_PBS_WAIT_TIME
                self._pbs_priority  = Local.Config.Config.DEFAULT_PBS_PRIORITY

            except AttributeError ,err:
                if self._batch == "PBS" or "PBS" in self._particular_batch.values() :

                    msg = str( err ).split()[ -1 ]+" not found in Local/Config/Config.py\nIf you use PBS queueinq system you must defined PBS_ROOT, DEFAULT_PBS_WAIT_TIME, DEFAULT_PBS_PRIORITY"
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError ,msg
                else:
                    self._pbs_root      = None
                    self._pbs_wait_time = None
                    self._pbs_priority  = None

            try:
                self._programs_installation_include = Local.Config.Config.LOCAL_INSTALL_INCLUDE
            except ( AttributeError , NameError ):
                self._programs_installation_include = ['*']
                
            try:
                self._programs_installation_exclude = Local.Config.Config.LOCAL_INSTALL_EXCLUDE
            except ( AttributeError , NameError ):
                self._programs_installation_exclude = []
                
            try:
                self._programs_installation_order = Local.Config.Config.LOCAL_INSTALL_ORDER
            except ( AttributeError , NameError ):
                self._programs_installation_order = [ 'include' , 'exclude' ]

                
            #############################
            #
            # disabled service
            #
            #############################

            try:
                self._disable_all = Local.Config.Config.DISABLE_ALL
            except AttributeError:
                self._disable_all = False
                msg = "DISABLE_ALL not found in  Local/Config/Config.py,set to False"
                self.log.info( msg )
            try:
                self._disabled_services = Local.Config.Config.DISABLED_SERVICES
            except AttributeError:
                self._disabled_services =[]
                msg = "DISABLED_SERVICES not found in  Local/Config/Config.py, "
                #self.log.info( msg )
                

            #########################################
            #
            # restriction services access
            #
            #########################################

            try:
                self._authorized_services = Local.Config.Config.AUTHORIZED_SERVICES
            except AttributeError:
                self._authorized_services = {}
                msg = "AUTHORIZED_SERVICES not found in  Local/Config/Config.py "
                #self.log.info( msg )

            #########################################
            #
            # Grid aspetcs
            #
            #########################################    

            try:
               self._all_portals = Local.Config.Config.PORTALS 
               for portal in self._all_portals .keys():
                   self._all_portals[ portal ][ 'url' ] = self._all_portals[ portal ][ 'url' ].rstrip( '/')
                   self._all_portals[ portal ][ 'repository' ] = self._all_portals[ portal ][ 'repository' ].rstrip( '/')
                   self._all_portals[ portal ][ 'jobsBase' ] = self._all_portals[ portal ][ 'jobsBase' ].rstrip( '/')
            except AttributeError:
               self._all_portals = {}

            try:
              self._exported_programs = Local.Config.Config.EXPORTED_SERVICES
            except AttributeError:
                self._exported_programs = []

        return self._ref

    
##############################################################

    
    ###############################
    #
    #    accessors
    #
    ##############################


    def debug( self , jobName = None ):
        """
        Returns the debug level for a job or the default debug level if jobName is not specified
        @param jobName:
        @type jobName: string
        @return: an int >=0 and <= 3
        @rtype: int
        """
        try:
            debug = self._particular_debug[ jobName ]
        except KeyError:
            debug = self._debug

        if debug < 0 or debug > 3:
            if jobName is None:
                msg =" DEBUG must be >= 0 and <= 3 I found DEBUG =%i, I set DEBUG = 0" % self._debug
            else:
                msg = jobName + " debug must be >= 0 and <= 3 I found %i, I set it to 0" % debug
            self.log.warning( msg )
        return debug

    def dirs( self, key = None ):
        """
        @param key: the name of a config variable in section direcories
        @type key: string
        @return: the path or url corresponding to the key
        @rtype: string
        @raise: KeyError 
        """
        if key is not None:
            return self._dirs[ key ]
        else:
            return self._dirs
    
    def document_root( self ):
        """
        @return:
        @rtype: string
        """
        return self._dirs[ 'document_root' ]
    
    
    def root_url( self ):
        """
        @return:
        @rtype: string
        """
        return self._dirs[ 'root_url' ]

    def cgi_dir( self ):
        """
        @return:
        @rtype: string
        """
        return self._dirs[ 'cgi_dir' ]

    def results_url( self ):
        """
        @return:
        @rtype: string
        """
        return self._dirs[ 'results_url' ]

    def results_path( self ):
        """
        @return:
        @rtype: string
        """
        return self._dirs[ 'results_path' ]

    def programs_url( self ):
        """
        @return:
        @rtype: string
        """
        return self._dirs[ 'programs_url' ]

    def programs_path( self ):
        """
        @return:
        @rtype: string
        """
        return self._dirs[ 'programs_path' ]
    
    
    def binary_path( self ):
        """
        @return:
        @rtype: string
        """
        return self._dirs[ 'binary_path' ]
    
    def programs_cache_path( self ):
        """
        @return: the path to the cache for imported services
        @rtype: string
        """
        return self._dirs[ 'programs_cache_path' ]


    def format_detector_cache_path( self ):
        """
        @return:
        @rtype: string
        """
        return self._dirs[ 'format_detector_cache_path' ]


    def user_sessions_path( self ):
        """
        @return:
        @rtype: string
        """
        return self._dirs[ 'user_sessions_path' ]
    
    def log_dir( self ):
        """
        @return:
        @rtype: string
        """

        return self._dirs['log']
       
    
    def htdocs_url( self ):
        """
        @return:
        @rtype: string
        @todo: a preciser la signification de cette varaiable
        """
        return self._dirs['htdocs_url']
    
    def cgi_url( self ):
        """
        @return:
        @rtype: string
        @todo: a preciser la signification de cette varaiable
        """
        return self._dirs['cgi_url']
    

    def programs( self , category  ):
        """
        @return:
        @rtype: 
        """
        try:
            return self._programs[ category ]
        except KeyError:
            return None

    def categories( self ):
        """
        @return:
        @rtype:
        """
        return self._programs.keys()
    

    def response( self ):
        """
        @return:
        @rtype:
        """
        return [ datatype['datatype'] for datatype in self._response['datatypes'] ]


    def getDatabanksConfig(self):
        """
        @return: the list of databanks
         along with their typing information and label
        @rtype: dictionary
        """
        return self._databanks_config
        

##     def klassMap( self , xmlName = None ):
##         """
##         @return: The name of the Mobyle class corresponding to the xml class
##         @rtype: string
##         """
##         if xmlName is None:
##             return self._knownclass
##         else:
##             try:
##                 return self._knownclass[ xmlName ]
##             except KeyError:
##                 return None
        
    def maintainer( self ):
        """
        @return: the email address of the Mobyle server  maintainer
        @rtype: string
        """
        return self._maintainer

    def mailHelp( self ):
        """
        @return: The email address to the hot line for this Mobyle server
        @rtype: string
        """
        return self._help
    
    
    def sender( self ):
        """
        the sender adress is used as From: for mail sended by mobyle
        - notification of long job
        - results
        - session confirmation  
        @return: the sender email address ( From: ) of this Mobyle server
        @rtype: string
        """
        return self._sender
    

    def mailhost( self ):
        """
        @return:  
        @rtype: string
        """
        return self._mailhost


    def opt_email( self , jobName = None ):
        """
        @return: True if the user email is not mandatory to run a job, False otherwise
        @rtype: Boolean
        """
        if jobName is None:
            return self._opt_email
        else:
            try:
                return self._particular_opt_email[ jobName ]
            except KeyError:
                return self._opt_email
            
    
    def anonymousSession( self ):
        """
        @return: 
          - 'no'       if the anonymous sessions are not allowed.
          - 'yes'      if the anonymous sessions are allowed, without any restrictions.
          - 'captcha'  if the anonymous sessions are allowed, with a captcha challenge.
        @rtype: string
        """
        return self._anonymous_session
    
    
    def authenticatedSession( self ):
        """
        @return: 
          - 'no'     if authenticated session are not allowed.
          - 'yes'    if authenticated session are allowed and activated without any restriction.
          - 'email'  if authenticated session are allowed but an email confirmation is needed to activate it.
        @rtype: string
        """
        return self._authenticated_session
    
    
    def dnsResolver( self ):
        return self._dns_resolver

    
    def mailResults( self ):
        """
        @return: dont_email_result , mailmaxsize in a tuple
        @rtype: ( boolean , int )
        """
        return ( self._dont_email_result , self._maxmailsize )

    def remainResults( self ):
        """
        @return: how long ( in days ) the results files remains on the server.
        @rtype: int
        """
        return self._result_remain
        
    def lang( self ):
        """
        @return: The default language used by this Mobyle server (used to internationalise the form) (2 letter code ) (default = "en")
        @rtype: string
        """
        return self._lang

    def filelimit( self ):
        """
        @return:
        @rtype:
        @todo: not used 
        """
        return self._filelimit

    def sessionlimit( self ):
        """
        @return: the max size for a session directory ( in byte )
        @rtype: int
        """
        return self._sessionlimit
    
    def previewDataLimit( self ):
        """
        @return: the max size for a result to be preview in job result
        @rtype: int
        """
        return self._previewDataLimit   

    
    def seqconverter( self , prgName = None ):
        """
        @return: the path to the sequence converter name prgName. if prgName is None  return the list of progName defined ( squizz has a higher priority than redaseq )
        @rtype: string
        """
        if prgName is None:
            l = []
            if 'SQUIZZ' in self._seqconverter:
                l.append( 'SQUIZZ' )
            if 'READSEQ' in self._seqconverter:
                l.append( 'READSEQ' )
            return l
        else:
            try:
                return self._seqconverter[ prgName.upper() ]
            except KeyError:
                return None
    
    def timeout( self ):
        """
        @return: 
        @rtype: int 
        """
        return self._timeout

    
    def batch( self , jobName = None ):
        """
        @return: the batch system used by this jobName (Sys ,PBS ,SGE ...). if jobName is None return the default value
        @rtype: string
        """
        try:
            batch = upper(self._particular_batch[jobName])
        except KeyError:
            batch = self._batch
        return batch


    def queue( self , jobName = None ):
        """
        @return: the queue name for this jobName. if jobName is None return the default queue
        @rtype: string
        """

        # rc_log.debug("queue for: %s" % jobName)
        try:
            queue  = self._particular_Q[ jobName ]
        except KeyError:
            queue = {"queue": self._default_Q}
        # rc_log.debug("queue is: %s" % queue)

        return queue

    def queueProperties( self, Qname ):
        """
        @param Qname = the name of the queue
        @type Qname: strinh
        @return: the properties associated with a queue
        @rtype: dict
        """
        # rc_log.debug("queueProperties for: %s" % Qname)
        try:
            rs = self._Qproperties[ Qname ]
            return rs
        except KeyError:
            rc_log.critical("queueProperties problem for %s" % Qname)
            return None
            
    def sge( self ):
        """
        @return: the values of variables SGEROOT , SGECELL
        @rtype: tuple (string , string )
        """
        return ( self._sge_root , self._sge_cell )

##     def pbsqueues( self ):
##         """
##         @return: the informations about the queues
##         @rtype: dictionnary
##         """
##         return self._Qproperties

    def pbs( self ):
        """
        @return: the value of variables PBSROOT
        @rtype: string
        """
        return (self._pbs_root, self._pbs_wait_time, self._pbs_priority)


    def accounting( self ):
        """
        @return: Turn on/off the accounting
        @rtype: boolean
        """
        return self._accounting

    
    def isDisabled( self , programName = None ):
        """
        @param programName: the name of the program (e.g., topppred)
        @type: string
        @return: True if the service is disabled, False otherwise
        @rtype: boolean
        """
        
        if programName is None:
            return self._disable_all
        
        else:
            if self._disable_all :
                return True
            else:
                return programName in self._disabled_services 




    def restrictedServices( self ):
        """
        @return: The list of service which have a restrictive access
        @rtype: list of strings
        """
        return self._authorized_services.keys()


    def isAuthorized( self , serviceName , addr = None ):
        """
        @param service: the name of the service
        @type service: type
        @param addr: the ip address to test. if addr is None the environemnt REMOTE_ADDR will be used
        @type addr: string
        @return: True if the user is authorized to used the service (based on IP), False otherwise.
        @rtype: boolean
        """
        if addr is None:
            try: 
                addr = os.environ[ 'REMOTE_ADDR' ]
            except KeyError:
                return True
        try:
            pattern = self._authorized_services[ serviceName ]
        except KeyError:
            # by default if no pattern are specified there is no restriction
            # to acces a this service
            return True
        
        if pattern :
            pattern = '|'.join( pattern )
            pattern = pattern.replace('.' , '\.' )
            pattern = pattern.replace('*' , '.*')
            pattern = "^(%s)$" % pattern
            auto = re.compile( pattern )
        else:
            # by default if no pattern are specified there is no restriction
            # to acces a this service
            return True
        
        if auto.search( addr ):
            return True
        else:
            return False

    def getPrivilegeTable( self ):
        """
        @return: a dict where keys are the service names and values a list of Ip mask wchich are autorized to acces to this service
        @rtype: dict { string : [ strings ] }
        """
        return self._authorized_services

    def imported_programs( self ):
        """
        @return: the urls of the programs imported by this server
        @rtype: list of strings
        """
        return self._imported_programs
    
    def exported_programs( self ):
        """
        @return: the name of the programs which are exported by this server
        @rtype: list of strings
        """
        return self._exported_programs
    
    def portals( self ):
        """
        @return: all the Mobyle portals belonging to the "Mobyle grid"
        @rtype: list of dictionnaries  [ { 'server' : 'url of Mobyle server' ,
                                            'email' : 'the contact email of this server'
                                         }
                                        ]
        """
        return self._all_portals
    
    def programs_installation_include(self):
        """
        @return: the programs installation include masks list
        @rtype: list of strings
        """
        return self._programs_installation_include
    
    def programs_installation_exclude(self):
        """
        @return: the programs installation exclude masks list
        @rtype: list of strings
        """
        return self._programs_installation_exclude    
    
    def programs_installation_order(self):
        """
        @return: the programs installation include/exclude priority order
        @rtype: list of strings
        """
        return self._programs_installation_order

    
    
    
class PassChecker:
    """

    """

    _cfg = Config()


    def __init__( self ):
        """
        build a dictionary with all IP masks found in Config.Config.AUTHORIZED_SERVICES as keys and the services authorized for each mask as values
        """
        
        serviceInfo = {}
    
        for path in glob.glob( os.path.join ( os.environ['MOBYLEHOME'] , "Local" , "Programs" ,"*.xml" ) ):

            serviceInfo [ os.path.basename( path )[:-4] ] = None

        for path in  glob.glob( os.path.join ( os.environ['MOBYLEHOME']  , "Programs" ,"*.xml" ) ):
            service = os.path.basename( path )[:-4]

            if not serviceInfo.has_key( service ):
                serviceInfo [ service ] = None


        self.maskService = { '*.*.*.*' : None }
        self.authorizedServices = PassChecker._cfg.getPrivilegeTable()
        restrictedServices = self.authorizedServices.keys()
        
        for service in restrictedServices :

            for mask in self.authorizedServices[ service ]:

                if self.maskService.has_key( mask ):
                    continue
                else:
                    self.maskService[ mask ] = None

        errors = []
        for mask in self.maskService.keys():
            newServiceInfo = copy.deepcopy( serviceInfo )

            for service in restrictedServices :

                if not PassChecker._cfg.isAuthorized( service , mask ):
                    try:
                        del newServiceInfo[ service ]
                    except KeyError:
                        msg = "the service :\"%s\" is in your Local.Config.Config.AUTHORIZED_SERVICES but the xml description of this service does no exist" %service
                        if msg not in errors :
                            errors.append( msg )
            self.maskService[mask] = newServiceInfo.keys() 

        for error in errors:
            PassChecker._cfg.log.warning( error )
            
            

    def getMask( self , addr ):
        """
        @param addr: an ip adresse (ipv4) like '192.168.3.1'
        @type addr: string
        @return: the mask which fit the best ( permit to acces at the most services )to the addr. or None if any mask match 
        @rtype: string
        """
        masks = []
        
        for mask in self.maskService.keys():
            pattern = mask.replace( '.' , '\.' )
            pattern = pattern.replace( '*' , '.*')
            pattern = "^(%s)$" % pattern
            auto = re.compile( pattern )
            if auto.search( addr ):
                masks.append( mask )

        if len( masks ) == 1:
            return masks[ 0 ]
        else:
            #celui qui permet d'acceder aux maximum de services.
            bestmask = None
            bestscore = 0
            
            for mask in masks:
                score = len( self.maskService[ mask ] )
                if score > bestscore:
                    bestscore = score
                    bestmask = mask

            return bestmask


    def getAllMasks( self ):
        """
        @return: the list of all mask
        @rtype: list of strings
        """
        return self.maskService.keys()


    def itermasks( self ):
        """
        @return: an iterator on all masks
        """
        for m in self.maskService.keys() :
            yield m


    def __getitem__( self , mask ):
        """
        @param mask: an ip mask (ipv4) like '192.168.*.*'
        @type mask: string
        @return: the list of services accessible for an ip address witch match to the mask
        @rtype: list of string
        """
        return self.maskService[ mask ]


    def has_mask( self , mask ):
        """
        @param mask: an ip mask (ipv4) like '192.168.*.*'
        @type mask: string
        @return: True if the mask is in this PassChecker instance, False othrewise
        @rtype: boolean
        """
        return self.maskService.has_key( mask )

