########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.        #
#                                                                                      #
########################################################################################

"""
Content the Mobyle Parameter types related to the genomic bioloy
"""
 

import os , os.path , re
import logging

import Mobyle.SequenceConverter
import Mobyle.AlignmentConverter
import Mobyle.Utils
from Mobyle.MobyleError import MobyleError , UserValueError , UnDefAttrError

from Mobyle.Classes.Core import *

c_log = logging.getLogger( 'mobyle.core' )
b_log = logging.getLogger('mobyle.builder')








class SequenceDataType( AbstractTextDataType ):

    @staticmethod
    def convert(  value , param ):
        """
        convert the sequence contain in the file fileName in the rigth format
        throws an UnsupportedFormatError if the output format is not supported
        or a MobyleError if something goes wrong during the conversion.

        @param value: is a tuple ( destFileName , data , dest , src , srcFileName) 
          - filename is mandatory
          - data is a string reprensenting a sequence. it could be None if src is specify
          - dest is L{Job} or L{Session} instance is where the data will be store
          - src is L{Job} or L{Session} instance is where the data come from (it must be specify only if data is None).
        @type value: ( string filename, string data , L{Job} or L{Session} instance dest ,L{Job} or L{Session} instance, src)
        @return: the fileName ( basename ) of the  sequence file
        """
        if param.isout():
            raise UserValueError( parameter = param , msg = "out parameter can't be modify by users" )

        if value is None:
            return None
         
        elif len( value ) == 5 :
             data , dest , destFileName , src , srcFileName  = value
        else:
            raise MobyleError , "value must be a tuple of 5 elements: ( data , dest , destFileName , src , srcFileName )"

        if destFileName is None:
            return None
        
        if dest is None:
            raise MobyleError, "the destination is mandatory"

        if  data and src :
            raise MobyleError, "you cannot specify data and src in the same times"

        if not data and ( not dest or not src ) :
            raise MobyleError , "if data is not specify, dest and src must be defined"

        if src and not srcFileName :
            raise MobyleError , "if src is specify , srcFileName must be also specify"

        try:
            jobstate = dest.jobState
        except AttributeError :
            jobstate = None
        
        if param.isInfile():

            fileName  , sizeIn = SequenceDataType._toFile( param , data  , dest , destFileName , src , srcFileName )
            absFileName = os.path.join( dest.getDir() , fileName )
            prg , code , format , seq_nb = Mobyle.SequenceConverter.detect( absFileName )
            
            codeList = param.getAcceptedDataFormats()
            if not codeList:
                service = param.getFather()
                
                if service is not None:
                    while not service.isService():
                        service = service.getFather()
                              
                if format is not None: 
                        if jobstate is not None:
                            lang = param.cfg.lang()
                            if param.promptHas_lang( lang ):
                                prompt = param.getPrompt( lang = lang )
                            else:
                                prompt = None
                                lang   = None
    
                            jobstate.setInputDataFile( param.getName() ,
                                                       ( prompt , lang ) ,
                                                       param.getType() , 
                                                        ( fileName  ,  sizeIn ,  format ) ,
                                                       fmtProgram = prg 
                                                       ) 
                            jobstate.commit()
                        param.setDataFormat( ( format , seq_nb , prg ) )  #permit at the session to have format information  
                        return os.path.basename( fileName )
                
                else:
                    msg = " invalid Sequence format"
                    raise UserValueError( parameter = param , msg = msg )

            try:
                forceSeqfmt = param.forceReformating()
            except UnDefAttrError :
                forceSeqfmt = False

            # I must work with absolute path because the portal use this
            # and it not work in the same directory as the sequence files

            # the decision to convert the sequence if the format detected is in codeList or not
            # is take by Mobyle.SequenceConverter.convert
            fmtPrg , fmtIn , inFileName ,fmtOut , outFileName , sq_nb = Mobyle.SequenceConverter.convert( absFileName , codeList , force = forceSeqfmt )

            param.setDataFormat( ( fmtOut , seq_nb , fmtPrg ) ) #permit at the session to have format information
            
            if fmtIn is None :
                msg ="Cannot detect the sequence format. The supported sequence formats are: " + str( Mobyle.SequenceConverter.supportedFormat() )
                                       
                if jobstate is not None:
                    lang = param.cfg.lang()
                    if param.promptHas_lang( lang ):
                        prompt = param.getPrompt( lang = lang )
                    else:
                        prompt = None
                        lang   = None

                    jobstate.setInputDataFile( param.getName() ,
                                               ( prompt , lang ) ,
                                               param.getType() ,
                                               (  fileName  , sizeIn , "UNKNOWN" ) ,
                                               fmtProgram = prg 
                                               )                
                    jobstate.commit()

                raise UserValueError( parameter = param , msg = msg )    
                        
            elif jobstate is not None:
                lang = param.cfg.lang()
                if param.promptHas_lang( lang ):
                    prompt = param.getPrompt( lang = lang )
                else:
                    prompt = None
                    lang = None
                
                if fmtOut is None: # the data has not been converted
                    jobstate.setInputDataFile( param.getName() ,
                                               ( prompt , lang ) ,
                                               param.getType() ,
                                               ( inFileName , sizeIn , fmtIn ) ,
                                               fmtProgram = fmtPrg
                                               )
                    jobstate.commit()
                else:# the data has been converted
                    sizeOut = os.path.getsize( outFileName )
                    jobstate.setInputDataFile( param.getName() ,
                                               ( prompt , lang ) ,
                                               param.getType() ,
                                               ( inFileName , sizeIn , fmtIn ) ,
                                               fmtProgram = fmtPrg  ,
                                               formattedFile = ( outFileName , sizeOut , fmtOut )
                                               )                    
                jobstate.commit()
                
            outFileName = os.path.basename( outFileName )
            return outFileName


        else: #it's a file in output
            fileName = Mobyle.Utils.safeFileName( destFileName )
        
        return fileName 
       
    @staticmethod
    def detect(  param  ):
        """
        detect the sequence contain in the file fileName .
        throws an UnsupportedFormatError if the output format is not supported
       
        @param value: is a tuple ( filename , data , dest ) 
          - filename is mandatory
          - data is a string reprensenting a sequence. 
          - dest is L{Job} or L{Session} instance is where the data will be store
        @type value: ( string filename, string data , L{Job} or L{Session} instance dest)
        @return: the program which detect the format
                 the sequence format
                 the number of sequences
                 the filename
        @rtype: ( string prg , int code , string format , int seq_nb , string fileName ) 
        """

          
        if value is None :
            return None

        elif len( value ) == 4:
            data , dest , destFileName , src , srcFileName = value
        else:
            raise MobyleError ,"value must be a tuple of 4 elements:( data , dest , destFileName , src , srcFileName ) "

        if fileName is None:
            return None
        
        if dest is None:
            raise MobyleError, "the destination is mandatory"

        if  data and src :
            raise MobyleError, "you cannot specify data and src in the same times"

        if not data and ( not dest or not src ) :
            raise MobyleError , "if data is not specify, dest and src must be defined"

        if param.isInfile() :
            fileName , size =   SequenceDataType._toFile( param , data , dest , destFileName , src , srcFileName )
            absInFileName = os.path.join( dest.getDir() , fileName )
            fmtPrg , code , fmtOut , seq_nb = Mobyle.SequenceConverter.detect( absInFileName )
            
            param.setDataFormat( ( fmtOut , seq_nb , fmtPrg ) ) #permit at the session to have format information
            
            return ( fmtPrg , code , fmtOut , seq_nb , fileName )

        else:
            fileName = Mobyle.Utils.safeFileName( fileName )
       
        return fileName 
       


    @staticmethod
    def validate( param ):
        """
        """
        value = param.getValue()

        if param.isout():
            if value is not None : #un parametre isout ne doit pas etre modifier par l'utilisateur 
                return False
            else:
                
                
                #####################################################
                #                                                   #
                #  check if the Parameter have a secure filenames   #
                #                                                   #
                #####################################################
                
                try:
                    debug = param.getDebug()
                    if debug > 1:
                        b_log.debug( "check if the Parameter have a secure filename" )
    
                    #getFilenames return list of strings representing a unix file mask which is the result of a code evaluation
                    #getFilenames return None if there is no mask for a parameter.
                    filenames = param.getFilenames( ) 
    
                    for filename in filenames :
                        if filename is None:
                            continue
                        mask = safeMask( filename )
    
                        b_log.debug( "filename= %s    safeMask = %s"%(filename, mask))
                        if  not mask or mask != filename :
            
                            raise MobyleError , "have an unsecure filenames value before safeMask: %s , after safeMask: %s"%( filename , mask )
                        else:
                            if debug > 1:
                                b_log.debug( "filename = %s ...........OK" % filename )
    
                                               
                except UnDefAttrError :
                    self.build_log.debug("no filenames")

        else:
            if value is None:
                return True
            else:
                prg , code , format , seq_nb = Mobyle.SequenceConverter.detect( value )
                
                try:
                    #codeList = self.getSeqfmt()
                    codeList = param.getAcceptedDataFormats()
                except UnDefAttrError:
                    return True
        
                if code in codeList:
                    return True
                else:
                    return False

    
class AlignmentDataType( AbstractTextDataType ):
#class AlignmentDataType( SequenceDataType ):
    
    @staticmethod
    def convert( value , param  ):
        """
        convert the alignment contain in the file fileName in the rigth format
        throws an UnsupportedFormatError if the output format is not supported
        or a MobyleError if something goes wrong during the conversion.

        @param value: is a tuple the first element
          - filename is mandatory
          - data is a string reprensenting an alignment. it could be None if src is specify
          - dest is L{Job} or L{Session} instance is where the data will be store
          - src is L{Job} or L{Session} instance is where the data come from (it must be specify only if data is None).
        @type value: ( string filename, string data , L{Job} or L{Session} instance dest ,L{Job} or L{Session} instance, src)
        @return: the fileName ( basename ) of the alignment file
        """
        if param.isout():
            raise UserValueError( parameter = param , msg = "out parameter can't be modify by users" )
        #if param.isout():
        #    if value is None :
        #        raise UserValueError( parameter = param , msg = msg )
        #    else:
        #        safeFileName = Mobyle.Utils.safeFileName( value )
        #        return safeFileName

        if value is None:
            return None

        elif len( value ) == 5 :
            data , dest , destFileName , src , srcFileName = value
        else:
            raise MobyleError ,"value must be a tuple of 5 elements: ( data , dest , destFileName , src , srcFileName )"

        if destFileName is None:
            return None
        
        if dest is None:
            raise MobyleError, "the destination is mandatory"

        if  data and src :
            raise MobyleError, "you cannot specify data and src in the same times"

        if not data and ( not dest or not src ) :
            raise MobyleError , "if data is not specify, dest and src must be defined"

        try:
            jobstate = dest.jobState
        except AttributeError:
            jobstate = None

        if src and not srcFileName :
            raise MobyleError , "if src is specify , srcFileName must be also specify"
        
        if param.isInfile():
            fileName , sizeIn = AlignmentDataType._toFile( param , data , dest , destFileName , src , srcFileName )
            
            absInFileName = os.path.join( dest.getDir() , fileName )
            sizeIn = os.path.getsize( absInFileName )
            prg , code , format , al_nb = Mobyle.AlignmentConverter.detect( absInFileName )
            
            codeList = param.getAcceptedDataFormats()
            if not codeList:
                service = param.getFather()
                
                if service is not None:
                    while not service.isService():
                        service = service.getFather()

                if format is not None:#codelist =None , format != None => it's a session no conversion
                    if jobstate is not None:
                        lang = param.cfg.lang()
                        if param.promptHas_lang( lang ):
                            prompt = param.getPrompt( lang = lang )
                        else:
                            prompt = None
                            lang   = None

                        jobstate.setInputDataFile( param.getName() ,
                                                   ( prompt , lang ) ,
                                                   param.getType() ,
                                                    ( fileName  , sizeIn , format ) ,
                                                   fmtProgram = prg 
                                                   )
                        jobstate.commit()
                    param.setDataFormat( ( format , al_nb , prg ) )    #permit at the session to have format information
                    return  fileName 
                
                else:
                    msg = " invalid Alignment format"
                    raise UserValueError( parameter = param , msg = msg )

            try:
                forceSeqfmt = param.forceReformating()
            except UnDefAttrError :
                forceSeqfmt = False

            # I must work with absolute path because the portal use this
            # and it not work in the same directory as the sequence files

            # the decision to convert the Alignment if the format detected is in codeList or not
            # is take by Mobyle.AlignmentConverter.convert
            fmtPrg , fmtIn , inFileName , fmtOut , outFileName , al_nb = Mobyle.AlignmentConverter.convert( absInFileName ,
                                                                                                     codeList , 
                                                                                                     force = forceSeqfmt )

            param.setDataFormat( ( fmtOut , al_nb , fmtPrg ) ) #permit at the session to have format information

            if fmtIn is None :
                msg ="Cannot detect the alignment format. The supported alignment formats are: " + str( Mobyle.AlignmentConverter.supportedFormat() )
                if jobstate is not None:
                    lang = param.cfg.lang()
                    if param.promptHas_lang( lang ):
                        prompt = param.getPrompt( lang = lang )
                    else:
                        prompt = None
                        lang   = None
    
                    jobstate.setInputDataFile( param.getName() ,
                                               ( prompt , lang ) ,
                                               param.getType() ,
                                               (  fileName ,  sizeIn , "UNKNOWN" ) ,
                                               fmtProgram = prg 
                                               )                
                    jobstate.commit()
                    
    
                raise UserValueError( parameter = param , msg = msg )    
                        
            elif jobstate is not None:
                lang = param.cfg.lang()
                if param.promptHas_lang( lang ):
                    prompt = param.getPrompt( lang = lang )
                else:
                    prompt = None
                    lang   = None

                
                if fmtOut is None: # the data was not convert
                    jobstate.setInputDataFile( param.getName() ,
                                               ( prompt , lang ) ,
                                               param.getType() ,
                                               ( inFileName , sizeIn , fmtIn ) ,
                                               fmtProgram = fmtPrg
                                               )
                    jobstate.commit()
                else:# the was conveerted
                    sizeOut = os.path.getsize( os.path.join( dest.getDir() , outFileName) )
                    jobstate.setInputDataFile( param.getName() ,
                                               ( prompt , lang ) ,
                                               param.getType() ,
                                               ( inFileName , sizeIn , fmtIn ) ,
                                               fmtProgram = fmtPrg  ,
                                               formattedFile = ( outFileName , sizeOut , fmtOut )
                                               )
                jobstate.commit()
                
            outFileName = os.path.basename( outFileName )
            return outFileName

        else:
            fileName = Mobyle.Utils.safeFileName( fileName )
        
        return fileName 
       

    @staticmethod
    def detect(  value  ):
        """
        convert the sequence contain in the file fileName in the rigth format
        throws an UnsupportedFormatError if the output format is nos supported
        or a MobyleError if something goes wrong during the conversion.

        @param value: the fileName of the sequence file
        @type value: String
        @return: the fileName of the  sequence file
        """
          
        if value is None :
            return None

        elif len( value ) == 5 :
             data , dest , destFileName , src , srcFileName = value
        else:
            raise MobyleError ,"value must be a tuple of 5 elements: (  data , destination , destFileName , src , srcFileName )"

        if destFileName is None:
            return None
        
        if dest is None:
            raise MobyleError, "the destination is mandatory"

        if  data and src :
            raise MobyleError, "you cannot specify data and src in the same times"

        if not data and ( not dest or not src ) :
            raise MobyleError , "if data is not specify, dest and src must be defined"

        if src and not srcFileName :
            raise MobyleError , "if src is specify , srcFileName must be also specify"

        if param.isInfile() :
            try:
                fileName , size = AlignmentDataType._toFile( param , data , dest , destFileName , src , srcFileName )
            except UserValueError , err :
                pass  #a gerer idem convert TextDataType

            fmtPrg , code , fmtOut , al_nb = Mobyle.AlignmentConverter.detect( os.path.join( dest.getDir() , fileName ) )
            param.setDataFormat( ( fmtOut , al_nb , fmtPrg ) )

            return ( fmtPrg , code , fmtOut ,  al_nb , fileName )
        

        else:
            fileName = Mobyle.Utils.safeFileName( fileName )
        
        return fileName 


    @staticmethod
    def validate( param ):
        """
        filename = nom de fichier local tout est en local!
        cette methode ne doit etre apple que si l'objet est ratache a un service , job ...
        
        convert the sequence contain in the file fileName in the rigth format
        throws an UnsupportedFormatError if the output format is nos supported
        or a MobyleError if something goes wrong during the conversion.
        """
        value = param.getValue()


        if param.isout():
            if value is not None : #un parametre isout ne doit pas etre modifier par l'utilisateur 
                return False
            else:
                
                
                #####################################################
                #                                                   #
                #  check if the Parameter have a secure filenames   #
                #                                                   #
                #####################################################
                
                try:
                    debug = param.getDebug()
                    if debug > 1:
                        b_log.debug( "check if the Parameter have a secure filename" )
    
                    #getFilenames return list of strings representing a unix file mask which is the result of a code evaluation
                    #getFilenames return None if there is no mask for a parameter.
                    masks = param.getFilenames( ) 
    
                    for mask in masks :
                        if mask is None:
                            continue
                        mySafeMask = safeMask( mask )
    
                        b_log.debug( "filename= %s    safeMask = %s"%(mySafeMask, mask))
                        if  not mySafeMask or mySafeMask != mask :
                            raise MobyleError , "Mobyle Internal Server Error"
                        else:
                            if debug > 1:
                                b_log.debug( "filename = %s ...........OK" % mask )
    
                                               
                except UnDefAttrError :
                    self.build_log.debug("no filenames")

        else:
            if value is None:
                return True
            else:
                prg , code , format , al_nb = Mobyle.AlignmentConverter.detect( value )
                
                try:
                    codeList = param.getAcceptedDataFormats()
                except UnDefAttrError:
                    return True
        
                if code in codeList:
                    return True
                else:
                    return False
            




