/* golden.c - Golden retriever main functions */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#ifdef STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <ctype.h>

#include "access.h"
#include "error.h"
#include "list.h"
#include "locus.h"


#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS 0
#define EXIT_FAILURE 1
#endif

/* Functions prototypes */
static void usage(int status);

/* Global variables */
static char *prog;


/* Main function */
int main(int argc, char **argv) {
  FILE *f;
  int i, loc, acc;
  char *p, *q, *dbase, *name;
  size_t len;
  indix_t res;

  /* Inits */
  prog = *argv;
  if ((p = strrchr(prog, '/')) != NULL) prog = ++p;

  /* Checks command line options & arguments */
  i = loc = acc = 0;
  while((i = getopt(argc, argv, "ahil")) != -1) {
    switch(i) {
    case 'a':
      acc = 1; break;
    case 'h':
      usage(EXIT_SUCCESS); break;
    case 'i':
      loc = 1; break;
    case 'l':
      if (list_check()) error_fatal("databases", "cannot retrieve list");
      return EXIT_SUCCESS;
    default:
      usage(EXIT_FAILURE); break; }
  }
  if ((loc + acc) == 0) { loc = acc = 1; }
  if (argc - optind != 1)
    usage(EXIT_FAILURE);

  /* Parse database & query names */
  p = argv[optind]; len = strlen(p);
  if (strchr(p, ':') == NULL)
    error_fatal(p, "invalid query value");
  if ((dbase = (char *)malloc(len+1)) == NULL ||
      (name = (char *)malloc(len+1)) == NULL)
    error_fatal("memory", NULL);
  q = dbase; while(*p && *p != ':') *q++ = *p++; *q = '\0'; p++;
  q = name; while(*p) *q++ = (char)toupper((int)*p++); *q = '\0';

  /* Search name in databases indexes */
  if (acc) {
    res = access_search(dbase, name);
    if (res.name[0] != '\0') loc = 0; }
  if (loc) {
    res = locus_search(dbase, name); }
  if (res.name[0] == '\0')
    error_fatal(name, "no such entry in selected database");

  /* Get database flat file name */
  p = list_name(dbase, res.filenb);

  /* Display database entry */
  if ((f = fopen(p, "r")) == NULL)
    error_fatal(p, NULL);
  if (fseek(f, res.offset, SEEK_SET) != 0)
    error_fatal(p, NULL);
  if (entry_display(f))
    error_fatal(p, NULL);
  if (fclose(f))
    error_fatal(p, NULL);
  free(p);

  free(name); free(dbase);

  return EXIT_SUCCESS; }


/* Usage display */
static void usage(int status) {
  FILE *f = stdout;

  (void)fprintf(f, "usage: %s [options] <dbase:name>\n\n", prog);
  (void)fprintf(f, "options:\n");
  (void)fprintf(f, "  -a ... Search query by accession number.\n");
  (void)fprintf(f, "  -h ... Prints this message and exit.\n");
  (void)fprintf(f, "  -i ... Search query by entry name.\n");
  (void)fprintf(f, "  -l ... List available databases.\n");

  exit(status); }
