/* fqextract.c - Extract entries from a FASTQ sequence file */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <ctype.h>
#include <err.h>
#include <fcntl.h>
#include <libgen.h>
#include <stddef.h>
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif
#include <stdio.h>
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#include <string.h>
#include <unistd.h>

#include "fqlist.h"
#include "xfile.h"

static void usage(const char *);

int main(int argc, char **argv) {
  FILE *in, *ou = stdout;
  const char *prg, *sep;
  char *buf, *list = NULL, *ofile = NULL;
  int i, keep, pair = 1, excl = 0;
  size_t len;
  fqent_t ent;
  fqlist_t lst;

  /* Inits */
  prg = basename(*argv);
  sep = " /\n";

  /* Check command line */
  while ((i = getopt(argc, argv, "hl:o:px")) != -1) {
    switch (i) {
    case 'h':
      usage(prg); return EXIT_SUCCESS;
    case 'l':
      list = optarg; break;
    case 'o':
      ofile = optarg; break;
    case 'p':
      pair = 0; break;
    case 'x':
      excl = 1; break;
    default:
      usage(prg); return EXIT_FAILURE; }
  }
  if (argc - optind < 1) {
    usage(prg); return EXIT_FAILURE; }

  keep = 1 - excl;

  len = 1024;
  if ((buf = malloc(len)) == NULL)
    err(EXIT_FAILURE, "memory: malloc failed");

  /* Load provided entries list */
  if (list != NULL) {
    fqlist_init(&lst);
    if ((in = fopen(list, "r")) == NULL)
      err(EXIT_FAILURE, "%s: open failed", list);
    while (fgets(buf, (int)len, in) != NULL) {
      ent.buf = strdup(buf); ent.num = 0;
      ent.nam = ent.buf; ent.nlen = strcspn(ent.nam, sep);
      if (pair == 1) {
	const char *x = ent.nam + ent.nlen;
	if (*x && *x != '\n') ent.num = atoi(x+1); }
      fqlist_add(&lst, &ent); }
    if (fclose(in) == EOF)
      err(EXIT_FAILURE, "%s: close failed", list);
    fqlist_sort(&lst); }

  /* Open output file if requested */
  if (ofile != NULL && (ou = fopen(ofile, "w")) == NULL)
    err(EXIT_FAILURE, "%s: open failed", ofile);

  for (i = optind; i < argc; i++) {
    const char *ifile = *(argv+i);
    char *p, *q, *s, *xbuf;
    ptrdiff_t dif;
    size_t xfer, xinc, xmax;
    ssize_t ssiz, rsiz;
    uint64_t lig;
    xfile_t *xf;

    if ((xf = xfile_open(ifile, "r")) == NULL)
      err(EXIT_FAILURE, "%s: open failed", ifile);
    xfile_xfer(&xfer, xf); xinc = xfer; xmax = 10 * xinc;
    if ((xbuf = malloc(xfer+1)) == NULL)
      err(EXIT_FAILURE, "memory allocation failed");
    *(xbuf+xfer) = '\0';

    ssiz = rsiz = 0; lig = 0;
    while (/*CONSTCOND*/1) {
      ssiz = xfile_read(xf, xbuf + rsiz, xfer - (size_t)rsiz);
      if (ssiz < 1) break;

      rsiz += ssiz; p = xbuf;
      while (/*CONSTCOND*/ 1) {

        q = memchr(p, '\n', (size_t)rsiz);
        if (q == NULL) { break; }
        dif = q - p + 1; s = p; p = q + 1; lig++;
        rsiz -= dif;

	if (list != NULL && lig % 4 == 1) { /* Sequence header */
	  ent.buf = s; ent.nam = ent.buf + 1; ent.num = 0;
	  ent.nlen = strcspn(ent.nam, sep);
	  if (pair == 1) {
	    const char *x = ent.nam + ent.nlen;
	    if (*x && *x != '\n') ent.num = atoi(x+1); }
	  keep = 1 - fqlist_chk(&lst, &ent);
	  if (excl == 1) { keep = 1 - keep; }
	}

	if (keep == 0) continue;
	(void)fprintf(ou, "%.*s\n", (int)dif - 1, s);

      }

      if (rsiz == 0) continue;
      (void)memmove(xbuf, p, (size_t)rsiz);
      if (p != xbuf) continue;
      if (xfer == xmax)
        errx(EXIT_FAILURE, "%s: line too long", ifile);
      xfer += xinc;
      if ((xbuf = realloc(xbuf, xfer+1)) == NULL)
        err(EXIT_FAILURE, "realloc failed");
      *(xbuf+xfer) = '\0';
    }
    if (xfile_error(xf) != 0)
      err(EXIT_FAILURE, "%s: read failed", ifile);
    if (lig % 4 != 0)
      errx(EXIT_FAILURE, "%s: truncated file", ifile);

    free(xbuf);
    if (xfile_close(xf) == EOF)
      err(EXIT_FAILURE, "%s: close failed", ifile);
  }

  if (ofile != NULL && fclose(ou) == EOF)
    err(EXIT_FAILURE, "%s: close failed", ofile);

  if (list != NULL)
    fqlist_fini(&lst);

  free(buf);

  return 0; }


static void usage(const char *nam) {
  FILE *f = stderr;

  (void)fprintf(f, "usage: %s [options] <file> ...\n", nam);
  (void)fprintf(f, "\noptions:\n");
  (void)fprintf(f, "  -h        ... Print this message and exit.\n");
  (void)fprintf(f, "  -l <list> ... Select entries listed in <list>.\n");
  (void)fprintf(f, "  -o <file> ... Save output to file <file>.\n");
  (void)fprintf(f, "  -p        ... Do not use pair member information.\n");
  (void)fprintf(f, "  -x        ... Exclude entries listed in <list>.\n");

  return; }
