/* fastq.h - FASTQ sequence file functions */

#include <stddef.h>
#include <stdio.h>

/* FastQ sequence */

typedef struct {
  size_t lnam; char *snam;    /* Name */
  size_t lseq; char *sseq;    /* Sequence */
  size_t lqua; char *squa;    /* Quality */
} fqseq_t;

int  fastq_init(fqseq_t *);
int  fastq_fini(fqseq_t *);
void fastq_null(fqseq_t *);
int  fastq_read(FILE *, fqseq_t *);
int  fastq_write(FILE *, fqseq_t *);

/* FastQ scoring */

#define FASTQ_NONE        0x00
#define FASTQ_SANGER      0x01
#define FASTQ_SOLEXA      0x02
#define FASTQ_ILLUMINA13  0x04
#define FASTQ_ILLUMINA15  0x08
#define FASTQ_ILLUMINA17  0x10
#define FASTQ_ILLUMINA18  0x20
#define FASTQ_TORRENT30   0x40
#define FASTQ_SANGERQ50   0x80
#define FASTQ_UNKNOWN     (FASTQ_SANGER|FASTQ_SOLEXA|FASTQ_ILLUMINA13| \
                           FASTQ_ILLUMINA15|FASTQ_ILLUMINA17|FASTQ_ILLUMINA18| \
                           FASTQ_TORRENT30|FASTQ_SANGERQ50)

typedef struct {
  int fmt;         /* Format value */
  const char *dsc; /* Format description */
  int8_t min, max; /* Minimum/Maximum quality values */
  int off;         /* Offset from Phred scores */
  int dis;         /* Overlapping formats to be disabled */
} fqfmt_t;

static fqfmt_t fqfmts[] = {
  { FASTQ_NONE, "unknown", 0, 0, 0,
    FASTQ_NONE },
  { FASTQ_SANGER, "Sanger", 33, 73, 33,
    FASTQ_ILLUMINA18|FASTQ_TORRENT30|FASTQ_SANGERQ50 },
  { FASTQ_SOLEXA, "Solexa", 59, 104, 64,
    FASTQ_NONE },
  { FASTQ_ILLUMINA13, "Illumina1.3+", 64, 104, 64,
    FASTQ_SOLEXA },
  { FASTQ_ILLUMINA15, "Illumina1.5+", 66, 104, 64,
    FASTQ_SOLEXA|FASTQ_ILLUMINA13|FASTQ_ILLUMINA17 },
  { FASTQ_ILLUMINA17, "Illumina1.7+", 66, 105, 64,
    FASTQ_NONE },
  { FASTQ_ILLUMINA18, "Illumina1.8+", 33, 74, 33,
    FASTQ_TORRENT30|FASTQ_SANGERQ50 },
  { FASTQ_TORRENT30, "Torrent3.0+", 33, 78, 33,
    FASTQ_SANGERQ50 },
  { FASTQ_SANGERQ50, "SangerQ50", 33, 83, 33,
    FASTQ_NONE }
};
