"""
a module for parsing the NOE violation files
important for the free R-factor refinement
"""
__author__   = "$Author $"
__revision__ = "$Revision $"
__date__     = "$Date $"

import os, re, string

class ViolationList:
    """
    This class contains all the data from one violation.list file.
    self.noeClasses is a list of noeClass instances which contain the
    restraints in the form of NoeRestraint instances

    public attributes:
        name           name of the violation list
        noeClasses     a list of noeClass instances

    public methods:
        ReadFileName   for reading from a file
        ReadStream     for reading from a stream
        Stdout         writes all the data to stdout
    """

    def __init__(self, name = 'ViolationList'):
        self.name = name
        self.noeClasses = []
        
    def __repr__(self):
        if len(self.noeClasses) > 1:
            return self.name + ' with ' + str(len(self.noeClasses)) + ' NOE classes'
        return self.name + ' with 1 NOE class'

    
    def ReadFileName(self, fileName):
        """
        reads an NOE violation list file
        INPUT: a filename (absolute path)
        """
        #check, if file exists:
        if not os.path.exists(fileName):
            print 'WARNING:', fileName, 'does not exist.'
            print 'ReadFileName() method aborted!'
        print 'reading', fileName
        fileHandle = open(fileName)
        self.ReadStream(fileHandle)

    def ReadStream(self, stream):
        """
        reads an NOE violation list file
        INPUT: a fileIO object
        """
        #compile some patterns:
        classRE = re.compile('CLASS\s+(\S+)\s*\+')
        restraintRE = re.compile('restraint\s*(\d+)\s*=')
        allValuesRE = re.compile('R<average>=\s*([0-9.Ee\-+]+)\s*NOE=\s*([0-9.Ee\-+]+)\s*\(-\s*([0-9.Ee\-+]+)/\+\s*([0-9.Ee\-+]+)\)\s*Delta=\s*([0-9.Ee\-+]+)\s*E\(NOE\)=\s*([0-9.Ee\-+]+)')
        for eachLine in stream.readlines():
            classSE = classRE.search(eachLine)
            if classSE:
                if 'NOECLASS' in dir():
                    self.noeClasses.append(NOECLASS)
                className = classSE.group(1)
                NOECLASS = NoeClass(className)
            restraintSE = restraintRE.search(eachLine)
            if restraintSE:
                restraintNo = restraintSE.group(1)
            allValuesSE = allValuesRE.search(eachLine)            
            if allValuesSE:
                rAverage = allValuesSE.group(1)
                rNoe = allValuesSE.group(2)
                rMinus = allValuesSE.group(3)
                rPlus = allValuesSE.group(4)
                rDelta = allValuesSE.group(5)
                energy = allValuesSE.group(6)
                #add the restraint to the NoeClass
                NOECLASS.AddRestraint(restraintNo, rAverage, rNoe, rMinus,\
                                      rPlus, rDelta, energy)
        #for the last noeclass:
        self.noeClasses.append(NOECLASS)

    def Stdout(self):
        """
        writes all the data to stdout
        """
        for CLASS in self.noeClasses:
            print 'class', CLASS.name, 'contains the restraints:'
            print 'restraintNo rAverage rNoe rMinus rPlus rDelta energy'
            for RES in CLASS.restraints:
                print ' ', RES.restraintNo, RES.rAverage, RES.rNoe, RES.rMinus,\
                      RES.rPlus, RES.rDelta, RES.energy
                
class NoeClass:
    """
    represents the restraint violations of one CNS NOE CLASS

    public attributes:
        name             name of the CNS NOE class
        restraints       a list of NoeRestraint instances
        restraintsDic    key: restraint number, value: NoeRestraint instance
        
    public methods:
        AddRestraint     add a restraint
        
    """
    def __init__(self, name):
        self.name = name
        self.restraints = []
        self.restraintsDic = {}
        
    def __repr__(self):
        return self.name + ' with ' + str(len(self.restraints)) + ' restraints'
    
    def AddRestraint(self, restraintNo, rAverage, rNoe, rMinus,\
                     rPlus, rDelta, energy):
        NOE = NoeRestraint(restraintNo, rAverage, rNoe, rMinus,\
                           rPlus, rDelta, energy)
        self.restraints.append(NOE)
        self.restraintsdic[restraintNo] = NOE

class NoeRestraint:
    """
    represents one restraint with the following attributes:
        restraintNo    restraint number
        rAverage       distance average in the structure
        rNoe           distance calculated from the NOE volume
        rMinus         lower
        rPlus          upper
        rDelta         delta
        energy         NOE energy

    access this class only through its constructor
    """
    def __init__(self, restraintNo, rAverage, rNoe, rMinus,\
                 rPlus, rDelta, energy):
        self.restraintNo = restraintNo
        self.rAverage = rAverage
        self.rNoe = rNoe
        self.rMinus = rMinus
        self.rPlus = rPlus
        self.rDelta = rDelta
        self.energy = energy
        
    def __repr__(self):
        return 'restraint number ' + self.restraintNo



if __name__ == "__main__":
    testString = """
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 +++++++++++++++++++++++++++++++ CLASS DIST +++++++++++++++++++++++++++++++++++
 for this class: SCALe=   1.000 AVERage=sum    POTEntial=soft-square
                 SQCOnstant=   1.000 SQEXponent=    2 SQOFfsets(+/-)=   0.000   0.000
                 SOEXponent=    1 RSWItch=   1.000 ASYMptote=   2.000
 
 ========== restraint   139 ==========
 set-i-atoms
               6    LYS  HG2 
 set-j-atoms
               6    LYS  HN  
 R<average>=   3.888 NOE= 2.80 (- 1.00/+ 1.00) Delta=  -0.088  E(NOE)=   0.008
 
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 +++++++++++++++++++++++++++++++ CLASS AMBI +++++++++++++++++++++++++++++++++++
 for this class: SCALe=   1.000 AVERage=sum    POTEntial=soft-square
                 SQCOnstant=   1.000 SQEXponent=    2 SQOFfsets(+/-)=   0.000   0.000
                 SOEXponent=    1 RSWItch=   1.000 ASYMptote=   2.000
 
 ========== restraint  2819 ==========
 set-i-atoms
               22   LYS  HD2 
 set-j-atoms
               38   PHE  HZ  
 set-i-atoms
               22   LYS  HB2 
 set-j-atoms
               38   PHE  HZ  
 set-i-atoms
               22   LYS  HD3 
 set-j-atoms
               38   PHE  HZ  
 set-i-atoms
               57   ARG  HB3 
 set-j-atoms
               38   PHE  HZ  
 set-i-atoms
               5    ALA  HB2 
               5    ALA  HB1 
               5    ALA  HB3 
 set-j-atoms
               38   PHE  HZ  
 set-i-atoms
               10   LEU  HB3 
 set-j-atoms
               38   PHE  HZ  
 R<average>=   1.811 NOE= 3.20 (- 1.30/+ 1.30) Delta=   0.089  E(NOE)=   0.008
"""
    print 'testing ViolationList.py with the two classes DIST and AMBI.'
    VL = ViolationList('test')
    import StringIO
    stream = StringIO.StringIO(testString)
    print 'reading the restraints from the following string:', testString
    VL.ReadStream(stream)
    VL.Stdout()
    print 'ciao'
