"""
The general ARIA data format for the chemical shifts
together with the converting scripts to XEASY, NMRVIEW and AURELIA.
All the chemical shift data handling is done with this module.
It contains two classes:
-PpmList: the class represents an atom chemical shift list
-Atom:    represents one atom (with ppm, atomname, etc.)

Instantiate PpmList, then read and write the data with its methods:
#example code:
import PpmList
PPM = PpmList.PpmList('test', 3)  #name='test', dimension=3
PPM.ReadXeasyProt('/home/linge/aria/example/n15.prot')
PPM.WriteAureliUserInfo('/home/linge/writeppm.aui')
PPM.WriteChem('/home/linge/writeppm.chem')
PPM.WritePpm('/home/linge/writeppm.ppm')
PPM.WriteXeasyProt('/home/linge/writeppm.prot')
PPM.Stdout()
"""
__author__   = "$Author: linge $"
__revision__ = "$Revision: 1.5 $"
__date__     = "$Date: 2000/03/20 15:41:37 $"

import math, os, re, string
from Aria.Comments import DeleteCnsComments, DeleteComments
from Aria.ThirdParty import  FortranFormat, TextFile
from Aria.Nomenclature import AminoAcid, Nomenclature

###############################################################################
class PpmList:
    """
    Contains the chemical shifts of all atoms of one spectrum
    supplies methods for input and output in various formats
    
    public attributes:
        atomlist       list of atom objects
        atomdicfa      dictionary (key: (residuenumber, atomname),
                       value: atom object)
        comment        comment for the NOE list
        dimension      dimensionality of the spectrum (2D, 3D or 4D)
        name           name of the spectrum
        fileName       name of the file from which the data come from
        
    public methods:
        AddAtom               adds one atomobject
        ReadAnsig             reads an ANSIG crosspeaks export file
        ReadAureliaUserInfo   reads an user info file of AURELIA
        ReadChem              reads an ARIA .chem file
        ReadNmrView           reads a NMRView .out file
        ReadPipp              reads a PIPP .shifts file
        ReadPpm               reads an ARIA .ppm file
        ReadRegine            reads an Regine chemical shift file
        ReadXeasyProt         reads a XEASY .prot file
        RemoveAtom            removes one atomobject
        RemoveDoubleQuotes    removes double quotation marks from all the atomnames
        Stdout                writes the data to stdout
        StdoutAll             writes all the data to stdout
        WriteAureliaUserInfo  writes an user info file of AURELIA
        WriteBioMagResBank    writes a file for BioMagResBank deposition
        WriteChem             writes an .chem file
        WriteId               writes an ARIA .id file
        WritePpm              writes an ARIA .ppm file
        WriteXeasyProt        writes a XEASY .prot file
        WriteXML2String       returns a string containing the sequence in XML format
        WriteXML2File         writes the sequence to an XML file  
    NOTE:
        all fields are treated internally as strings, except:
        atomnames are always tuples of strings
        dimension is always an integer
    """
    def __init__(self, name = 'chemical shifts' , dimension = 0,\
                 comment = 'Aria Data Format'):
        self.atomlist = []
        self.atomdicfa = {}
        self.comment = comment
        self.dimension = dimension
        self.fileName = ''
        self.name = name
        

    def __repr__(self):
        return self.name

    def AddAtom(self, aobj):
        self.atomlist.append(aobj)
        self.atomdicfa[(aobj.residuenumber, aobj.atomname)] = aobj
        

    def ReadAnsig(self, fileName):
        """
        reads an ANSIG crosspeaks export file or ANSIG crosspeaks storage file
        just extracts the chemical shift assignments
        """
        if _DoesFileExist(fileName) == 0:
            return
        print 'reading the ANSIG crosspeaks export file:\n ', fileName
        
        #important - clean atomlist and atomdicfa:
        self.atomlist = []
        self.atomdicfa = {}
        self.fileName = fileName
        
        cpHandle = TextFile.TextFile(fileName)
        
        #there are two possible cases:
        #1. ANSIG v3.3 export crosspeaks file
        #   second line contains 'totalNumber dimensionality'
        #   spectrumName appears in every line
        #2. ANSIG v3.3 storage crosspeaks file
        #   second line contains 'spectrumName totalNumber dimensionality'
        #   spectrumName appears only in the header
        
        #get total number of crosspeaks and dimensionality from the first two lines:
        eachLine = cpHandle.readline()
        eachLine = cpHandle.readline()
        
        totAndDim = string.split(eachLine)
        totalNumber = totAndDim[-2]
        self.dimension = int(totAndDim[-1])
        
        if len(totAndDim) == 2:  #for the ANSIG crosspeaks files           
            format2 = FortranFormat.FortranFormat('3E13.6,A12,7I6,6A4')
            format3 = FortranFormat.FortranFormat('4E13.6,A12,9I6,9A4')
            format4 = FortranFormat.FortranFormat('5E13.6,A12,11I6,12A4')

            #read the assignments:
            ppmdic = {}
            if self.dimension == 2:
                for eachLine in cpHandle:
                    if len(eachLine) < 40:
                        continue
                    line = FortranFormat.FortranLine(eachLine, format2)
                    # {(residuenumber, 3-lettercode, atomname): ppm}
                    ppmdic[(string.strip(str(line[11])), string.strip(string.upper(line[13])),\
                            string.strip(line[15]))] = string.strip(str(line[0]))
                    ppmdic[(string.strip(str(line[12])), string.strip(string.upper(line[14])),\
                            string.strip(line[16]))] = string.strip(str(line[1]))

            elif self.dimension == 3:
                for eachLine in cpHandle:
                    if len(eachLine) < 40:
                        continue
                    line = FortranFormat.FortranLine(eachLine, format3)
##                    for eachBla in line: #test
##                        print eachBla, #test
##                    print '' #test
                    # {(residuenumber, 3-lettercode, atomname): ppm}
                    ppmdic[(string.strip(str(line[14])), string.strip(string.upper(line[17])),\
                            string.strip(line[20]))] = string.strip(str(line[0]))
                    ppmdic[(string.strip(str(line[15])), string.strip(string.upper(line[18])),\
                            string.strip(line[21]))] = string.strip(str(line[1]))
                    ppmdic[(string.strip(str(line[16])), string.strip(string.upper(line[19])),\
                            string.strip(line[22]))] = string.strip(str(line[2]))

            elif self.dimension == 4:
                for eachLine in cpHandle:
                    if len(eachLine) < 40:
                        continue
                    line = FortranFormat.FortranLine(eachLine, format4)
                    # {(residuenumber, 3-lettercode, atomname): ppm}
                    ppmdic[(string.strip(str(line[17])), string.strip(string.upper(line[21])),\
                            string.strip(line[25]))] = string.strip(str(line[0]))
                    ppmdic[(string.strip(str(line[18])), string.strip(string.upper(line[22])),\
                            string.strip(line[26]))] = string.strip(str(line[1]))
                    ppmdic[(string.strip(str(line[19])), string.strip(string.upper(line[23])),\
                            string.strip(line[27]))] = string.strip(str(line[2]))
                    ppmdic[(string.strip(str(line[20])), string.strip(string.upper(line[24])),\
                            string.strip(line[28]))] = string.strip(str(line[3]))

            #set some default values:
            segid = None
            shifterror = None

            #transfer the data from ppmdic to the Atom objects:
            #don't use residuenumber 0 => all the ppms of residue 0 are not used!
##            print ppmdic.keys() #test
            for keyword in ppmdic.keys():
                if (string.strip(keyword[0]) == '') or \
                   (string.strip(keyword[0]) == '0') or \
                   (len(string.strip(keyword[1])) != 3) or \
                   (string.strip(keyword[2]) == ''):
                    continue
                ATOM = Atom(keyword[0], keyword[1], segid, (keyword[2],),\
                            ppmdic[keyword], shifterror)
                self.AddAtom(ATOM)
          
        else:  #for the ANSIG storage files
            format2 = FortranFormat.FortranFormat('3E13.6,5I6,6A4')
            format3 = FortranFormat.FortranFormat('4E13.6,7I6,9A4')
            format4 = FortranFormat.FortranFormat('5E13.6,9I6,12A4')
            spectrumName = totAndDim[0]

            #read the assignments:
            ppmdic = {}
            if self.dimension == 2:
                for eachLine in cpHandle:
                    if len(eachLine) < 40:
                        continue
                    line = FortranFormat.FortranLine(eachLine, format2)
                    # {(residuenumber, 3-lettercode, atomname): ppm}
                    ppmdic[(string.strip(str(line[8])), string.strip(string.upper(line[10])),\
                            string.strip(line[12]))] = string.strip(str(line[0]))
                    ppmdic[(string.strip(str(line[9])), string.strip(string.upper(line[11])),\
                            string.strip(line[13]))] = string.strip(str(line[1]))

            elif self.dimension == 3:
                for eachLine in cpHandle:
                    if len(eachLine) < 40:
                        continue
                    line = FortranFormat.FortranLine(eachLine, format3)
                    # {(residuenumber, 3-lettercode, atomname): ppm}
                    ppmdic[(string.strip(str(line[10])), string.strip(string.upper(line[13])),\
                            string.strip(line[16]))] = string.strip(str(line[0]))
                    ppmdic[(string.strip(str(line[11])), string.strip(string.upper(line[14])),\
                            string.strip(line[17]))] = string.strip(str(line[1]))
                    ppmdic[(string.strip(str(line[12])), string.strip(string.upper(line[15])),\
                            string.strip(line[18]))] = string.strip(str(line[2]))

            elif self.dimension == 4:
                for eachLine in cpHandle:
                    if len(eachLine) < 40:
                        continue
                    line = FortranFormat.FortranLine(eachLine, format4)
                    # {(residuenumber, 3-lettercode, atomname): ppm}
                    ppmdic[(string.strip(str(line[14])), string.strip(string.upper(line[18])),\
                            string.strip(line[22]))] = string.strip(str(line[0]))
                    ppmdic[(string.strip(str(line[15])), string.strip(string.upper(line[19])),\
                            string.strip(line[23]))] = string.strip(str(line[1]))
                    ppmdic[(string.strip(str(line[16])), string.strip(string.upper(line[20])),\
                            string.strip(line[24]))] = string.strip(str(line[2]))
                    ppmdic[(string.strip(str(line[17])), string.strip(string.upper(line[21])),\
                            string.strip(line[25]))] = string.strip(str(line[3]))

            #set some default values:
            segid = None
            shifterror = None

            #transfer the data from ppmdic to the Atom objects:
            #don't use residuenumber 0 => all the ppms of residue 0 are not used!
            for keyword in ppmdic.keys():
                if (string.strip(keyword[0]) == '') or \
                   (string.strip(keyword[0]) == '0') or \
                   (len(string.strip(keyword[1])) != 3) or \
                   (string.strip(keyword[2]) == ''):
                    continue
                ATOM = Atom(keyword[0], keyword[1], segid, keyword[2],\
                            ppmdic[keyword], shifterror)
                self.AddAtom(ATOM)

            
    def ReadAureliaUserInfo(self, fileName):
        """
        reads an Aurelia User Info File
        """
        if _DoesFileExist(fileName) == 0:
            return
        print 'reading an Aurelia User Info File:\n  ', fileName
        print 'We always use the following format for the User Info Files:'
        print '  # 8.17 NH 7 2FMR'
        print '  # ppm atomname residuenumber segid'
        print '  segid should contain 4 letters or should be blank'
        print '  other formats can not be read in by this method!'
        #important - clean atomlist and atomdicfa:
        self.atomlist = []
        self.atomdicfa = {}
        self.fileName = fileName
        auihandle = TextFile.TextFile(fileName)
        for line in auihandle:
            linelist = string.split(line)
            if len(linelist) < 4:
                continue
            ATOM = Atom()
            ATOM.shift = linelist[1]
            ATOM.atomname = linelist[2]
            ATOM.residuenumber = linelist[3]
            try:
                ATOM.segid = linelist[4]
            except:
                ATOM.segid = '    '
            self.AddAtom(ATOM)
        auihandle.close()

    
    def ReadChem(self, fileName):
        """
        reads an ARIA .chem file

        example for one line:
        do ( store1 = 60.709 ) ( resid 1 and name CA )

        comments like '! bla until lineend' or '{bla}' are neglected
        use DeleteCnsComments to parse comments
        """
        if _DoesFileExist(fileName) == 0:
            return
        print 'reading an ARIA chemical shift file', fileName

        #important - clean atomlist and atomdicfa:
        self.atomlist = []
        self.atomdicfa = {}
        self.fileName = fileName

        #get the file without the comments:
        bigstring = DeleteCnsComments.GetString(fileName)
#        print bigstring #test
        #split the string in lines:
        lines = string.split(bigstring, '\n')

        ppmAssign = re.compile('do\s*\(\s*store1\s*=\s*([0-9-+.Ee]+)\s*\)\s*\(\s*resid\s*(\d+)\s*and\s*name\s*(\S+)\s*\)')
        
        for line in lines:
            #for wrong or empty lines:
            if len(line) < 20:
                continue
#            print line #test
            linelist = string.split(line)
            ATOM = Atom()
            ppmSearch = ppmAssign.search(line)

            ATOM.residuenumber = ppmSearch.group(2)
            ATOM.aminoacid = None
            ATOM.segid = None
            ATOM.atomname = (ppmSearch.group(3), )
            ATOM.shift = ppmSearch.group(1)
            ATOM.shifterror = None
            self.AddAtom(ATOM)
    
    def ReadNmrView(self, fileName):
        """
        reads a NMRView chemical shift file

        each line contains:
        residuenumber.atomname shift

        example (one line of an .out file):
        2.HA       4.166 0

        comments like '# bla until lineend' are neglected
        use DeleteComments to parse comments
        """
        if _DoesFileExist(fileName) == 0:
            return
        print 'reading a NMRView .out file', fileName

        #important - clean atomlist and atomdicfa:
        self.atomlist = []
        self.atomdicfa = {}
        self.fileName = fileName

        #get the file without the comments:
        bigstring = DeleteComments.GetString(fileName)

        #split the string in lines:
        lines = string.split(bigstring, '\n')

        for line in lines:
            linelist = string.split(line)
            #for wrong or empty lines:
            if len(linelist) < 3:
                continue
            ATOM = Atom()
            firstFieldList = string.split(linelist[0], '.')
            ATOM.residuenumber = firstFieldList[0]
            ATOM.aminoacid = None
            ATOM.segid = None
            ATOM.atomname = (firstFieldList[1],)
            ATOM.shift = linelist[1]
            ATOM.shifterror = None
            self.AddAtom(ATOM)

    def ReadPipp(self, fileName):
        """
        reads a PIPP .shifts file
        """
        if _DoesFileExist(fileName) == 0:
            return
        print 'reading a PIPP .shifts file', fileName
        
        #important - clean atomlist and atomdicfa:
        self.atomlist = []
        self.atomdicfa = {}
        self.fileName = fileName
        assignNow = 0

        shiftHandle = TextFile.TextFile(fileName)
        
        #split the string in lines:
        for line in shiftHandle:
            if line[0] == '#':
                continue
            linelist = string.split(line)
                
            if linelist == []:
                continue
            elif linelist[0] == 'RES_ID':
                residueID = linelist[1]
            elif linelist[0] == 'RES_TYPE':
                residueType = linelist[1]
            elif linelist[0] == 'SPIN_SYSTEM_ID':
                spinSystemID = linelist[1]
                assignNow = 1
                continue
            elif linelist[0] == 'END_RES_DEF':
                assignNow = 0
            elif linelist[0] == 'SHIFT_FL_FRMT':
                fileFormat = linelist[1]
            elif linelist[0] == 'FIRST_RES_IN_SEQ':
                firstResidue = linelist[1]

            if assignNow:
                ATOM = Atom()
                ATOM.residuenumber = residueID
                ATOM.aminoacid = residueType
                ATOM.atomname = tuple(string.split(linelist[0], '|'))
                ATOM.shift = linelist[1]
                ATOM.shifterror = None    #default
                ATOM.segid = None         #default
                self.AddAtom(ATOM)

        print fileName, 'used the format', fileFormat, 'with first residue', firstResidue
        
    
    
    def ReadPpm(self, fileName):
        """
        reads an ARIA .ppm file
        each line contains:
        residuenumber
        aminoacid
        segid
        atomname
        shift
        shifterror

        unused fields are set to None
        
        uses DeleteComments, the following comments are recognized:
        { bla }
        { bla { bla } bla }
        ! bla until end of line
        # bla until end of line
        
        """
        if _DoesFileExist(fileName) == 0:
            return
        print 'reading an ARIA. ppm file', fileName
        #important - clean atomlist and atomdicfa:
        self.atomlist = []
        self.atomdicfa = {}
        self.fileName = fileName
        #get the file without the comments:
        bigstring = DeleteComments.GetString(fileName)
        #split the string in lines:
        lines = string.split(bigstring, '\n')
        for line in lines:
            linelist = string.split(line)
            #for wrong or empty lines:
            if len(linelist) < 6:
                continue
            ATOM = Atom()
            ATOM.residuenumber = linelist[0]
            ATOM.aminoacid = linelist[1]
            ATOM.segid = linelist[2]
            ATOM.atomname = (linelist[3], )
            ATOM.shift = linelist[4]
            ATOM.shifterror = linelist[5]
            self.AddAtom(ATOM)
        
    
            
    def ReadRegine(self, fileName):
        """
        reads an Regine chemical shift file

        format:

            ALA 12 HA 3.5131

            3-lettercode residueNumber atomName ppm

        Please note that the atomnames are converedt from IUPAC to CNS
        and vice versa!
        """
        if _DoesFileExist(fileName) == 0:
            return
        print 'reading an Aurelia User Info File:\n  ', fileName
        #important - clean atomlist and atomdicfa:
        self.atomlist = []
        self.atomdicfa = {}
        self.fileName = fileName
        fileStream = open(fileName)
        for eachLine in fileStream.readlines():
            lineList = string.split(eachLine)
            if len(lineList) < 4:
                continue
            ATOM = Atom()
            ATOM.shift = lineList[3]
            ATOM.aminoacid = string.upper(lineList[0])
            ATOM.atomname = (Nomenclature.ConvertCnsProtonNames(ATOM.aminoacid, lineList[2]),)
            ATOM.residuenumber = lineList[1]
            ATOM.segid = '    '
            self.AddAtom(ATOM)
        fileStream.close()

    
    def ReadXeasyProt(self, fileName):
        """
        reads a XEASY .prot file
        uses the ReadXeasy module
        """
        #for the XEASY
        import ReadXeasy
        if _DoesFileExist(fileName) == 0:
            return
        #important - clean atomlist and atomdicfa:
        self.atomlist = []
        self.atomdicfa = {}
        print 'reading the .prot file', fileName
        self.fileName = fileName
        XPROT = ReadXeasy.XeasyProt()
        XPROT.ReadProt(fileName)
        for EACH in XPROT.atomlist:
            ATOM = Atom()
            ATOM.residuenumber = EACH.fragmentnumber
            ATOM.atomname = EACH.ariaatomname
            ATOM.shift = EACH.shift
            ATOM.shifterror = EACH.shifterror
            ATOM.xeasyatomname = EACH.xeasyatomname
            ATOM.xeasyatomnumber = EACH.atomnumber
            self.AddAtom(ATOM)
        self.RemoveDoubleQuotes() #conversion of " into ''

    def RemoveAtom(self, aobj):
        """
        removes a given atom object from atomlist and atomdic
        """
        if aobj in self.atomlist:
            self.atomlist.remove(aobj)
        if self.atomdicfa.has_key((aobj.residuenumber, aobj.atomname)):
            del(self.atomdicfa[(aobj.residuenumber, aobj.atomname)])

        
    def RemoveDoubleQuotes(self):
        """
        removes double quotation marks from all the atomnames
        this is useful e.g. for XEASY RNA atomnames
        all the double quotation marks are replaced by two single
        quotation marks
        """
        doubleQuote =  re.compile('"')
        for eachP in self.atomlist:
            tmpList = []
            for eachA in eachP.atomname:
                if eachA:
                    tmpList.append(doubleQuote.sub("''", eachA))
                else:
                    tmpList.append(eachA)
            eachP.atomname = tuple(tmpList)
                
    
    def Stdout(self):
        print 'residuenumber aminoacid segid atomname ppm dppm comment'
        for EACH in self.atomlist:
            print EACH.residuenumber, EACH.aminoacid, EACH.segid,\
                  EACH.atomname, EACH.shift, EACH.shifterror, EACH.comment
        

    def StdoutAll(self):
        """prints out all the attributes + the xeasy stuff"""
        print 'residuenumber aminoacid segid atomname ppm dppm comment',\
              'xatomname xatomnumber'
        for EACH in self.atomlist:
            print EACH.residuenumber, EACH.aminoacid, EACH.segid,\
                  EACH.atomname, EACH.shift, EACH.shifterror, EACH.comment,\
                  EACH.xeasyatomname, EACH.xeasyatomnumber
        
    
    def WriteAureliaUserInfo(self, fileName):
        """
        writes an Aurelia User Info File
        Does not use the chemical shifts with values 999.000
        if there are more than one atomnames in the tuple, only the
        first will be used!
        """
        print 'writing an Aurelia User Info File:\n  ', fileName
        print 'We always use the following format for the User Info Files:'
        print '  # 8.17 NH 7 2FMR'
        print '  # ppm atomname residuenumber segid'
        print '  segid should contain 4 letters or 4 spaces'
        auihandle = TextFile.TextFile(fileName, 'w')
        for EACH in self.atomlist:
            #those with 999.000 don't have an assignment:
            if EACH.shift != '999.000':
                if EACH.segid == None:
                    outsegid = '    '  #4 spaces
                else:
                    outsegid = EACH.segid
                auihandle.write('# ' + EACH.shift + ' ' +\
                                EACH.atomname[0] + ' ' +\
                                EACH.residuenumber +\
                                outsegid + '\n')
        
    
    def WriteAureliaUserInfo(self, fileName):
        """
        writes an Aurelia User Info File
        Does not use the chemical shifts with values 999.000
        if there are more than one atomnames in the tuple, only the
        first will be used!
        """
        print 'writing an Aurelia User Info File:\n  ', fileName
        print 'We always use the following format for the User Info Files:'
        print '  # 8.17 NH 7 2FMR'
        print '  # ppm atomname residuenumber segid'
        print '  segid should contain 4 letters or 4 spaces'
        auihandle = TextFile.TextFile(fileName, 'w')
        for EACH in self.atomlist:
            #those with 999.000 don't have an assignment:
            if EACH.shift != '999.000':
                if EACH.segid == None:
                    outsegid = '    '  #4 spaces
                else:
                    outsegid = EACH.segid
                auihandle.write('# ' + EACH.shift + ' ' +\
                                EACH.atomname[0] + ' ' +\
                                EACH.residuenumber +\
                                outsegid + '\n')
        
    
    def WriteBioMagResBank(self, fileName):
        """
        writes a file for BioMagResBank deposition
        
        Does not use the chemical shifts with values 999.000
        if there are more than one atomnames in the tuple, only the
        first will be used!
        """
        print 'writing the BioMagResBank StarFormat file', fileName
        bioHandle = open(fileName, 'w')

        #write out the amino acid sequence:
        molResidueSequence = '_Mol_residue_sequence\n;\n'
        bioHandle.write(molResidueSequence)
        
        aaSequence = ''
        oldResidueNumber = -999
        for EACH in self.atomlist:
            if EACH.aminoacid:
                if string.atoi(EACH.residuenumber) > oldResidueNumber:
                    aaSequence = aaSequence + AminoAcid.AminoAcid(EACH.aminoacid)[0]
                    oldResidueNumber = EACH.residuenumber
        bioHandle.write(aaSequence + '\n;\n\n')
        
        if aaSequence == '':
            print 'WARNING: aminoacid sequence not known from input data!'

        #write out the amino acid sequence with numbering:
        resLoop = """loop_
	_Residue_seq_code
	_Residue_author_seq_code
	_Residue_label

"""
        bioHandle.write(resLoop)
        iii = 1
        while iii < (len(aaSequence)+1):
            bioHandle.write(str(iii) + '   @   ' + self.aalist[iii] + '     ')
            if math.modf((iii)/5.0)[0] == 0.0:
                bioHandle.write('\n')
            iii = iii + 1
        bioHandle.write('stop_\n')

        #chemical shift table:
        commentsForPpm = """
###################################################################
#      Chemical Shift Ambiguity Code Definitions                  #
#                                                                 #
#    Codes            Definition                                  #
#                                                                 #
#      1             Unique                                       #
#      2             Ambiguity of geminal atoms or geminal methyl #
#                         proton groups                           #
#      3             Aromatic atoms on opposite sides of the ring #
#                        (e.g. Tyr HE1 and HE2 protons)           #
#      4             Intraresidue ambiguities (e.g. Lys HG and    #
#                         HD protons)                             #
#      5             Interresidue ambiguities (Lys 12 vs. Lys 27) #
#      9             Ambiguous, specific ambiguity not defined    #
#                                                                 #
###################################################################

loop_
     _Atom_shift_assign_ID
     _Residue_seq_code
     _Residue_label
     _Atom_name
     _Atom_type
     _Chem_shift_value
     _Chem_shift_value_error
     _Chem_shift_ambiguity_code

#
#Atom	Residue
#shift	Seq	Residue	Atom	Atom	Shift/	Error/	Ambiguity
#assign	code	Label	Name	Type	ppm	ppm	Code
#---------------------------------------------------------------
#
"""
        bioHandle.write(commentsForPpm)

        atomShiftAssignCounter = 1
        
        for EACH in self.atomlist:
            if EACH.shift != '999.000' and EACH.shift != None:
                if EACH.aminoacid:
                    outAA = EACH.aminoacid
                else:
                    outAA = '@'  # @  instead of whitespace or None
                #for the ambiguity:
                if len(EACH.atomname) > 1:
                    ambiguity = 9
                else:
                    ambiguity = 1
                if EACH.shifterror:
                    outError = EACH.shifterror
                else:
                    outError = '@' # @  instead of whitespace or None
                outString = '''%i %s %s %s %s %s %i\n''' %\
                            (atomShiftAssignCounter,\
                             EACH.residuenumber,\
                             outAA,\
                             EACH.atomname[0],  #always take first alternative!
                             EACH.shift,\
                             outError,\
                             ambiguity)
                atomShiftAssignCounter = atomShiftAssignCounter + 1
                bioHandle.write(outString)

        moreComments = """
stop_

# The following loop is used to define sets of Atom-shift assignment IDs that
# represent related ambiguous assignments taken from the above list of
# assigned chemical shifts.  Each element in the set should be separated by a
# comma, as shown in the example below, and is the assignment ID for a chemical
# shift assignment that has been given as ambiguity code of 4 or 5.  Each set
# indicates that the observed chemical shifts are related to the defined 
# atoms, but have not been assigned uniquely to a specific atom in the set.

loop_
  _Atom_shift_assign_ID_ambiguity   

#
#    Sets of Atom-shift Assignment Ambiguities
               #              
#    ------------------------------------------
# Example:    5,4,7
#
                @
stop_
"""
        bioHandle.write(moreComments)
        bioHandle.close()

    
    def WriteChem(self, fileName):
        """
        writes a .chem list which can be read by cns
        Does not use the chemical shifts with values 999.000
        if there are more than one atomnames in the tuple, only the
        first will be used!
        """
        print 'writing a .chem file', fileName
        chemhandle = TextFile.TextFile(fileName, 'w')
        
        chemhandle.write('! derived from the file:\n')
        chemhandle.write('! ' + self.fileName + '\n')
        for EACH in self.atomlist:
            #those with 999.000 don't have an assignment:
            if EACH.shift != '999.000':
                chemhandle.write('do ( store1 = ' + EACH.shift +\
                                 ' ) ( resid ' + EACH.residuenumber +\
                                 ' and name ' + EACH.atomname[0] + ' )\n')
    
    def WriteId(self, fileName):
        """
        writes an ARIA .id file which can be read by cns
        if there are more than one atomname in the tuple, only the
        first will be used!
        if there aren't any atomnumbers, they will be added (counting from 1)
        """
        print 'writing an .id file', fileName
        idhandle = TextFile.TextFile(fileName, 'w')
        idhandle.write('! derived from the file:\n')
        idhandle.write('! ' + self.fileName + '\n')
        atomCounter = 1
        for EACH in self.atomlist:
            if EACH.xeasyatomnumber == None:
                outAtomNumber = str(atomCounter)
            else:
                outAtomNumber = EACH.xeasyatomnumber
            idhandle.write('do ( store2 = ' + outAtomNumber +\
                           ' ) ( resid ' + EACH.residuenumber +\
                           ' and name ' + EACH.atomname[0] + ' )\n')
            atomCounter = atomCounter + 1

    
    def WritePpm(self, fileName):
        """
        writes a list in ARIA's .ppm format
        CNS will read the restraints in store1
        does not use the chemical shifts with values 999.000
        if there are more than one atomnames in the tuple, only the
        first will be used!
        """
        print 'writing a .ppm file', fileName
        ppmhandle = TextFile.TextFile(fileName, 'w')
        ppmhandle.write('! derived from the file:\n')
        ppmhandle.write('! ' + self.fileName + '\n')
        for EACH in self.atomlist:
            #those with 999.000 don't have an assignment:
            if EACH.shift != '999.000':
                if EACH.residuenumber:
                    outResidueNumber = EACH.residuenumber
                else:
                    outResidueNumber = '-'
                if EACH.aminoacid:
                    outAminoAcid = EACH.aminoacid
                else:
                    outAminoAcid = '-'
                if EACH.segid:
                    outSegid = EACH.segid
                else:
                    outSegid = '-'
                if EACH.atomname:
                    outAtomname = EACH.atomname[0]
                else:
                    outAtomname = '-'
                if EACH.shift:
                    outShift = EACH.shift
                else:
                    outShift = '-'
                if EACH.shifterror:
                    outShiftError = EACH.shifterror
                else:
                    outShiftError = '-'
##                print outResidueNumber + ' ' +\
##                      outAminoAcid + ' ' +\
##                      outSegid + ' ' +\
##                      outAtomname + ' ' +\
##                      outShift + ' ' +\
##                      outShiftError
                ppmhandle.write(outResidueNumber + ' ' +\
                                outAminoAcid + ' ' +\
                                outSegid + ' ' +\
                                outAtomname + ' ' +\
                                outShift + ' ' +\
                                outShiftError + '\n')
        


    def WriteXeasyProt(self, fileName, useall=1):
        """
        writes a XEASY .prot file

        useall=1:    writes out all the atoms (default)
        otherwise:   does not write atoms with shifts 999.000

        if there are more than one atomnames in the tuple, only the
        first will be used!
        """
        print 'writing a XEASY .prot file', fileName
        ppmhandle = TextFile.TextFile(fileName, 'w')
        for EACH in self.atomlist:
            if EACH.xeasyatomnumber == None:
                EACH.xeasyatomnumber = 0
            if EACH.shift == None:
                outshift = 0
            else:
                outshift = EACH.shift
            if EACH.shifterror == None:
                outshifterror = 0
            else:
                outshifterror = EACH.shifterror
            if EACH.residuenumber == None:
                EACH.residuenumber = 0
            if (EACH.shift != '999.000') or (useall == 1):
                ppmhandle.write('%4i %7.3f %5.3f %-5s %3i \n' %\
                                (string.atoi(EACH.xeasyatomnumber),\
                                 string.atof(outshift),\
                                 string.atof(outshifterror),\
                                 EACH.atomname[0],\
                                 string.atoi(EACH.residuenumber)))

    

    def WriteXML2String(self):
        """
        returns a string containing the ppm data in XML format
        """
        beginTag = '<PPM_LIST>\n'
        endTag = '</PPM_LIST>\n'
        outString = ''
        iii = 1 #test
        for eachObj in self.atomlist:
            print str(iii) #test
            iii = iii + 1 #test
            eachPpmStart = '  <PPM>\n'
            if eachObj.shift:
                eachPpmValue = '    <PPM_VALUE>%s</PPM_VALUE>\n' % (str(eachObj.shift))
            else:
                eachPpmValue = ''
            if eachObj.shifterror:
                eachPpmError = '    <PPM_ERROR>%s</PPM_ERROR>\n' % (str(eachObj.shifterror))
            else:
                eachPpmError = ''
            outString = outString + eachPpmStart + eachPpmValue + eachPpmError
            for eachAtom in eachObj.atomname:
                atomString = """    <ATOM>
      <ATOM_NAME>%s</ATOM_NAME>
      <THREE_LETTER_CODE>%s</THREE_LETTER_CODE>
      <RESIDUE_NUMBER>%s</RESIDUE_NUMBER>
      <SEGMENT>%s</SEGMENT>
    </ATOM>
""" % (eachAtom, eachObj.aminoacid, eachObj.residuenumber, eachObj.segid)
                outString = outString + atomString
            eachPpmEnd = '  </PPM>\n'
            outString =  outString + eachPpmEnd
        outString = beginTag + outString + endTag
        return outString

    
    def WriteXML2File(self, fileName):
        """writes the ppm data to an XML file"""
        outString = self.WriteXML2String()
        try:
            outhandle = TextFile.TextFile(fileName, 'w')
        except IOError:
            print 'could not open the file', fileName
            print 'Abort WriteXML2File method.'
            return
        print 'writing to the file:', fileName
        outhandle.write(outString)
        outhandle.close()
        
###############################################################################
class Atom:
    """
    represents the chemical shift assignment of one atom

    ambiguities are handle which contains tuples of atomnames
    
    This object contains all the data as attributes, there are no
    methods available.

    all the attributes contain strings, except atomname which contains a tupel
    of strings!
    
    attributes:
        general attributes:
        residuenumber        residuenumber
        aminoacid            3-letter code
        segid                segid (must have 4 characters)
        atomname             tuple of atomnames (ARIA/CNS nomenclature)
        shift                chemical shift
        shifterror           chemical shift error
        comment
        
        other attributes:
        xeasyatomname        XEASY atomname
        xeasyatomnumber      XEASY atomnumber
    """
    def __init__(self, residuenumber = None, aminoacid = None, segid = None,\
                 atomname = None, shift = None, shifterror = None,\
                 comment = None, xeasyatomname = None, xeasyatomnumber = None):
        self.residuenumber = residuenumber
        self.aminoacid = aminoacid
        self.segid = segid
        self.atomname = atomname
        self.shift = shift
        self.shifterror = shifterror
        self.comment = comment

        self.xeasyatomname = xeasyatomname
        self.xeasyatomnumber = xeasyatomnumber
        
    
###############################################################################
def _DoesFileExist(fileName):
    if os.path.exists(fileName) == 0:
        print 'WARNING:', fileName, 'does not exist.'
        return 0
    return 1

###############################################################################
